﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    /*
     *
     *  SvcSet
     *      Types           map from TypeName to Type
     *      Operations      list of Operation
     *
     *      Categories      list of Category
     *                          Inludes         list of string
     *                          Types           list of Type
     *                          Operations      list of Operation
     *
     */
    internal class SvcSet
    {
        private class TypeSet
        {
            public bool IsSizeValid { get; private set; }
            public bool IsPredefined { get; private set; }
            public bool IsCommon { get; private set; }
            public bool IsPointerSize { get; private set; }
            public bool IsRegisterSize { get; private set; }
            public string Name { get; private set; }
            public string RealName { get; private set; }
            public string Namespace { get; private set; }
            public string CategoryName { get; private set; }
            public SizeAlignment Size { get; private set; }
            public Parser.TypeDeclaration SourceItem { get; private set; }

            public TypeSet(string n, string ns, string cn, Parser.TypeDeclaration td)
            {
                IsSizeValid = false;
                IsPredefined = false;
                IsCommon = td.IsCommon;
                IsPointerSize = false;
                IsRegisterSize = false;
                Name = n;
                RealName = td.RealName;
                Namespace = ns;
                CategoryName = cn;
                SourceItem = td;
            }
            private TypeSet(string n, string ns, int size, bool isPointerSize, bool isRegisterSize)
            {
                IsSizeValid = true;
                IsPredefined = true;
                IsCommon = true;
                IsPointerSize = isPointerSize;
                IsRegisterSize = isRegisterSize;
                Name = n;
                RealName = n;
                Namespace = ns;
                CategoryName = string.Empty;
                Size = new SizeAlignment(size, size);
                SourceItem = null;
            }
            public void SetSize(SizeAlignment sa)
            {
                Size = sa;
                IsSizeValid = true;
            }
            public static TypeSet MakePredefined(string n, string ns, int size, bool isPointerSize, bool isRegisterSize)
            {
                return new TypeSet(n, ns, size, isPointerSize, isRegisterSize);
            }
        }
        private class SizeAlignment
        {
            public static readonly SizeAlignment VariableSize =
                new SizeAlignment(Type.UnknownSizeValue, Type.UnknownSizeValue);

            public int Size { get; private set; }
            public int Alignment { get; private set; }

            public SizeAlignment(int s, int a)
            {
                Size = s;
                Alignment = a;
            }
        }

        public Dictionary<string, Type> Types { get; private set; }
        public Operation[] Operations { get; private set; }
        public CategorySet[] Categories { get; private set; }

        public static SvcSet Construct(Parser.SvcDefinition sd, int pointerSize, int registerSize, string typeNamespace, string commonTypeNamespace)
        {
            // includes を処理
            var incMap = MakeIncludeMap(sd);

            // types を処理
            var typeMap = MakeTypeMap(sd, pointerSize, registerSize, typeNamespace, commonTypeNamespace);
            var types = typeMap.SelectMany(x => x.Value).ToDictionary(x => x.Name);

            // operations を処理
            var operationMap = MakeOperationMap(sd, types);

            // Category を作成
            var categories = MakeCategories(incMap, typeMap, operationMap);

            var ss = new SvcSet();
            ss.Types = types;
            ss.Operations = operationMap.SelectMany(x => x.Value).OrderBy(x => x.FunctionNumber).ToArray();
            ss.Categories = categories;
            return ss;
        }

        private static Dictionary<string, string[]> MakeIncludeMap(Parser.SvcDefinition sd)
        {
            return sd.OfType<Parser.Includes>()
                     .ToLookup(x => x.CategoryName)
                     .ToDictionary(x => x.Key,
                                   x => x.SelectMany(inc => inc.IncludeFiles).ToArray());
        }

        private static void AddPredefinedType(Dictionary<string, TypeSet> map, string name, string ns, int size, bool isPointerSize, bool isRegisterSize)
        {
            map.Add(name, TypeSet.MakePredefined(name, ns, size, isPointerSize, isRegisterSize));
        }
        private static Dictionary<string, Type[]> MakeTypeMap(
            Parser.SvcDefinition sd, int pointerSize, int registerSize, string typeNamespace, string commonTypeNamespace)
        {
            var map = new Dictionary<string, TypeSet>();

            // 型のマップを作る
            foreach (var types in sd.OfType<Parser.Types>())
            {
                foreach (var t in types.TypeDeclarations)
                {
                    var ns = t.IsCommon ? commonTypeNamespace : typeNamespace;
                    var set = new TypeSet(t.Name, ns, types.CategoryName, t);
                    map.Add(set.Name, set);
                }
            }

            // 組み込み型を追加する
            foreach (var t in PredefinedTypes.List)
            {
                var size = t.IsPointerSize ? pointerSize : (t.IsRegisterSize? registerSize : t.Size);
                map.Add(t.Name, TypeSet.MakePredefined(t.Name, t.Namespace, size, t.IsPointerSize, t.IsRegisterSize));
            }

            // サイズを解決する
            map.Values.ForEach(x => CalculateSize(x, map));

            {
                var typesMap = map.Values
                          .ToLookup(x => x.CategoryName)
                          .ToDictionary(x => x.Key,
                                        x => x.Select(set => MakeType(set, pointerSize, registerSize)).ToArray());

                // typedef を解決する
                var types = typesMap.SelectMany(x => x.Value).ToDictionary(x => x.Name);
                foreach (var type in types.Values)
                {
                    var set = map[type.Name];
                    if (set.RealName != set.Name)
                    {
                        type.UpdateRealType(types[set.RealName]);
                    }
                }

                return typesMap;
            }
        }
        private static SizeAlignment CalculateSize(TypeSet set, Dictionary<string, TypeSet> map)
        {
            if (!set.IsSizeValid)
            {
                var td = set.SourceItem;
                SizeAlignment size;

                if (td.IsSizeUnknown)
                {
                    if (td is Parser.TypeDeclarationStruct)
                    {
                        var typeStruct = (Parser.TypeDeclarationStruct)td;

                        int offset = 0;
                        int maxAlignment = 0;
                        foreach (var m in typeStruct.Members)
                        {
                            var memSize = CalculateSize(m, map);
                            if (memSize == SizeAlignment.VariableSize)
                            {
                                return memSize;
                            }
                            offset = Util.RoundUp(offset, memSize.Alignment);
                            offset += memSize.Size;
                            maxAlignment = Math.Max(maxAlignment, memSize.Alignment);
                        }
                        offset = Util.RoundUp(offset, maxAlignment);

                        size = new SizeAlignment(offset, maxAlignment);
                    }
                    else if (td is Parser.TypeDeclarationUnion)
                    {
                        var typeUnion = (Parser.TypeDeclarationUnion)td;

                        int maxSize = 0;
                        int maxAlignment = 0;
                        foreach (var m in typeUnion.Members)
                        {
                            var memSize = CalculateSize(m, map);
                            if (memSize == SizeAlignment.VariableSize)
                            {
                                return memSize;
                            }
                            maxSize = Math.Max(maxSize, memSize.Size);
                            maxAlignment = Math.Max(maxAlignment, memSize.Alignment);
                        }

                        size = new SizeAlignment(maxSize, maxAlignment);
                    }
                    else if (td is Parser.TypeDeclarationAlias)
                    {
                        var typeAlias = (Parser.TypeDeclarationAlias)td;
                        var type = Util.TrappedGetValue(map, typeAlias.TypeName);
                        size = CalculateSize(type, map);
                    }
                    else
                    {
                        throw new ErrorException(
                            "内部エラー",
                            string.Format("unknown TypeDeclaration type ({0})", td.GetType().Name));
                    }
                }
                else
                {
                    size = new SizeAlignment(td.Size, td.Alignment);
                }

                set.SetSize(size);
            }

            return set.Size;
        }
        private static Type MakeType(TypeSet set, int pointerSize, int registerSize)
        {
            Func<string> writer = null;
            if ((!set.IsPredefined) && (!set.SourceItem.IsAbstract))
            {
                writer = set.SourceItem.GetTypeWriter(pointerSize, registerSize);
            }

            return new Type(
                set.Name,
                set.Namespace,
                set.Size.Size,
                set.Size.Alignment,
                set.IsPredefined,
                set.IsCommon,
                set.IsPointerSize,
                set.IsRegisterSize,
                writer);
        }
        private static SizeAlignment CalculateSize(
            Parser.VariableDeclaration vd, Dictionary<string, TypeSet> map)
        {
            var type = Util.TrappedGetValue(map, vd.TypeName);
            return CalculateSize(vd, CalculateSize(type, map));
        }
        private static SizeAlignment CalculateSize(
            Parser.VariableDeclaration vd, SizeAlignment typeSize)
        {
            if (!vd.IsArray)
            {
                return typeSize;
            }

            if (vd.ArraySize != Parser.VariableDeclaration.ArraySizeUnknown)
            {
                return new SizeAlignment(
                    typeSize.Size * vd.ArraySize,
                    typeSize.Alignment);
            }
            else
            {
                return SizeAlignment.VariableSize;
            }
        }

        private static Dictionary<string, Operation[]> MakeOperationMap(
            Parser.SvcDefinition sd, Dictionary<string, Type> types)
        {
            var map = new Dictionary<string, List<Operation>>();
            int functionNumber = 1;

            foreach (var operations in sd.OfType<Parser.Operations>())
            {
                var list = Util.GetValueOrAddDefault(map, operations.CategoryName);

                foreach (var op in operations.OperationContents)
                {
                    if (op is Parser.OperationDeclaration)
                    {
                        var od = (Parser.OperationDeclaration)op;
                        if (od.FunctionNumber != Parser.OperationDeclaration.FunctionNumberAuto)
                        {
                            functionNumber = od.FunctionNumber;
                        }

                        list.Add(MakeOpeation(functionNumber, od, types));

                        functionNumber++;
                    }
                    else if (op is Parser.SetNextFunctionNumber)
                    {
                        functionNumber = ((Parser.SetNextFunctionNumber)op).Number;
                    }
                    else
                    {
                        throw new ErrorException("内部エラー",
                            string.Format("不明な OperationContent です。({0})", op.GetType().Name));
                    }
                }
            }

            return map.ToDictionary(x => x.Key, x => x.Value.ToArray());
        }
        private static OperationParameter.DirectionType Convert(Parser.ParameterDirectionType x)
        {
            switch (x)
            {
            case Parser.ParameterDirectionType.In: return OperationParameter.DirectionType.In;
            case Parser.ParameterDirectionType.Out: return OperationParameter.DirectionType.Out;
            case Parser.ParameterDirectionType.InPtr: return OperationParameter.DirectionType.InPtr;
            case Parser.ParameterDirectionType.OutPtr: return OperationParameter.DirectionType.OutPtr;
            }
            throw new ErrorException("内部エラー", string.Format("不明な引数入出力指定です。({0})", x));
        }
        private static Operation MakeOpeation(
            int functionNumber,
            Parser.OperationDeclaration od,
            Dictionary<string, Type> types)
        {
            var list = new List<OperationParameter>();

            foreach (var pop in od.OperationParameters)
            {
                var type = Util.TrappedGetValue(types, pop.Variable.TypeName);
                var size = CalculateSize(pop.Variable, new SizeAlignment(type.Size, type.Alignment));

                var op = new OperationParameter(
                    pop.Variable.Name,
                    type,
                    Convert(pop.ParameterDirection),
                    type.IsPointerSize ? Type.PointerSizeValue : (type.IsRegisterSize ? Type.RegisterSizeValue : size.Size),
                    type.IsPointerSize ? Type.PointerSizeValue : (type.IsRegisterSize ? Type.RegisterSizeValue : size.Alignment),
                    pop.Variable.IsArray);
                list.Add(op);
            }

            return new Operation(functionNumber, od.Name, types[od.ReturnTypeName], list.ToArray());
        }

        private static CategorySet[] MakeCategories(
            Dictionary<string, string[]> incMap,
            Dictionary<string, Type[]> typeMap,
            Dictionary<string, Operation[]> operationMap)
        {
            var names = incMap.Keys.Concat(typeMap.Keys)
                                   .Concat(operationMap.Keys)
                                   .ToLookup(x => x)
                                   .Select(x => x.Key);

            return names.Select(name =>
                {
                    var incs = Util.GetValueOrDefault(incMap, name);
                    var types = Util.GetValueOrDefault(typeMap, name);
                    var operations = Util.GetValueOrDefault(operationMap, name);
                    if (types != null)
                    {
                        // コード生成すべきものだけを抽出する
                        types = types.Where(x => !x.IsAbstract).ToArray();
                        if (types.Length == 0)
                        {
                            types = null;
                        }
                    }
                    if (operations != null)
                    {
                        // コード生成時の出力順を機能番号順にするためにここでソートしておく
                        operations = operations.OrderBy(x => x.FunctionNumber).ToArray();
                    }
                    return new CategorySet(name, incs, types, operations);
                })
                .ToArray();
        }
    }
}
