﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer.Parser
{
    internal enum ParameterDirectionType
    {
        In,
        Out,
        InPtr,
        OutPtr,
    }

    internal class Includes : SvcDefinitionContent
    {
        public string CategoryName { get; private set; }
        public IncludeFileList IncludeFiles { get; private set; }

        public Includes(string name, IncludeFileList files)
        {
            this.CategoryName = name;
            this.IncludeFiles = files;
        }
    }

    internal class Types : SvcDefinitionContent
    {
        public string CategoryName { get; private set; }
        public TypeDeclarationList TypeDeclarations { get; private set; }

        public Types(string name, TypeDeclarationList dec)
        {
            this.CategoryName = name;
            this.TypeDeclarations = dec;
        }
    }

    internal class Operations : SvcDefinitionContent
    {
        public string CategoryName { get; private set; }
        public OperationContentList OperationContents { get; private set; }

        public Operations(string name, OperationContentList ocl)
        {
            this.CategoryName = name;
            this.OperationContents = ocl;
        }
    }

    internal class OperationDeclaration : OperationContent
    {
        public const int FunctionNumberAuto = -1;

        public string ReturnTypeName { get; private set; }
        public string Name { get; private set; }
        public int FunctionNumber { get; private set; }
        public OperationParameterList OperationParameters { get; private set; }

        public OperationDeclaration(string ret, string name) : this(ret, name, FunctionNumberAuto) { }

        public OperationDeclaration(string ret, string name, int no)
        {
            this.ReturnTypeName = ret;
            this.Name = name;
            this.FunctionNumber = no;
        }

        public static OperationDeclaration AddParameters(OperationDeclaration arg1, OperationParameterList arg2)
        {
            arg1.OperationParameters = arg2;
            return arg1;
        }
    }

    internal class OperationParameter
    {
        public ParameterDirectionType ParameterDirection { get; private set; }
        public VariableDeclaration Variable { get; private set; }

        public OperationParameter(ParameterDirectionType dir, VariableDeclaration vd)
        {
            ParameterDirection = dir;
            Variable = vd;
        }
    }

    internal class VariableDeclaration
    {
        public const int ArraySizeUnknown = Type.UnknownSizeValue;

        public string TypeName { get; private set; }
        public string Name { get; private set; }
        public bool IsArray { get; private set; }
        public int ArraySize { get; private set; }
        public string ArraySizeVariableName { get; private set; }

        public VariableDeclaration(string tn, string name)
        {
            TypeName = tn;
            Name = name;
            IsArray = false;
            ArraySize = ArraySizeUnknown;
            ArraySizeVariableName = null;
        }

        public static VariableDeclaration MakeArray(string arg1, string arg2)
        {
            var ret = new VariableDeclaration(arg1, arg2);
            ret.IsArray = true;
            return ret;
        }
        public static VariableDeclaration MakeArray(string arg1, string arg2, string arg3)
        {
            var ret = MakeArray(arg1, arg2);
            ret.ArraySizeVariableName = arg3;
            return ret;
        }
        public static VariableDeclaration MakeArray(string arg1, string arg2, int arg3)
        {
            var ret = MakeArray(arg1, arg2);
            ret.ArraySize = arg3;
            return ret;
        }

        public Tuple<string, string> FormatCText(int pointerSize, int registerSize)
        {
            var map = PredefinedTypes.GetReplaceMap(pointerSize, registerSize);
            var typename = Util.GetValueOrDefault(map, TypeName, TypeName);

            var nameText = Name;
            if (IsArray)
            {
                nameText = string.Format("{0}[{1}]",
                    Name,
                    (ArraySize != ArraySizeUnknown) ? ArraySize.ToString() : string.Empty);
            }

            return new Tuple<string, string>(typename, nameText);
        }
    }

    internal abstract class TypeDeclaration
    {
        public bool IsAbstract { get; private set; }
        public bool IsCommon { get; private set; }
        public string Name { get; private set; }
        public virtual string RealName { get { return Name; } }
        public virtual int Size { get { return Type.UnknownSizeValue; } }
        public virtual int Alignment { get { return Type.UnknownSizeValue; } }

        public bool IsSizeUnknown { get { return Size == Type.UnknownSizeValue; } }

        public TypeDeclaration(string name)
        {
            IsAbstract = false;
            Name = name;
        }

        internal static TypeDeclaration SetAbstract(TypeDeclaration arg1)
        {
            arg1.IsAbstract = true;
            return arg1;
        }
        internal static TypeDeclaration SetCommon(TypeDeclaration arg1)
        {
            arg1.IsCommon = true;
            return arg1;
        }

        internal abstract Func<string> GetTypeWriter(int pointerSize, int registerSize);

        protected string MakeTypeText(string structName, VariableDeclarationList members, int pointerSize, int registerSize)
        {
            var pairs = members.Select(x => x.FormatCText(pointerSize, registerSize));
            var maxTypeLength = pairs.Max(x => x.Item1.Length);

            var format = string.Format("{0}{1}{2}", "    {0,-", maxTypeLength, "} {1};\r\n");

            var sb = new StringBuilder();
            sb.AppendFormat("{0} {1}\r\n", structName, Name);
            sb.AppendLine("{");
            foreach (var pair in pairs)
            {
                sb.AppendFormat(format, pair.Item1, pair.Item2);
            }
            sb.AppendLine("};");
            return sb.ToString();
        }
    }
    internal abstract class OperationContent
    {
    }
    internal abstract class SvcDefinitionContent
    {
    }

    internal class TypeDeclarationStruct : TypeDeclaration
    {
        public VariableDeclarationList Members { get; private set; }

        public TypeDeclarationStruct(string arg1, VariableDeclarationList arg2)
            : base(arg1)
        {
            this.Members = arg2;
        }

        internal override Func<string> GetTypeWriter(int pointerSize, int registerSize)
        {
            return () => this.MakeTypeText("struct", this.Members, pointerSize, registerSize);
        }
    }
    internal class TypeDeclarationUnion : TypeDeclaration
    {
        public VariableDeclarationList Members { get; private set; }

        public TypeDeclarationUnion(string arg1, VariableDeclarationList arg2)
            : base(arg1)
        {
            this.Members = arg2;
        }

        internal override Func<string> GetTypeWriter(int pointerSize, int registerSize)
        {
            return () => this.MakeTypeText("union", this.Members, pointerSize, registerSize);
        }
    }
    internal class TypeDeclarationEnum : TypeDeclaration
    {
        public override int Size { get { return 4; } }
        public override int Alignment { get { return 4; } }

        public TypeDeclarationEnum(string arg1)
            : base(arg1)
        {
        }
        internal override Func<string> GetTypeWriter(int pointerSize, int registerSize) { return null; }
    }
    internal class TypeDeclarationAlias : TypeDeclaration
    {
        public string TypeName { get; private set; }
        public override string RealName { get { return this.TypeName; } }

        public TypeDeclarationAlias(string arg1, string arg2)
            : base(arg2)
        {
            this.TypeName = arg1;
        }

        internal override Func<string> GetTypeWriter(int pointerSize, int registerSize)
        {
            return () =>
                {
                    var map = PredefinedTypes.GetReplaceMap(pointerSize, registerSize);
                    var typename = Util.GetValueOrDefault(map, TypeName, TypeName);
                    return string.Format("typedef {0} {1};\r\n", typename, Name);
                };
        }
    }
    internal class TypeDeclarationPredefined : TypeDeclaration
    {
        public TypeDeclarationPredefined(string name) : base(name) { }

        internal override Func<string> GetTypeWriter(int pointerSize, int registerSize) { return null; }
    }

    internal class SetNextFunctionNumber : OperationContent
    {
        public int Number { get; private set; }

        public SetNextFunctionNumber(int arg1)
        {
            this.Number = arg1;
        }
    }
}
