﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer.Parser
{
    internal static class SvcParserHelper
    {
        private static string MakeExceptionMessage(Parser p, TokenStream ts, Exception e, string path)
        {
            int errorPos = Math.Min(ts.Position, Math.Max(ts.Text.Length - 1, 0));

            // エラー行
            int lineBegin = ts.Text.LastIndexOfLineHead(errorPos);
            int lineEnd = ts.Text.IndexOfLineEnd(errorPos);
            string lineSample = ts.Text.Substring(lineBegin, lineEnd - lineBegin);

            string sample = " " + lineSample;

            // エラー行の前の行
            int preLineEnd = ts.Text.LastIndexOfLineEnd(errorPos);
            if (preLineEnd >= 0)
            {
                int preLineBegin = ts.Text.LastIndexOfLineHead(preLineEnd);
                string preLineSample = ts.Text.Substring(preLineBegin, preLineEnd - preLineBegin);

                sample = " " + preLineSample + "\n" + sample;
            }

            var tp = Util.GetTextPoint(ts.Text, errorPos);
            int posInSample = tp.Chars;

            int numTab = lineSample.Substring(0, posInSample).Count(x => x == '\t');
            int tabOffset = numTab * (4 - 1);
            posInSample += tabOffset;

            sample = sample.Replace('\r', ' ');
            sample = sample.Replace("\t", "    ");

            string posMarker = new string(' ', posInSample) + "↑";

            var sb = new StringBuilder();

            sb.Append(e.Message)
                .Append(" at ").AppendLine(Util.FormatTextPoint(path, tp, tabOffset))
                .AppendLine("  以下の「↑」の付近にエラーがあります。")
                .AppendLine("----------------------------------------")
                .AppendLine(sample)
                .AppendLine(posMarker)
                .AppendLine("----------------------------------------");

            return sb.ToString();
        }

        private static string MakeParserStackString(Parser.Stack stack, string text)
        {
            var sb = new StringBuilder();
            sb.AppendLine("パーサースタック:");

            for (int i = stack.get_stack_level() - 1; i >= 1; --i)
            {
                var sf = stack.get_arg(i, 0);
                var ttp = Util.GetTextPoint(text, sf.pos);
                sb.AppendFormat("  line {0,4}, pos {1,3}: ", ttp.Lines, ttp.Chars);
                if (sf.token != null)
                {
                    sb.AppendLine(sf.token);
                }
                else
                {
                    sb.AppendLine("--");
                }
            }

            return sb.ToString();
        }
        private static string MakeExpectedString(string[] expected)
        {
            var sb = new StringBuilder();

            sb.AppendLine();
            sb.AppendLine("この場所には次のいずれかがあるべきです:");
            foreach (var s in expected)
            {
                sb.AppendFormat("  {0}\n", s);
            }

            return sb.ToString();
        }

        public static SvcDefinition Parse(string text, string path)
        {
            var a = new SvcParserSemanticAction();
            var p = new Parser(a);
            var ts = new TokenStream(text);

            try
            {
                Parse(p, a, ts);
            }
            catch (InvalidCharactorException e)
            {
                throw new ErrorException(
                    MakeExceptionMessage(p, ts, e, path),
                    "認識できない文字です");
            }
            catch (SyntaxErrorException e)
            {
                throw new ErrorException(
                    MakeExceptionMessage(p, ts, e, path),
                    MakeParserStackString(p.stack, text)
                    + MakeExpectedString(e.ExpectedTokens));
            }

            object t;
            p.accept(out t);
            return (SvcDefinition)t;
        }

        private static void Parse(Parser p, SvcParserSemanticAction a, TokenStream ts)
        {
            for (;;)
            {
                var tv = ts.GetNextToken();

                var t = Token.token_eof;
                object v = tv.Value;

                switch (tv.Type)
                {
                case TokenStream.TokenType.Number:
                    {
                        t = Token.token_NUMBER;
                        v = int.Parse(tv.Value);
                    }
                    break;

                case TokenStream.TokenType.String:
                    {
                        t = Token.token_STRING;
                    }
                    break;

                case TokenStream.TokenType.Mark:
                case TokenStream.TokenType.Symbol:
                    {
                        switch (tv.Value)
                        {
                        case "struct":
                            t = Token.token_STRUCT;
                            break;
                        case "union":
                            t = Token.token_UNION;
                            break;
                        case "enum":
                            t = Token.token_ENUM;
                            break;
                        case "types":
                            t = Token.token_TYPES;
                            break;
                        case "operations":
                            t = Token.token_OPERATIONS;
                            break;
                        case "includes":
                            t = Token.token_INCLUDES;
                            break;
                        case "typedef":
                            t = Token.token_TYPEDEF;
                            break;
                        case "abstract":
                            t = Token.token_ABSTRACT;
                            break;
                        case "common":
                            t = Token.token_COMMON;
                            break;

                        case "(":
                            t = Token.token_PAREN_LEFT;
                            break;
                        case ")":
                            t = Token.token_PAREN_RIGHT;
                            break;
                        case "{":
                            t = Token.token_BRACE_LEFT;
                            break;
                        case "}":
                            t = Token.token_BRACE_RIGHT;
                            break;
                        case "[":
                            t = Token.token_BRACKET_LEFT;
                            break;
                        case "]":
                            t = Token.token_BRACKET_RIGHT;
                            break;
                        case ";":
                            t = Token.token_SEMICOLON;
                            break;
                        case ",":
                            t = Token.token_COMMA;
                            break;
                        case "=":
                            t = Token.token_EQUAL;
                            break;
                        default:
                            {
                                if (a.HasType(tv.Value))
                                {
                                    t = Token.token_TYPE_NAME;
                                }
                                else
                                {
                                    t = Token.token_IDENTIFIER;
                                }
                            }
                            break;
                        }
                    }
                    break;

                case TokenStream.TokenType.EOF:
                    {
                        t = Token.token_eof;
                    }
                    break;

                default:
                    {
                        throw new ErrorException("internal error");
                    }
                }

                if (p.post(t, v, tv.Pos))
                {
                    break;
                }
            }
        }
    }
}
