﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    internal class SvcHandlerHeaderGenerator : SourceGenerator
    {
        private PathSet m_pathSet;

        public SvcHandlerHeaderGenerator(CodeGenNames p, PathSet ps) : base(p)
        {
            this.m_pathSet = ps;
        }

        public void Generate(
            SvcSet ss,
            Dictionary<string, AbiLayout> abi,
            Dictionary<string, LayoutConversion> conv,
            string templatePath,
            string path)
        {
            this.Generate(ss, abi, null, conv, templatePath, path);
        }

        protected override string Generate(
            SvcSet ss,
            Dictionary<string, AbiLayout> abi,
            Dictionary<string, SvcLayout> svc,
            Dictionary<string, LayoutConversion> conv)
        {
            var parts = ss.Operations.Select(x => this.Generate(x, abi[x.Name]));
            var returnMaxLength = parts.Max(x => x.Item1.Length);
            var nameMaxLength   = parts.Max(x => x.Item2.Length);

            var lineFormat = string.Format("{0}{1}{2}{3}{4}",
                "    {0,-",
                returnMaxLength,
                "} {1,-",
                nameMaxLength,
                "} {2};\r\n");

            var includes = ss.Categories.Where(c => c.Includes != null)
                                        .SelectMany(c => c.Includes)
                                        .ToLookup(x => x)
                                        .Select(x => x.Key)
                                        .OrderBy(x => x)
                                        .ToArray();

            var sb = new StringBuilder();
            sb.Append(SourceGenerator.MakePragmaOnce());
            sb.AppendLine();
            foreach (var inc in includes)
            {
                sb.Append(SourceGenerator.MakeIncludeLine(inc));
            }
            sb.AppendLine();
            foreach (var category in ss.Categories)
            {
                if (category.Types != null)
                {
                    var inc = ApplyPathTemplate(this.m_pathSet.TypeIncludeTemplate, category.Name);
                    sb.Append(SourceGenerator.MakeIncludeLine(inc));
                }
            }

            sb.AppendLine();
            sb.Append(SourceGenerator.MakeNameSpaceBeginText(CodeGenNames.KernelCommonNamespace));
            sb.AppendLine();

            foreach (var part in parts)
            {
                sb.AppendFormat(lineFormat, part.Item1, part.Item2, part.Item3);
            }

            sb.AppendLine();
            sb.Append(SourceGenerator.MakeNameSpaceEndText(CodeGenNames.KernelCommonNamespace));
            sb.AppendLine();
            return sb.ToString();
        }

        private Tuple<string, string, string> Generate(Operation op, AbiLayout al)
        {
            var paramText = GenerateParamText(op, al, true);

            return new Tuple<string, string, string>(
                op.ReturnType.Name,
                string.Format("{0}{1}{2}", CodeGenNames.HandlerNamePrefix, op.Name, Params.HandlerNamePostfix),
                paramText);
        }
    }
}
