﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace MakeNso.Elf
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.IO;
    using System.Linq;
    using System.Text;

    /// <summary>
    /// ELF ファイルのセグメントを表します。
    /// </summary>
    public sealed class ElfSegmentInfo
    {
        private ElfSegmentInfo(ElfInfo elfInfo, long offset)
        {
            this.FileName = elfInfo.FileName;

            this.Architecture = elfInfo.Architecture;

            this.ByteOrder = elfInfo.ByteOrder;

            using (var fs = File.OpenRead(elfInfo.FileName))
            using (var er = new ElfBinaryReader(fs, elfInfo.Architecture, elfInfo.ByteOrder))
            {
                uint flag = 0;

                fs.Seek(offset, SeekOrigin.Begin);

                this.SegmentType = ConvertToElfSegmentType(er.ReadUInt32());

                if (elfInfo.Architecture == ElfArchitectureType.Elf64)
                {
                    flag = er.ReadUInt32();
                }

                this.SegmentOffset = er.ReadWord();

                this.VirtualAddress = er.ReadWord();

                er.ReadWord(); // physical address

                this.SegmentSize = er.ReadWord();

                this.MemorySize = er.ReadWord();

                if (elfInfo.Architecture == ElfArchitectureType.Elf32)
                {
                    flag = er.ReadUInt32();
                }

                this.Executable = (flag & 0x01) != 0;

                this.Writable = (flag & 0x02) != 0;

                this.Readable = (flag & 0x04) != 0;

                this.Alignment = er.ReadWord();
            }

            this.SegmentNames = (new List<string>()).AsReadOnly();
        }

        /// <summary>
        /// ELF ファイルのパスを取得します。
        /// </summary>
        public string FileName { get; private set; }

        /// <summary>
        /// アーキテクチャタイプを取得します。
        /// </summary>
        public ElfArchitectureType Architecture { get; private set; }

        /// <summary>
        /// バイトオーダタイプを取得します。
        /// </summary>
        public ElfByteOrderType ByteOrder { get; private set; }

        /// <summary>
        /// セグメントタイプを取得します。
        /// </summary>
        public ElfSegmentType SegmentType { get; private set; }

        /// <summary>
        /// セグメントオフセットを取得します。
        /// </summary>
        public ulong SegmentOffset { get; private set; }

        /// <summary>
        /// バーチャルアドレスを取得します。
        /// </summary>
        public ulong VirtualAddress { get; private set; }

        /// <summary>
        /// セグメントサイズを取得します。
        /// </summary>
        public ulong SegmentSize { get; private set; }

        /// <summary>
        /// メモリサイズを取得します。
        /// </summary>
        public ulong MemorySize { get; private set; }

        /// <summary>
        /// アライメントを取得します。
        /// </summary>
        public ulong Alignment { get; private set; }

        /// <summary>
        /// セグメント中に含まれるセクションの名前を取得します。
        /// </summary>
        public IReadOnlyList<string> SegmentNames { get; private set; }

        private bool Executable { get; set; }

        private bool Writable { get; set; }

        private bool Readable { get; set; }

        /// <summary>
        /// 実行可能なセグメントかどうかを示す値を返します。
        /// </summary>
        /// <returns>実行可能なセグメントかどうかを示す値です。</returns>
        public bool IsExecutable()
        {
            return this.Executable;
        }

        /// <summary>
        /// 書き込み可能なセグメントかどうかを示す値を返します。
        /// </summary>
        /// <returns>書き込み可能なセグメントかどうかを示す値です。</returns>
        public bool IsWritable()
        {
            return this.Writable;
        }

        /// <summary>
        /// 読み込み可能なセグメントかどうかを示す値を返します。
        /// </summary>
        /// <returns>読み込み可能なセグメントかどうかを示す値です。</returns>
        public bool IsReadable()
        {
            return this.Readable;
        }

        /// <summary>
        /// セグメントからバイナリデータを読み取ります。
        /// </summary>
        /// <returns>読み取ったバイナリデータです。</returns>
        public byte[] GetContents()
        {
            if (this.SegmentOffset > long.MaxValue)
            {
                throw new ArgumentException(Properties.Resources.Message_InvalidSegumentOffset);
            }

            if (this.SegmentSize > int.MaxValue)
            {
                throw new ArgumentException(Properties.Resources.Message_InvalidSegmentSize);
            }

            using (var fs = File.OpenRead(this.FileName))
            using (var br = new BinaryReader(fs))
            {
                fs.Seek((long)this.SegmentOffset, SeekOrigin.Begin);

                return br.ReadBytes((int)this.SegmentSize);
            }
        }

        /// <summary>
        /// ELF データのプログラムヘッダテーブルからセグメントを読み取ります。
        /// </summary>
        /// <param name="elfInfo">ELF データです。</param>
        /// <param name="offset">プログラムヘッダテーブルの位置を示すオフセットです。</param>
        /// <param name="size">プログラムヘッダテーブルのエントリサイズです。</param>
        /// <param name="length">プログラムヘッダテーブルのエントリ数です。</param>
        /// <param name="sectionInfos">ELF データ中に含まれるセクションです。</param>
        /// <returns>読み取ったセグメントです。</returns>
        internal static IReadOnlyList<ElfSegmentInfo> GetElfSegmentInfos(ElfInfo elfInfo, ulong offset, ushort size, ushort length, IReadOnlyList<ElfSectionInfo> sectionInfos)
        {
            if (!IsValidElfSegmentHeaderEntrySize(size, elfInfo.Architecture))
            {
                throw new ArgumentException(Properties.Resources.Message_InvalidProgramHeaderEntrySize);
            }

            if (offset > long.MaxValue)
            {
                throw new ArgumentException(Properties.Resources.Message_InvalidProgramHeaderTableOffset);
            }

            var list = new List<ElfSegmentInfo>();

            for (long i = 0; i < length; ++i)
            {
                var entryOffset = (long)offset + (size * i);

                list.Add(new ElfSegmentInfo(elfInfo, entryOffset));
            }

            foreach (var segmentInfo in list)
            {
                if (segmentInfo.SegmentType == ElfSegmentType.LOAD)
                {
                    segmentInfo.SegmentNames = GetSegmentNamesFromSectionInfos(segmentInfo, sectionInfos);
                }
            }

            return new ReadOnlyCollection<ElfSegmentInfo>(list);
        }

        private static ElfSegmentType ConvertToElfSegmentType(uint value)
        {
            switch (value)
            {
                case 1: return ElfSegmentType.LOAD;
                case 6: return ElfSegmentType.PHDR;
                default: return ElfSegmentType.Unknown;
            }
        }

        private static bool IsValidElfSegmentHeaderEntrySize(ushort size, ElfArchitectureType architecture)
        {
            if (architecture == ElfArchitectureType.Elf32 && size == 32)
            {
                return true;
            }

            if (architecture == ElfArchitectureType.Elf64 && size == 56)
            {
                return true;
            }

            return false;
        }

        private static IReadOnlyList<string> GetSegmentNamesFromSectionInfos(ElfSegmentInfo segmentInfo, IReadOnlyList<ElfSectionInfo> sectionInfos)
        {
            var names = new List<string>();

            foreach (var sectionInfo in sectionInfos)
            {
                if (segmentInfo.VirtualAddress > sectionInfo.VirtualAddress)
                {
                    continue;
                }

                if (segmentInfo.VirtualAddress + segmentInfo.MemorySize < sectionInfo.VirtualAddress + sectionInfo.SectionSize)
                {
                    continue;
                }

                names.Add(sectionInfo.SectionName);
            }

            return names.AsReadOnly();
        }
    }
}
