﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace MakeNro.Elf
{
    using System;

    /// <summary>
    /// ELF ファイルのノートセクションを表します。
    /// </summary>
    public class ElfNoteSectionInfo
    {
        public ElfNoteSectionInfo(ElfSectionInfo section)
        {
            byte[] contents = section.GetContents();

            this.NameSize = BitConverter.ToUInt32(contents, 0);
            this.DescSize = BitConverter.ToUInt32(contents, 4);
            this.NoteType = BitConverter.ToUInt32(contents, 8);

            long offset = 12;
            this.Name = new byte[this.NameSize];
            int align = (section.Architecture == ElfArchitectureType.Elf64) ? 8 : 4;

            Array.Copy(contents, offset, this.Name, 0, (long)this.NameSize);
            offset += (long)this.NameSize;
            offset = (long)RoundUp((ulong)offset, (ulong)align);

            this.Desc = new byte[this.DescSize];
            Array.Copy(contents, offset, this.Desc, 0, (long)this.DescSize);
        }

        /// <summary>
        /// ノートセクションの名前のサイズを取得します。
        /// </summary>
        public ulong NameSize { get; private set; }

        /// <summary>
        /// ノートセクションの記述のサイズを取得します。
        /// </summary>
        public ulong DescSize { get; private set; }

        /// <summary>
        /// ノートセクションの種類を取得します。
        /// </summary>
        public ulong NoteType { get; private set; }

        /// <summary>
        /// ノートセクションの名前を取得します。
        /// </summary>
        public byte[] Name { get; private set; }

        /// <summary>
        /// ノートセクションの記述を取得します。
        /// </summary>
        public byte[] Desc { get; private set; }

        private ulong RoundUp(ulong src, ulong align)
        {
            return (src + (align - 1)) & ~(align - 1);
        }
    }
}
