﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace MakeInitialProgram.Elf
{
    using System;
    using System.IO;

    /// <summary>
    /// プリミティブ データ型を特定のアーキテクチャタイプおよびバイトオーダタイプのバイナリ値として読み取ります。
    /// </summary>
    internal sealed class ElfBinaryReader : IDisposable
    {
        /// <summary>
        /// ElfBinaryReader クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="fs">入力ファイルストリームです。</param>
        /// <param name="architecture">アーキテクチャタイプです。</param>
        /// <param name="byteOrder">バイトオーダタイプです。</param>
        internal ElfBinaryReader(FileStream fs, ElfArchitectureType architecture = ElfArchitectureType.Elf32, ElfByteOrderType byteOrder = ElfByteOrderType.LittleEndian)
        {
            this.BinaryReader = new BinaryReader(fs);

            this.Architecture = architecture;

            this.ByteOrder = byteOrder;
        }

        /// <summary>
        /// アーキテクチャタイプを取得または設定します。
        /// </summary>
        internal ElfArchitectureType Architecture { get; set; }

        /// <summary>
        /// バイトオーダタイプを取得または設定します。
        /// </summary>
        internal ElfByteOrderType ByteOrder { get; set; }

        private BinaryReader BinaryReader { get; set; }

        /// <summary>
        /// ElfBinaryReader クラスの現在のインスタンスによって使用されているすべてのリソースを解放します。
        /// </summary>
        public void Dispose()
        {
            if (this.BinaryReader != null)
            {
                this.BinaryReader.Close();
            }
        }

        /// <summary>
        /// ファイルストリームから 1 バイトを読み取り、ストリームの位置を 1 バイト進めます。
        /// </summary>
        /// <returns>読み取ったバイトです。</returns>
        internal byte ReadByte()
        {
            return this.BinaryReader.ReadByte();
        }

        /// <summary>
        /// ファイルストリームから指定されたバイト数を読み取り、ストリームの位置をそのバイト数だけ進めます。
        /// </summary>
        /// <param name="count">読み取るバイト数です。</param>
        /// <returns>読み取ったバイトです。</returns>
        internal byte[] ReadBytes(int count)
        {
            return this.BinaryReader.ReadBytes(count);
        }

        /// <summary>
        /// ファイルストリームから 2 バイト符号なし整数を読み取り、ストリームの位置を 2 バイト進めます。
        /// </summary>
        /// <returns>読み取った 2 バイト符号なし整数です。</returns>
        internal ushort ReadUInt16()
        {
            var value = this.BinaryReader.ReadUInt16();

            if (this.ByteOrder == ElfByteOrderType.LittleEndian)
            {
                return value;
            }

            return (ushort)(((value & 0x00FF) << 8) | ((value & 0xFF00) >> 8));
        }

        /// <summary>
        /// ファイルストリームから 4 バイト符号なし整数を読み取り、ストリームの位置を 4 バイト進めます。
        /// </summary>
        /// <returns>読み取った 4 バイト符号なし整数です。</returns>
        internal uint ReadUInt32()
        {
            var value = this.BinaryReader.ReadUInt32();

            if (this.ByteOrder == ElfByteOrderType.LittleEndian)
            {
                return value;
            }

            return (uint)(((value & 0x000000FFU) << 24) |
                          ((value & 0x0000FF00U) << 8) |
                          ((value & 0x00FF0000U) >> 8) |
                          ((value & 0xFF000000U) >> 24));
        }

        /// <summary>
        /// ファイルストリームから 8 バイト符号なし整数を読み取り、ストリームの位置を 8 バイト進めます。
        /// </summary>
        /// <returns>読み取った 8 バイト符号なし整数です。</returns>
        internal ulong ReadUInt64()
        {
            var value = this.BinaryReader.ReadUInt64();

            if (this.ByteOrder == ElfByteOrderType.LittleEndian)
            {
                return value;
            }

            return (ulong)(((value & 0x00000000000000FFUL) << 56) |
                           ((value & 0x000000000000FF00UL) << 40) |
                           ((value & 0x0000000000FF0000UL) << 24) |
                           ((value & 0x00000000FF000000UL) << 8) |
                           ((value & 0x000000FF00000000UL) >> 8) |
                           ((value & 0x0000FF0000000000UL) >> 24) |
                           ((value & 0x00FF000000000000UL) >> 40) |
                           ((value & 0xFF00000000000000UL) >> 56));
        }

        /// <summary>
        /// ファイルストリームから 1 ワードを読み取り、ストリームの位置を 1 ワード進めます。
        /// </summary>
        /// <returns>読み取った 1 ワードです。</returns>
        internal ulong ReadWord()
        {
            if (this.Architecture == ElfArchitectureType.Elf32)
            {
                return (ulong)this.ReadUInt32();
            }
            else
            {
                return this.ReadUInt64();
            }
        }
    }
}
