﻿using System;
using System.IO;
using System.IO.Compression;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Collections.ObjectModel;
using System.Text.RegularExpressions;
using Nintendo.Foundation.IO;
using YamlDotNet.Serialization;
using CommandUtility;
using MakeFirmwareArchive;

namespace MakeFirmwareArchive
{
    public class CreateNfaArguments
    {
        [CommandLineOption('o', "output", Description = " ", IsRequired = true)]
        public string OutputPath { get; set; }

        [CommandLineOption('i', "input", Description = " ", IsRequired = true)]
        public string[] InputPath { get; set; }

        [CommandLineOption('D', "define", Description = " ", IsRequired = false)]
        public string[] Arguments { get; set; }

        [CommandLineOption('v', "verbose", Description = " ", IsRequired = false)]
        public bool IsVerbose { get; set; }

        public Dictionary<string, string> MakeVariablesDictionary()
        {
            var dictionary = new Dictionary<string, string>();
            if (Arguments == null)
            {
                return new Dictionary<string, string>();
            }
            else
            {
                foreach (var text in Arguments)
                {
                    var splitted = text.Split(new char[] { '=' }, 2);

                    if (splitted.Length != 2)
                    {
                        throw new ArgumentException("Invalid variable definition format: " + text);
                    }

                    dictionary[splitted[0]] = splitted[1];
                }

                return dictionary;
            }
        }
    }
    public class CreateNfaCommand
    {
        //!< コンフィグに含まれる全てのファイルが存在するか調べます
        static bool checkAllFileIsExist(FirmwareArchiveConfig archiveConfig)
        {
            bool result = true;
            foreach (var attribute in archiveConfig.Attributes)
            {
                foreach (var path in attribute.Path)
                {
                    if (!File.Exists(path))
                    {
                        Console.WriteLine("{0} is not found.", attribute.Path);
                        result = false;
                    }
                }
            }

            return result;
        }

        //!< コンフィグに含まれるパーティション名が正しいかを調べます
        static bool checkAllPartitionNameIsCollect(FirmwareArchiveConfig archiveConfig)
        {
            bool result = true;
            foreach (var attribute in archiveConfig.Attributes)
            {
                if (!FirmwareArchivePartition.IsValidPartitionName(attribute.Name))
                {
                    Console.WriteLine("Invalid partition name = {0}", attribute.Name);
                    result = false;
                }
            }

            return result;
        }

        //!< 生成されたコンフィグのベリフィケーションを行います
        static bool VerifyArchiveConfig(FirmwareArchiveConfig archiveConfig)
        {
            bool result = true;
            result &= checkAllFileIsExist(archiveConfig);
            result &= checkAllPartitionNameIsCollect(archiveConfig);

            return result;
        }

        //!< 指定されたアーカイブコンフィグを出力します
        public static void OutputArchiveConfig(FirmwareArchiveConfig archiveConfig)
        {
            Console.WriteLine("ConfigName = {0}", archiveConfig.Name);
            Console.WriteLine("Description = {0}", archiveConfig.Description);
            foreach (var attribute in archiveConfig.Attributes)
            {
                Console.WriteLine("AttributeName = {0}", attribute.Name);
                foreach (var path in attribute.Path)
                {
                    Console.WriteLine("- Path = {0}", path);
                }
            }
        }

        public static string ReplaceVariables(string text, Dictionary<string, string> dictionary)
        {
            var patterns = from tuple in dictionary
                           select new Tuple<string, Regex>(tuple.Value, new Regex("(?<!\\\\)\\$\\{" + tuple.Key + "\\}"));
            return patterns.Aggregate<Tuple<string, Regex>, string>(
                text,
                (replaced, variable) => {
                    return variable.Item2.Replace(replaced, variable.Item1);
                });
        }

        //!< yaml ファイルをパースします
        public static FirmwareArchiveConfig ParseFirmwareArchiveConfig(string inputPath, Dictionary<string, string> variables)
        {
            using (var input = new StreamReader(inputPath, Encoding.UTF8))
            {
                var text = input.ReadToEnd();
                var replacedText = ReplaceVariables(text, variables);

                using (var replacedTextReader = new StringReader(replacedText))
                {
                    var yamlDeserializer = new Deserializer();
                    var config = yamlDeserializer.Deserialize<FirmwareArchiveConfig>(replacedTextReader);
                    return config;
                }
            }
        }

        //!< sourcePath にあるディレクトリの内容を再帰的にコピーします。
        public static void CopyDirectory(string sourcePath, string destinationPath)
        {
            Directory.CreateDirectory(destinationPath);
            foreach (var file in Directory.GetFiles(sourcePath))
            {
                File.Copy(file, Path.Combine(destinationPath, Path.GetFileName(file)));
            }

            foreach (var dir in Directory.GetDirectories(sourcePath))
            {
                CopyDirectory(dir, Path.Combine(destinationPath, Path.GetFileName(dir)));
            }
        }

        //!< 初期化イメージコンフィグをアーカイブに保存します
        public static void SaveInitialImageConfig(DirectoryInfo workingDirectoryRoot,
                                                  Dictionary<string, string> variables)
        {
            //!< 初期化イメージコンフィグの保存先のキー
            const string initialImageConfigKey = "INITIALIMAGE_CONFIG";
            if (!variables.ContainsKey(initialImageConfigKey))
            {
                return;
            }

            string path = variables[initialImageConfigKey];
            string destinationPath = workingDirectoryRoot.FullName;

            destinationPath = Path.Combine(destinationPath, "config");

            Directory.CreateDirectory(destinationPath);

            File.Copy(path, Path.Combine(destinationPath, Path.GetFileName(path)));

            replaceInitialImageConfig(Path.Combine(destinationPath, Path.GetFileName(path)), variables);
        }

        //!< 初期化イメージコンフィグを置き換えます
        public static void replaceInitialImageConfig(string path, Dictionary<string, string> variables)
        {
            string replacedText;

            using (var input = new StreamReader(path, Encoding.UTF8))
            {
                var text = input.ReadToEnd();
                replacedText = ReplaceVariables(text, variables);
            }

            using (var writer = new StreamWriter(path, false, Encoding.UTF8))
            {
                writer.Write(replacedText);

            }
        }

        //!< yaml ファイルからアーカイブ作成に必要なコンフィグデータを作成します。
        public static FirmwareArchiveConfig MakeArchiveConfigFromYaml(string inputPath,
                                                                      DirectoryInfo workingDirectoryRoot,
                                                                      Dictionary<string, string> variables)
        {
            var config = ParseFirmwareArchiveConfig(inputPath, variables);

            //一時領域に移して zip 化する
            foreach (var attribute in config.Attributes)
            {
                var workingDirectory = workingDirectoryRoot.CreateSubdirectory(attribute.Name);

                foreach (var path in attribute.Path)
                {
                    if (File.Exists(path))
                    {
                        string targetPath = Path.Combine(workingDirectory.FullName, Path.GetFileName(path));

                        File.Copy(path, targetPath, true);
                    }
                    else if(Directory.Exists(path))
                    {
                        CopyDirectory(path, workingDirectory.FullName);
                    }
                    else
                    {
                        throw new Exception(string.Format("Target path is not found: {0}", path));
                    }
                }
            }

            SaveInitialImageConfig(workingDirectoryRoot, variables);

            return config;
        }

        //!< 作業ディレクトリからアーカイブ作成に必要なコンフィグデータを作成します
        public static FirmwareArchiveConfig MakeArchiveConfigFromWorkingDir(DirectoryInfo workingDirectoryRoot, string name)
        {
            var workingDirectory = workingDirectoryRoot.FullName;

            var config = new FirmwareArchiveConfig();
            //対象のアーカイブの名前を Name に設定
            config.Name = Path.GetFileName(name);
            //説明は失われるので入れない
            config.Description = "";

            //ディレクトリを再帰探索する
            foreach (var path in Directory.EnumerateDirectories(workingDirectory))
            {
                var attribute = new ContentAttribute();
                attribute.Name = Path.GetFileName(path);

                foreach (var filePath in Directory.EnumerateFiles(path))
                {
                    attribute.Path.Add(filePath);
                }

                config.Attributes.Add(attribute);
            }

            return config;
        }

        //!< アーカイブからアーカイブ内容に関する config ファイルを作成します
        public static FirmwareArchiveConfig MakeArchiveConfigFromNfa(string inputPath)
        {
            if (!IsNfaExtension(inputPath))
            {
                throw new Exception("Invalid file : " + inputPath);
            }

            var mergedContents = new Dictionary<string, HashSet<string>>();
            var config = new FirmwareArchiveConfig();

            using (ZipArchive archive = ZipFile.OpenRead(inputPath))
            {
                foreach (ZipArchiveEntry entry in archive.Entries)
                {
                    var directoryName = Path.GetDirectoryName(entry.FullName);
                    var fileName = Path.Combine(inputPath, entry.FullName);

                    if (!mergedContents.ContainsKey(directoryName))
                    {
                        mergedContents[directoryName] = new HashSet<string>();
                    }

                    mergedContents[directoryName].Add(fileName);
                }
            }

            foreach (var content in mergedContents)
            {
                var attribute = new ContentAttribute();
                attribute.Name = content.Key;
                attribute.Path = content.Value;

                config.Attributes.Add(attribute);
            }

            return config;

        }

        //!< アーカイブからアーカイブ作成に利用する yaml ファイルを作成します。
        public static void MakeYamlFromArchiveConfig(FirmwareArchiveConfig config, string outputPath)
        {
            var yamlSerializer = new Serializer();

            using (var streamWriter = new StreamWriter(outputPath))
            {
                streamWriter.AutoFlush = true;
                var textWriter = TextWriter.Synchronized(streamWriter);

                yamlSerializer.Serialize(textWriter, config);
            }
        }

        static bool IsNfaExtension(string path)
        {
            var extension = Path.GetExtension(path);
            return extension == NfaExtension;
        }

        static bool IsYamlExtension(string path)
        {
            var extension = Path.GetExtension(path);
            return extension == YamlExtension;
        }

        static void AssembleFirmwareComponents(string inputPath, DirectoryInfo workingDirectoryRoot, Dictionary<string, string> variables)
        {
            if (IsYamlExtension(inputPath))
            {
                MakeArchiveConfigFromYaml(inputPath, workingDirectoryRoot, variables);
            }
            else if (IsNfaExtension(inputPath))
            {
                FirmwareArchiveDirectory.FromFirmwareArchive(workingDirectoryRoot, new FileInfo(inputPath));
            }
            else
            {
                throw new Exception("Unsupported file format: " + inputPath);
            }
        }

        //!< FirmwareArchive を作成します。
        public static bool MakeFirmwareArchive(string[] inputPath, string outputPath, Dictionary<string, string> variables)
        {
            using (var tempHolder = new TemporaryFileHolder("MakeFirmwareArchive"))
            {
                var name = Path.GetFileName(outputPath);
                var workingDirectoryRoot = tempHolder.CreateTemporaryDirectory(name);

                foreach (var inputFile in inputPath)
                {
                    AssembleFirmwareComponents(inputFile, workingDirectoryRoot, variables);
                }

                var config = MakeArchiveConfigFromWorkingDir(workingDirectoryRoot, name);

                if (!VerifyArchiveConfig(config))
                {
                    throw new Exception("Verification failed.");
                }

                var firmwareArchiveDirectory = new FirmwareArchiveDirectory(
                                                    new DirectoryInfo(workingDirectoryRoot.FullName));

                firmwareArchiveDirectory.ArchiveNfa(new FileInfo(outputPath));
            }

            return true;
        }

        const string ProdExtension = ".prod.nfa";
        const string NfaExtension = ".nfa";
        const string YamlExtension = ".yml";
    }
}
