﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Xml;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;
using System.Security.Cryptography;

namespace MakeDesc
{
    /// <summary>
    /// .desc ファイルを管理するクラス
    /// </summary>
    internal class DescFile
    {
        public KeyManager KeyManager { get; set; }

        private Acid AcidFile;

        private DescModel.InputDescModel InputDesc;

        // For Acid
        private FacDescriptor Facd;
        private SacDescriptor Sacd;
        private KcDescriptor Kcd;

        // For Aci or Meta
        private FacData Fac;
        private SacData Sac;
        private KcData Kc;

        /// <summary>
        /// DescFile のコンストラクタ
        /// </summary>
        /// <param name="info">システムコールの情報</param>
        /// <param name="kernelVersionInfo">オプションで与えたカーネルバージョン</param>
        /// <param name="descVersionInfo">オプションで与えた desc バージョン</param>
        public DescFile(SystemCallInfo info, string kernelVersionInfo, string descVersionInfo)
        {
            AcidFile = new Acid();
            Facd = new FacDescriptor();
            Sacd = new SacDescriptor();
            Kcd = new KcDescriptor(info, kernelVersionInfo);
            if (descVersionInfo != null)
            {
                AcidFile.DescVersion = byte.Parse(descVersionInfo);
            }
            AcidFile.Facd = Facd;
            AcidFile.Sacd = Sacd;
            AcidFile.Kcd = Kcd;

            Fac = new FacData();
            Sac = new SacData();
            Kc = new KcData(info, kernelVersionInfo);
        }

        /// <summary>
        /// desc ファイルを読み込みます
        /// </summary>
        /// <param name="desc">.desc ファイル</param>
        public void ImportDescFile(DescModel.InputDescModel desc)
        {
            this.InputDesc = desc;

            if (desc.Default == null)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_NotFoundElement, "Default"));
            }

            if (this.InputDesc.MemoryRegion == 0xFF)
            {
                // 未設定なら ProgramType に応じてデフォルト値を設定
                if (this.InputDesc.KernelCapabilityDescriptor == null || this.InputDesc.KernelCapabilityDescriptor.MiscParams == null || this.InputDesc.KernelCapabilityDescriptor.MiscParams.ProgramTypeValue == 0)
                {
                    // システムプロセス（3 = NonSecure にする場合は独自に設定が必要）
                    this.InputDesc.MemoryRegion = 2;
                }
                else if (this.InputDesc.KernelCapabilityDescriptor.MiscParams.ProgramTypeValue == 1)
                {
                    // アプリケーション
                    this.InputDesc.MemoryRegion = 0;
                }
                else if (this.InputDesc.KernelCapabilityDescriptor.MiscParams.ProgramTypeValue == 2)
                {
                    // アプレット
                    this.InputDesc.MemoryRegion = 1;
                }
                else
                {
                    Console.WriteLine("Invalid MemoryRegion: {0}", this.InputDesc.MemoryRegion);
                }
            }

            desc.Default.CheckReadSuccess();

            AcidFile.ProgramIdMin = this.InputDesc.ProgramIdMinValue;

            AcidFile.ProgramIdMax = this.InputDesc.ProgramIdMaxValue;

            AcidFile.ProductionFlag = this.InputDesc.ProductionFlagValue;

            AcidFile.MemoryRegion = this.InputDesc.MemoryRegion;

            // 個別 desc で desc version が指定されている場合は一括指定の値を上書き
            if (this.InputDesc.DescVersion != 0)
            {
                AcidFile.DescVersion = this.InputDesc.DescVersion;
            }

            AcidFile.UnqualifiedApproval = this.InputDesc.UnqualifiedApprovalValue;

            Facd.ImportDescFile(desc);
            Sacd.ImportDescFile(desc);
            Kcd.ImportDescFile(desc);

            Fac.ImportDescFile(desc);
            Sac.ImportDescFile(desc);
            Kc.ImportDescFile(desc);

            Kcd.CheckCapabilities(Kc);
            Sacd.CheckCapabilities(Sac);
        }

        /// <summary>
        /// 作成した .desc ファイルを書きだします。
        /// </summary>
        /// <param name="fs">書き出すファイルストリーム</param>
        public void OutputDescFile(Stream fs)
        {
            DescModel.OutputDescModel model = new DescModel.OutputDescModel();

            model.MemoryRegionValue = this.InputDesc.MemoryRegion;

            model.ProgramIdMin = this.InputDesc.ProgramIdMin;
            model.ProgramIdMax = this.InputDesc.ProgramIdMax;

            model.Default = new DescModel.DefaultModelWithSvcInfo();
            model.Default.Is64BitInstruction = this.InputDesc.Default.Is64BitInstruction;
            model.Default.ProcessAddressSpace = this.InputDesc.Default.ProcessAddressSpace;
            model.Default.MainThreadPriority = this.InputDesc.Default.MainThreadPriority;
            model.Default.MainThreadCoreNumber = this.InputDesc.Default.MainThreadCoreNumber;
            model.Default.MainThreadStackSize = this.InputDesc.Default.MainThreadStackSize;

            Facd.ExportDescFile(ref model);
            Sacd.ExportDescFile(ref model);
            Kcd.ExportDescFile(ref model);
            Fac.ExportDescFile(ref model);
            Sac.ExportDescFile(ref model);
            Kc.ExportDescFile(ref model);


            if (this.KeyManager.HasKey())
            {
                model.RSAKeyValue = this.KeyManager.HeaderKey;
            }

            byte[] acidBinary = AcidFile.ExportAcidBinary(this.KeyManager);
            model.Acid = Convert.ToBase64String(acidBinary);

            System.Xml.Serialization.XmlSerializerNamespaces ns = new System.Xml.Serialization.XmlSerializerNamespaces();
            ns.Add(String.Empty, String.Empty);

            System.Xml.XmlWriterSettings settings = new XmlWriterSettings();
            settings.Indent = true;
            settings.IndentChars = "    ";

            System.Xml.XmlWriter xmlWriter = System.Xml.XmlWriter.Create(fs, settings);

            System.Xml.Serialization.XmlSerializer serializer = new System.Xml.Serialization.XmlSerializer(typeof(DescModel.OutputDescModel));
            serializer.Serialize(xmlWriter, model, ns);
        }

        internal class StreamWriterWithEncoding : StreamWriter
        {
            internal StreamWriterWithEncoding(Stream s, Encoding e)
                : base(s)
            {
                m_Encoding = e;
            }

            private readonly Encoding m_Encoding;

            public override Encoding Encoding
            {
                get
                {
                    return this.m_Encoding;
                }
            }
        }
    }
}
