﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeAddrRegionHeader
{
    internal struct Symbol : IEquatable<Symbol>
    {
        public string Name { get; private set; }
        public AddrValue Value { get; private set; }

        public Symbol(string name, AddrValue value)
            : this()
        {
            this.Name = name;
            this.Value = value;
        }

        #region IEquatable<Symbol> メンバー

        public bool Equals(Symbol other)
        {
            return Name.Equals(other.Name) && Value.Equals(other.Value);
        }

        #endregion

        public override bool Equals(object obj)
        {
            if (obj == null || GetType() != obj.GetType())
            {
                return false;
            }

            return Equals((Symbol)obj);
        }

        public override int GetHashCode()
        {
            return Name.GetHashCode() ^ (Value.GetHashCode() << 7);
        }

        public override string ToString()
        {
            return string.Format("{0}({1})", Name, Value);
        }
    }

    internal interface ISymbolEmitter
    {
        IEnumerable<Symbol> EnumerateSymbols();
    }

    internal class AddrConstant : ISymbolEmitter
    {
        public string Name { get; private set; }
        public AddrValue Value { get; private set; }

        public AddrConstant(string name, AddrValue value)
        {
            if (!AddrUtil.IsWithinAddrSpace(value))
            {
                throw new AddrOverflowException(
                    string.Format(CultureInfo.CurrentCulture,
                        "定数 {0}: 値がアドレス空間外です。({1})", name, value));
            }

            this.Name = name;
            this.Value = value;
        }

        public IEnumerable<Symbol> EnumerateSymbols()
        {
            yield return new Symbol(Name, Value);
        }
    }

    internal class AddrRegion : ISymbolEmitter
    {
        public string Name { get; private set; }
        public AddrValue Begin { get; private set; }
        public AddrValue Size { get { return End - Begin; } }
        public AddrValue End { get; private set; }

        public AddrRegion(string name, AddrValue begin, AddrValue end)
        {
            if (!(begin <= end))
            {
                throw new AddrRangeException(
                    string.Format(CultureInfo.CurrentCulture,
                        "領域 {0}: begin <= end でなければなりません。({1} > {2})", name, begin, end));
            }

            if (!AddrUtil.IsWithinAddrSpace(begin))
            {
                throw new AddrOverflowException(
                    string.Format(CultureInfo.CurrentCulture,
                        "領域 {0}: begin がアドレス空間外です。({1})", name, begin));
            }

            this.Name = name;
            this.Begin = begin;
            this.End = end;
        }

        public IEnumerable<Symbol> EnumerateSymbols()
        {
            yield return new Symbol(SymbolUtil.GetRegionBeginSymbolName(Name), Begin);
            // End がアドレス空間の終端であれば、出力しない
            if (End.Value != AddrUtil.AddrValueEnd(End.AddrWidth))
            {
                yield return new Symbol(SymbolUtil.GetRegionEndSymbolName(Name), End);
            }
            yield return new Symbol(SymbolUtil.GetRegionSizeSymbolName(Name), Size);
        }

        public AddrValue GetValue(RegionValueType regionValueType)
        {
            switch (regionValueType)
            {
                case RegionValueType.Begin:
                    return Begin;
                case RegionValueType.End:
                    return End;
                case RegionValueType.Size:
                    return Size;
                default:
                    throw new ArgumentException("RegionValueType として無効な値です。", "regionValueType");
            }
        }

        public override string ToString()
        {
            return string.Format(CultureInfo.InvariantCulture, "{0} [{1}, {2})", Name, Begin, End);
        }
    }
}
