﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Globalization;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace MakeAddrRegionHeader
{
    internal class StatementParser
    {
        private static readonly StatementDefinition[] StatementDefinitions;

        static StatementParser()
        {
            var statementTypes = new[]
            {
                typeof(BlankStatement),
                typeof(IncludeStatement),
                typeof(ConstantStatement),
                typeof(AbsoluteBeginSizeStatement),
                typeof(AbsoluteBeginEndStatement),
                typeof(RelativeInnerOuterStatement),
                typeof(RelativeShiftStatement),
                typeof(AliasRegionStatement),
                typeof(AliasConstantStatement),
            };

            // Statement の派生クラスから文のパターン（ArgumentType の列）とコンストラクタを取得して
            // statementDefinitions に設定します。
            StatementDefinitions = statementTypes.Select(t => MakeStatementDefinition(t)).ToArray();
        }

        private static StatementDefinition MakeStatementDefinition(Type statementType)
        {
            var pattern = (StatementPatternAttribute)Attribute.GetCustomAttribute(statementType, typeof(StatementPatternAttribute));
            Debug.Assert(pattern != null,
                string.Format(CultureInfo.CurrentCulture, "クラス {0} に StatementPattern 属性が設定されていません。", statementType.FullName));

            // Statement 派生クラスをリフレクション経由でコンストラクトするのは、
            // コンストラクタが別コンテキストで実行される可能性があり
            // 例外が TargetInvocationException 経由で throw されてくるため避ける。
            // Statement 派生クラス型からファクトリメソッドを取得する。
            var getFactory = statementType.GetMethod("GetFactory", BindingFlags.Public | BindingFlags.Static | BindingFlags.DeclaredOnly);
            Debug.Assert(getFactory != null,
                string.Format(CultureInfo.CurrentCulture, "クラス {0} に GetFactory static メソッドが存在しません。", statementType.FullName));

            var factory = (Func<IReadOnlyList<string>, ParseContext, Statement>)getFactory.Invoke(null, null);

            return new StatementDefinition(pattern, factory);
        }

        public static Statement Parse(string line, ParseContext context)
        {
            var arguments = LineSplitter.SplitToArguments(line);
            var statementDefinition = FindMatchingDefinition(arguments);
            if (statementDefinition == null)
            {
                throw new SyntaxErrorException(
                    string.Format(CultureInfo.CurrentCulture, "不明な構文です。: {0}", line));
            }

            return statementDefinition.CreateStatement(arguments.ToList(), context);
        }

        private static StatementDefinition FindMatchingDefinition(IEnumerable<string> arguments)
        {
            var argumentTypes = arguments.Select(s => ArgumentUtil.GetArgumentType(s));
            return StatementDefinitions.FirstOrDefault(a => a.Match(argumentTypes));
        }

        private static class LineSplitter
        {
            private static Regex commentRegex = new Regex(@"(#|//).*$", RegexOptions.Compiled);
            private static readonly char[] WhitespaceCharacters = new char[] { '\t', ' ' };

            public static string[] SplitToArguments(string line)
            {
                return Split(TrimComment(line));
            }

            public static string TrimComment(string commentedString)
            {
                return commentRegex.Replace(commentedString, string.Empty);
            }

            public static string[] Split(string s)
            {
                return s.Split(WhitespaceCharacters, StringSplitOptions.RemoveEmptyEntries);
            }
        }

        private class StatementDefinition
        {
            private StatementPatternAttribute pattern;
            private Func<IReadOnlyList<string>, ParseContext, Statement> statementFactory;

            public StatementDefinition(
                StatementPatternAttribute pattern, Func<IReadOnlyList<string>, ParseContext, Statement> statementFactory)
            {
                this.pattern = pattern;
                this.statementFactory = statementFactory;
            }

            public bool Match(IEnumerable<ArgumentType> argumentTypes)
            {
                return pattern.Match(argumentTypes);
            }

            public Statement CreateStatement(IReadOnlyList<string> arguments, ParseContext context)
            {
                return statementFactory(arguments, context);
            }
        }
    }

    [AttributeUsage(AttributeTargets.Class, Inherited = false, AllowMultiple = false)]
    internal sealed class StatementPatternAttribute : Attribute
    {
        private ArgumentType[] argumentPattern;

        public StatementPatternAttribute(params ArgumentType[] argumentPattern)
        {
            this.argumentPattern = argumentPattern;
        }

        public bool Match(IEnumerable<ArgumentType> suppliedArgumentTypes)
        {
            return argumentPattern.SequenceEqual(suppliedArgumentTypes);
        }
    }

    internal abstract class Statement
    {
        public FilePosition Position { get; private set; }

        protected Statement(ParseContext context)
        {
            this.Position = context.Position;
        }

        // 派生クラスでは、ファクトリメソッドを返す以下の静的メソッドを定義します。
        // public static Func<IReadOnlyList<string>, ParseContext, Statement> GetFactory()

        public void Evaluate(EvaluationContext context)
        {
            EvaluateCore(context);
        }

        // 文を評価し、EvaluationContext に領域か定数を追加します。
        protected abstract void EvaluateCore(EvaluationContext context);
    }

    [StatementPattern()]
    internal class BlankStatement : Statement
    {
        private BlankStatement(IReadOnlyList<string> arguments, ParseContext context)
            : base(context)
        {
        }

        public static Func<IReadOnlyList<string>, ParseContext, Statement> GetFactory()
        {
            return (IReadOnlyList<string> a, ParseContext c) => new BlankStatement(a, c);
        }

        protected override void EvaluateCore(EvaluationContext context)
        {
            // なにもしません。
        }
    }

    [StatementPattern(ArgumentType.Include, ArgumentType.StringLiteral)]
    internal class IncludeStatement : Statement
    {
        public string FileName { get; private set; }

        private IncludeStatement(IReadOnlyList<string> arguments, ParseContext context)
            : base(context)
        {
            this.FileName = ArgumentUtil.ParseAsStringLiteral(arguments[1]);
        }

        public static Func<IReadOnlyList<string>, ParseContext, Statement> GetFactory()
        {
            return (IReadOnlyList<string> a, ParseContext c) => new IncludeStatement(a, c);
        }

        protected override void EvaluateCore(EvaluationContext context)
        {
            // なにもしません。
            // パース時に特別処理を行い、指定したファイルの内容を取り込みます。
        }
    }

    [StatementPattern(ArgumentType.Identifier, ArgumentType.ConstantSpecifier)]
    internal class ConstantStatement : Statement
    {
        private string constantName;
        private AddrValue value;

        private ConstantStatement(IReadOnlyList<string> arguments, ParseContext context)
            : base(context)
        {
            this.constantName   = ArgumentUtil.ParseAsIdentifier(arguments[0]);
            this.value          = ArgumentUtil.ParseConstantSpecifier(arguments[1], context.AddrWidth);
        }

        public static Func<IReadOnlyList<string>, ParseContext, Statement> GetFactory()
        {
            return (IReadOnlyList<string> a, ParseContext c) => new ConstantStatement(a, c);
        }

        protected override void EvaluateCore(EvaluationContext context)
        {
            context.AddAddrConstant(
                AddrUtil.CreateAddrConstant(constantName, value));
        }
    }

    [StatementPattern(ArgumentType.Identifier, ArgumentType.BeginSpecifier, ArgumentType.SizeSpecifier)]
    internal class AbsoluteBeginSizeStatement : Statement
    {
        private string regionName;
        private AddrValue begin;
        private AddrValue size;

        private AbsoluteBeginSizeStatement(IReadOnlyList<string> arguments, ParseContext context)
            : base(context)
        {
            this.regionName = ArgumentUtil.ParseAsIdentifier(arguments[0]);
            this.begin      = ArgumentUtil.ParseBeginSpecifier(arguments[1], context.AddrWidth);
            this.size       = ArgumentUtil.ParseSizeSpecifier(arguments[2], context.AddrWidth);
        }

        public static Func<IReadOnlyList<string>, ParseContext, Statement> GetFactory()
        {
            return (IReadOnlyList<string> a, ParseContext c) => new AbsoluteBeginSizeStatement(a, c);
        }

        protected override void EvaluateCore(EvaluationContext context)
        {
            context.AddAddrRegion(
                AddrUtil.CreateRegionWithBeginSize(regionName, begin, size));
        }
    }

    [StatementPattern(ArgumentType.Identifier, ArgumentType.BeginSpecifier, ArgumentType.EndSpecifier)]
    internal class AbsoluteBeginEndStatement : Statement
    {
        private string regionName;
        private AddrValue begin;
        private AddrValue end;

        private AbsoluteBeginEndStatement(IReadOnlyList<string> arguments, ParseContext context)
            : base(context)
        {
            this.regionName = ArgumentUtil.ParseAsIdentifier(arguments[0]);
            this.begin      = ArgumentUtil.ParseBeginSpecifier(arguments[1], context.AddrWidth);
            this.end        = ArgumentUtil.ParseEndSpecifier(arguments[2], context.AddrWidth);
        }

        public static Func<IReadOnlyList<string>, ParseContext, Statement> GetFactory()
        {
            return (IReadOnlyList<string> a, ParseContext c) => new AbsoluteBeginEndStatement(a, c);
        }

        protected override void EvaluateCore(EvaluationContext context)
        {
            context.AddAddrRegion(
                AddrUtil.CreateRegionWithBeginEnd(this.regionName, this.begin, this.end));
        }
    }

    [StatementPattern(ArgumentType.Identifier, ArgumentType.Identifier, ArgumentType.InnerOuter, ArgumentType.OffsetSpecifier, ArgumentType.SizeSpecifier)]
    internal class RelativeInnerOuterStatement : Statement
    {
        private string regionName;
        private string baseRegionName;
        private InnerOuter innerOuter;
        private Offset offset;
        private AddrValue size;

        private RelativeInnerOuterStatement(IReadOnlyList<string> arguments, ParseContext context)
            : base(context)
        {
            this.regionName     = ArgumentUtil.ParseAsIdentifier(arguments[0]);
            this.baseRegionName = ArgumentUtil.ParseAsIdentifier(arguments[1]);
            this.innerOuter     = ArgumentUtil.ParseInnerOuter(arguments[2]);
            this.offset         = ArgumentUtil.ParseOffsetSpecifier(arguments[3], context.AddrWidth);
            this.size           = ArgumentUtil.ParseSizeSpecifier(arguments[4], context.AddrWidth);
        }

        public static Func<IReadOnlyList<string>, ParseContext, Statement> GetFactory()
        {
            return (IReadOnlyList<string> a, ParseContext c) => new RelativeInnerOuterStatement(a, c);
        }

        protected override void EvaluateCore(EvaluationContext context)
        {
            var baseRegion = context.GetAddrRegion(baseRegionName);
            context.AddAddrRegion(
                AddrUtil.CreateRegionWithInnerOuter(regionName, baseRegion, innerOuter, offset, size));
        }
    }

    [StatementPattern(ArgumentType.Identifier, ArgumentType.Identifier, ArgumentType.Shift, ArgumentType.OffsetSpecifier)]
    internal class RelativeShiftStatement : Statement
    {
        private string regionName;
        private string baseRegionName;
        private Offset offset;

        private RelativeShiftStatement(IReadOnlyList<string> arguments, ParseContext context)
            : base(context)
        {
            this.regionName     = ArgumentUtil.ParseAsIdentifier(arguments[0]);
            this.baseRegionName = ArgumentUtil.ParseAsIdentifier(arguments[1]);
            // shift
            this.offset         = ArgumentUtil.ParseOffsetSpecifier(arguments[3], context.AddrWidth);
        }

        public static Func<IReadOnlyList<string>, ParseContext, Statement> GetFactory()
        {
            return (IReadOnlyList<string> a, ParseContext c) => new RelativeShiftStatement(a, c);
        }

        protected override void EvaluateCore(EvaluationContext context)
        {
            var baseRegion = context.GetAddrRegion(baseRegionName);
            context.AddAddrRegion(
                AddrUtil.CreateRegionWithShift(regionName, baseRegion, offset));
        }
    }

    [StatementPattern(ArgumentType.Identifier, ArgumentType.Alias, ArgumentType.Identifier)]
    internal class AliasRegionStatement : Statement
    {
        private string regionName;
        private string baseRegionName;

        private AliasRegionStatement(IReadOnlyList<string> arguments, ParseContext context)
            : base(context)
        {
            this.regionName     = ArgumentUtil.ParseAsIdentifier(arguments[0]);
            // alias
            this.baseRegionName = ArgumentUtil.ParseAsIdentifier(arguments[2]);
        }

        public static Func<IReadOnlyList<string>, ParseContext, Statement> GetFactory()
        {
            return (IReadOnlyList<string> a, ParseContext c) => new AliasRegionStatement(a, c);
        }

        protected override void EvaluateCore(EvaluationContext context)
        {
            var baseRegion = context.GetAddrRegion(baseRegionName);
            context.AddAddrRegion(
                AddrUtil.CreateRegionWithBeginEnd(regionName, baseRegion.Begin, baseRegion.End));
        }
    }

    [StatementPattern(ArgumentType.Identifier, ArgumentType.Alias, ArgumentType.Identifier, ArgumentType.RegionValueType)]
    internal class AliasConstantStatement : Statement
    {
        private string constantName;
        private string baseRegionName;
        private RegionValueType regionValueType;

        private AliasConstantStatement(IReadOnlyList<string> arguments, ParseContext context)
            : base(context)
        {
            this.constantName       = ArgumentUtil.ParseAsIdentifier(arguments[0]);
            // alias
            this.baseRegionName     = ArgumentUtil.ParseAsIdentifier(arguments[2]);
            this.regionValueType    = ArgumentUtil.ParseRegionValueType(arguments[3]);
        }

        public static Func<IReadOnlyList<string>, ParseContext, Statement> GetFactory()
        {
            return (IReadOnlyList<string> a, ParseContext c) => new AliasConstantStatement(a, c);
        }

        protected override void EvaluateCore(EvaluationContext context)
        {
            var baseRegion = context.GetAddrRegion(baseRegionName);
            context.AddAddrConstant(
                AddrUtil.CreateAddrConstant(constantName, baseRegion.GetValue(regionValueType)));
        }
    }
}
