﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;

namespace HtcDaemon.Usb
{
    public class UsbLowStream : Stream
    {
        // ターゲット側の USB 受信バッファサイズ以下にすること
        private const int TransferCountMax = 64 * 1024;

        private ByteBlockQueue receiveBuffer = new ByteBlockQueue();
        private UsbDevice device;
        private bool disposed = false;

        public UsbLowStream(UsbDevice device)
        {
            this.device = device;
        }

        public override bool CanRead { get { return true; } }
        public override bool CanSeek { get { return false; } }
        public override bool CanWrite { get { return true; } }

        public override long Length { get { throw new NotSupportedException(); } }

        public override long Position
        {
            get { throw new NotSupportedException(); }
            set { throw new NotSupportedException(); }
        }

        public override void Flush()
        {
            // 送信をバッファリングしていないので、何もしない。
        }

        public override int Read(byte[] buffer, int offset, int count)
        {
            if (this.receiveBuffer.Length == 0)
            {
                int packetSize = ReadPacketSize();
                var packetBuffer = ReadPacketBody(packetSize);
                this.receiveBuffer.Enqueue(packetBuffer, 0, packetSize);
            }

            return this.receiveBuffer.Dequeue(buffer, offset, count);
        }

        private byte[] ReadPacketBody(int packetSize)
        {
            byte[] packetBuffer = new byte[packetSize];
            int totalBytesRead = 0;
            while (totalBytesRead < packetSize)
            {
                int bytesRead = (int)this.device.Read(packetBuffer, totalBytesRead, packetSize - totalBytesRead);
                totalBytesRead += bytesRead;
            }
            return packetBuffer;
        }

        private int ReadPacketSize()
        {
            byte[] packetBuffer = new byte[4]; // サイズは4バイト
            while (true)
            {
                int bytesRead = (int)this.device.Read(packetBuffer, 0, 4);
                if (bytesRead != 0)
                {
                    return BitConverter.ToInt32(packetBuffer, 0);
                }
                // 0 パケットならループ
            }
        }

        public override void Write(byte[] buffer, int offset, int count)
        {
            int bytesWritten = 0;
            while (bytesWritten < count)
            {
                int chunkBytes = Math.Min(count - bytesWritten, TransferCountMax);
                WritePacket(buffer, offset + bytesWritten, chunkBytes);
                bytesWritten += chunkBytes;
            }
        }

        private void WritePacket(byte[] buffer, int offset, int count)
        {
            // 0 パケット以外は、パケットサイズを送る
            if (count != 0)
            {
                WritePacketSize(count);
            }

            int bytesWritten = this.device.Write(buffer, offset, count);
            Debug.Assert(bytesWritten == count, "failed assertion: bytesWritten == count");
        }

        private void WritePacketSize(int size)
        {
            byte[] packetBuffer = BitConverter.GetBytes(size);
            int bytesWritten = this.device.Write(packetBuffer, 0, 4); // サイズは4バイト
            Debug.Assert(bytesWritten == 4, "failed assertion: cannot write packet size");
        }

        public override long Seek(long offset, SeekOrigin origin)
        {
            throw new NotSupportedException();
        }

        public override void SetLength(long value)
        {
            throw new NotSupportedException();
        }

        public void Cancel()
        {
            try
            {
                this.device.Abort();
            }
            catch (OperationCanceledException)
            {
                // 既に切断されている
            }
        }

        protected override void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    this.Cancel();
                    this.device.Dispose();
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }

            base.Dispose(disposing);
        }
    }
}
