﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace HtcDaemon
{
    internal class TcpServer : IDisposable
    {
        private bool disposed;
        private CancellationTokenSource cancellationTokenSource = new CancellationTokenSource();
        private Thread listenThread;
        private Socket listenSocket;

        public event EventHandler<TcpConnectedEventArgs> Connected;
        public IPEndPoint EndPoint { get { return (IPEndPoint)listenSocket.LocalEndPoint; } }

        public TcpServer(int port)
        {
            this.listenThread = new Thread(ListenThread) { Name = "TargetListener" };
            this.listenSocket = MakeListenSocket(port);
        }

        ~TcpServer()
        {
            this.Dispose(false);
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    this.cancellationTokenSource.Cancel();
                    this.listenSocket.Close();
                    this.listenThread.Join();
                    this.cancellationTokenSource.Dispose();
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }

        private Socket MakeListenSocket(int port)
        {
            var socket = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);
            socket.Bind(new IPEndPoint(IPAddress.Any, port));
            socket.Listen(1);
            return socket;
        }

        public void Start()
        {
            listenThread.Start();
        }

        public void Stop()
        {
            cancellationTokenSource.Cancel();
        }

        private void ListenThread()
        {
            while (!cancellationTokenSource.IsCancellationRequested)
            {
                if (listenSocket.Poll(1000 * 1000, SelectMode.SelectRead))
                {
                    var socket = listenSocket.Accept();
                    RaiseConnected(socket);
                }
            }
        }

        private void RaiseConnected(Socket socket)
        {
            if (Connected != null)
            {
                Connected(this, new TcpConnectedEventArgs(socket));
            }
        }
    }
}
