﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Htcs;

namespace HtcDaemon
{
    internal class SerializedExecutor : IDisposable
    {
        private bool disposed;
        private CancellationTokenSource cancellationTokenSource = new CancellationTokenSource();
        private ConcurrentQueue<Action> actionQueue = new ConcurrentQueue<Action>();
        private AutoResetEvent executeEvent = new AutoResetEvent(false);
        private Thread executeThread;

        public SerializedExecutor()
        {
            this.executeThread = new Thread(ExecuteThreadFunction) { Name = "SerializedExecutor.ExecuteThread" };
        }

        ~SerializedExecutor()
        {
            this.Dispose(false);
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    cancellationTokenSource.Cancel();
                    executeEvent.Set();

                    executeThread.Join();

                    executeEvent.Dispose();
                    cancellationTokenSource.Dispose();
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }

        public void Start()
        {
            executeThread.Start();
        }

        public void Stop()
        {
            Dispose();
        }

        public void ExecuteAsync(Action action)
        {
            actionQueue.Enqueue(action);
            executeEvent.Set();
        }

        private void ExecuteThreadFunction()
        {
            var token = cancellationTokenSource.Token;

            while (!(token.IsCancellationRequested && actionQueue.IsEmpty))
            {
                executeEvent.WaitOne();

                Action action;
                while (actionQueue.TryDequeue(out action))
                {
                    action();
                }
            }
        }
    }
}
