﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using System.Net.Sockets;
using System.Text;
using System.Threading;

namespace HtcDaemon.Builtin
{
    // ※ホストがサーバーの場合
    internal abstract class BuiltInServiceCommunicatorBase : IDisposable
    {
        protected NetworkStream Stream { get; private set; }
        private Thread thread;
        private bool disposed;

        public event EventHandler Disconnected;

        public BuiltInServiceCommunicatorBase(Socket socket_)
        {
            this.Stream = new NetworkStream(socket_, true);
        }

        ~BuiltInServiceCommunicatorBase()
        {
            this.Dispose(false);
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    this.Cleanup();
                    this.thread.Join();
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }

        public void Start()
        {
            this.thread = new Thread(this.RunWork);
            this.thread.Start();
        }

        private void RunWork()
        {
            try
            {
                this.Run();
            }
            finally
            {
                this.Cleanup();
                this.RaiseDisconnected();
            }
        }

        public void Stop()
        {
            this.Dispose();
        }

        private void Cleanup()
        {
            Stream.Close();
        }

        protected void RaiseDisconnected()
        {
            if (this.Disconnected != null)
            {
                this.Disconnected(this, EventArgs.Empty);
            }
        }

        protected abstract void Run();
    }
}
