﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Net;
using System.Net.Sockets;
using System.IO;
using System.Threading;
using Nintendo.Htcs;

namespace HtcDaemon.Builtin
{
    internal abstract class BuiltInServiceBase : IDisposable
    {
        private HtcsCommunicator slc = new HtcsCommunicator();
        private List<BuiltInServiceCommunicatorBase> communicators = new List<BuiltInServiceCommunicatorBase>();
        private TcpServer listener = new TcpServer(0);
        private bool disposed;

        public BuiltInServiceBase()
        {
            listener.Connected += ClientConnected;
        }

        ~BuiltInServiceBase()
        {
            this.Dispose(false);
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    this.listener.Dispose();
                    this.slc.Dispose();

                    BuiltInServiceCommunicatorBase[] communicatorListSnapShot;
                    lock (communicators)
                    {
                        communicatorListSnapShot = communicators.ToArray();
                    }

                    foreach (var communicator in communicatorListSnapShot)
                    {
                        communicator.Dispose();
                    }
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }

        private void PrintPortMap(object sender, HtcsInfoUpdatedEventArgs e)
        {
            Console.WriteLine("--- PortMap ---");
            foreach (PortMapItem pmi in e.PortMap)
            {
                Console.WriteLine(pmi.ToString());
            }
            Console.WriteLine("---------------");
        }

        public void Start()
        {
            this.slc.HtcsInfoUpdated += PrintPortMap;
            this.listener.Start();
            this.slc.Start();
            var descriptor = new PortMapItem(MakeServiceId(), listener.EndPoint);
            var result = this.slc.RegisterPort(descriptor);

            // TODO エラーハンドリング
            System.Diagnostics.Debug.Assert(result == 0, "RegisterPort が失敗しました。");
        }

        public void Stop()
        {
            this.slc.UnregisterPort(MakeServiceId());
            this.Dispose();
        }

        private HtcsPortDescriptor MakeServiceId()
        {
            return new HtcsPortDescriptor(HtcsPeerName.Any, PortName);
        }

        private void ClientConnected(object sender, TcpConnectedEventArgs e)
        {
            CreateAndRegisterCommunicator(e.Socket);
        }

        private void CreateAndRegisterCommunicator(Socket socket)
        {
            var communicator = CreateCommunicator(socket);
            communicator.Disconnected += OnAppDisconnected;
            lock (communicators)
            {
                communicators.Add(communicator);
            }
            communicator.Start();
        }

        private void OnAppDisconnected(object sender, EventArgs e)
        {
            lock (communicators)
            {
                communicators.Remove((BuiltInServiceCommunicatorBase)sender);
            }
        }

        // 派生クラスで、実際のインスタンスを生成するようにオーバーライドする
        protected abstract BuiltInServiceCommunicatorBase CreateCommunicator(Socket socket);
        protected abstract string PortName { get; }
    }
}
