﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using HtcDaemon.Bridging;

namespace HtcDaemon
{
    internal class BridgeManager : IDisposable
    {
        private bool disposed;
        private Bridge[] bridges = new Bridge[HtcConstants.HtcChannelCountMax];

        public event EventHandler<ChannelEventArgs> BridgeDisconnected;
        public event EventHandler<ChannelEventArgs> HostAppDisconnected;

        ~BridgeManager()
        {
            this.Dispose(false);
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    foreach (var bridge in bridges.Where(b => b != null))
                    {
                        bridge.Dispose();
                    }
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }

            //// 基底クラスのDisposeを呼び出す
            // base.Dispose(disposing);
        }
        public BridgeManager()
        {
        }

        public void CreateBridge(IChannel channel, Socket socket)
        {
            Debug.Assert(bridges[channel.ChannelId] == null, "channelId のブリッジはすでに生成されています。");

            var bridge = new Bridge(channel, socket);
            lock (bridges)
            {
                bridges[channel.ChannelId] = bridge;
            }
            bridge.TargetDisconnected += OnBridgeDisconnected;
            bridge.HostAppDisconnected += OnHostAppDisconnected;
            bridge.Start();
        }

        public void DestroyBridge(int channelId)
        {
            Debug.Assert(bridges[channelId] != null, "channelId のブリッジは生成されていません。");

            Bridge bridge;
            lock (bridges)
            {
                bridge = bridges[channelId];
                bridges[channelId] = null;
            }
            bridge.TargetDisconnected -= OnBridgeDisconnected;
            bridge.HostAppDisconnected -= OnHostAppDisconnected;
            bridge.Dispose();
        }

        private void OnBridgeDisconnected(object sender, ChannelEventArgs e)
        {
            // Destroy は上位に任せる
            // TODO: 同時に Destroy されているかもしれない。非同期 Destroy リクエストを行い、Disconnected イベントで破棄完了とする。
            RaiseBridgeDisconnected(e.Channel);
        }

        private void OnHostAppDisconnected(object sender, ChannelEventArgs e)
        {
            // 上位で チャンネルの Close 要求を発行する
            RaiseHostAppDisconnected(e.Channel);
        }

        protected void RaiseBridgeDisconnected(IChannel channel)
        {
            if (BridgeDisconnected != null)
            {
                BridgeDisconnected(this, new ChannelEventArgs(channel));
            }
        }

        protected void RaiseHostAppDisconnected(IChannel channel)
        {
            if (HostAppDisconnected != null)
            {
                HostAppDisconnected(this, new ChannelEventArgs(channel));
            }
        }
    }
}
