﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.FsFileCacheSimulator.FileSystem;
using Nintendo.FsFileCacheSimulator.Simulation;

namespace Nintendo.FsFileCacheSimulator
{
    internal class Program
    {
        private static void Main(string[] args)
        {
            var option = ParseCommandLineOptionOrDie(args);

            Log.Default.MinimumLoggingVerbosity = option.Verbosity;

            var accessLog = GetFileAccessLog(option);

            var simulationResults = RunSimulations(option, accessLog);

            ReportResults(option, accessLog, simulationResults);

            if (option.SaveSimulatedLog)
            {
                foreach (var simulationResult in simulationResults)
                {
                    var logPath = GetSimulatedAccessLogPath(option, simulationResult.label);

                    SaveSimulatedAccessLog(logPath, simulationResult.result);
                }
            }
        }

        private static CommandLineOption ParseCommandLineOptionOrDie(string[] args)
        {
            var option = default(CommandLineOption);
            try
            {
                option = CommandLineOption.Parse(args);

                if (option == null)
                {
                    // ヘルプメッセージを表示したので終了
                    Environment.Exit(0);
                }
            }
            catch (ArgumentException)
            {
                // コマンドラインオプションのパースに失敗したので終了
                Environment.Exit(1);
            }

            try
            {
                CommandLineOption.CheckCommandLineOptionOrThrow(option);
            }
            catch (ArgumentException ex)
            {
                Log.Default.ErrorLine(ex.Message);
                Environment.Exit(1);
            }

            return option;
        }

        private static AccessLog GetFileAccessLog(CommandLineOption option)
        {
            using (var reader = new StreamReader(option.AccessLog))
            {
                return new AccessLog(reader);
            }
        }

        private static IEnumerable<(string label, SimulationResult result)> RunSimulations(CommandLineOption option, AccessLog accessLog)
        {
            var offsetTable = default(IReadOnlyDictionary<string, FileRegion>);
            if (!string.IsNullOrEmpty(option.InputNspPath))
            {
                offsetTable = FileSystem.Utility.GetRomFsOffsetTable(option.InputNspPath, option.OriginalNspPath, option.KeyConfigPath);
            }

            var storageTypeGetter = CreateStorageTypeGetter(option);

            return new(string, SimulationResult)[]
            {
                ("NoCache", AccessSimulator.DoAccessSimulation(accessLog, storageTypeGetter, (mountName, fileSystemType) =>
                {
                    return new Simulation.Algorithm.NoCache();
                },
                option.SplitResultPerMilliseconds)),

                ("LRU", AccessSimulator.DoAccessSimulation(accessLog, storageTypeGetter, (mountName, fileSystemType) =>
                {
                    if (offsetTable != null && fileSystemType == FileSystemType.Rom)
                    {
                        return new Simulation.Algorithm.StorageOffsetBasedLeastRecentlyUsedCache(
                            option.CacheSize, option.PageSize, offsetTable, option.EnableReadAhead);
                    }
                    else
                    {
                        return new Simulation.Algorithm.FileOffsetBasedLeastRecentlyUsedCache(
                            option.CacheSize, option.PageSize, option.EnableReadAhead);
                    }
                },
                option.SplitResultPerMilliseconds)),
            };
        }

        private static Func<string, FileSystemType, StorageType> CreateStorageTypeGetter(CommandLineOption option)
        {
            return (mountName, fileSystemType) =>
            {
                if (fileSystemType == FileSystemType.SaveData)
                {
                    // セーブデータの保存先は NAND だけだったはず
                    return StorageType.Nand;
                }
                return option.StorageType;  // TODO: FileSystemType ごとに変えられるように
            };
        }

        private static void ReportResults(CommandLineOption option, AccessLog accessLog, IEnumerable<(string label, SimulationResult result)> simulationResults)
        {
            // キャッシュアルゴリズムごとにグループ化されているので、結果表示のためにマウント名ごとにグループ化する
            var fileSystemResultsByMountName = new Dictionary<string, List<(string label, FileSystemSimulationResult[] results)>>();
            foreach (var simulationResult in simulationResults)
            {
                var groups = simulationResult.result.FileSystemResults.GroupBy(x => x.MountName);
                foreach (var group in groups)
                {
                    var mountName = group.Key;
                    if (!fileSystemResultsByMountName.ContainsKey(mountName))
                    {
                        fileSystemResultsByMountName.Add(mountName, new List<(string label, FileSystemSimulationResult[] results)>());
                    }
                    fileSystemResultsByMountName[mountName].Add((simulationResult.label, group.ToArray()));
                }
            }
            // グループ内について、さらに同一タイミングにおける全キャッシュアルゴリズムの結果をまとめる
            var labelAndResultPairsGroupedByTiming = new Dictionary<string, List<Dictionary<string, FileSystemSimulationResult>>>();
            foreach (var mountName in fileSystemResultsByMountName.Keys)
            {
                var identicalTimingResultList = new List<Dictionary<string, FileSystemSimulationResult>>();
                labelAndResultPairsGroupedByTiming.Add(mountName, identicalTimingResultList);

                var groups = fileSystemResultsByMountName[mountName];
                var length = groups.First().results.Length;
                if (!groups.All(x => x.results.Length == length))
                {
                    throw new ImplementationErrorException();
                }

                for (var i = 0; i < length; i++)
                {
                    var identicalTimingResult = groups.ToDictionary(x => x.label, x => x.results[i]);

                    // ShowResultOfIpcGreaterThan よりも大きい IPC 回数を含む結果がある時のみ表示
                    if (identicalTimingResult.Any(
                        kv => kv.Value.FileSystemStatistics.ReadIpcCount > option.ShowResultOfIpcGreaterThan
                        || kv.Value.FileSystemStatistics.WriteIpcCount > option.ShowResultOfIpcGreaterThan))
                    {
                        identicalTimingResultList.Add(identicalTimingResult);
                    }
                }
            }

            // 結果表示
            foreach (var mountName in labelAndResultPairsGroupedByTiming.Keys)
            {
                var labelAndResultPairs = labelAndResultPairsGroupedByTiming[mountName];
                if (labelAndResultPairs.Count == 0)
                {
                    // フィルタリングされて結果がなくなってしまった
                    continue;
                }

                var fileSystemType = labelAndResultPairs[0].First().Value.FileSystemType;

                Log.Default.MessageLine($"====== mount name: {mountName} ({fileSystemType.ToString()}) ======");
                Log.Default.MessageLine();

                foreach (var identicalTimingResults in labelAndResultPairs)
                {
                    var noCacheResult = identicalTimingResults["NoCache"];

                    var noCacheLogBasedReadTime = noCacheResult.LogBasedReadTime;
                    var noCacheLogBasedWriteTime = noCacheResult.LogBasedWriteTime;
                    var noCacheIdealReadTime = noCacheResult.FileSystemStatistics.EstimatedReadTime;
                    var noCacheIdealWriteTime = noCacheResult.FileSystemStatistics.EstimatedWriteTime;

                    var startTime = TimeSpan.FromMilliseconds(noCacheResult.StartTimeMilliseconds - accessLog.StartTimeMillisecond);
                    var endTime = TimeSpan.FromMilliseconds(noCacheResult.EndTimeMilliseconds - accessLog.StartTimeMillisecond);

                    Log.Default.MessageLine($"+-- split {startTime} - {endTime}");
                    Log.Default.MessageLine($"|");

                    foreach (var label in identicalTimingResults.Keys)
                    {
                        var result = identicalTimingResults[label];

                        if (result.StartTimeMilliseconds != noCacheResult.StartTimeMilliseconds
                            || result.EndTimeMilliseconds != noCacheResult.EndTimeMilliseconds)
                        {
                            throw new ImplementationErrorException();
                        }

                        Log.Default.MessageLine($"| {label}");
                        Log.Default.MessageLine($"|   read");
                        Log.Default.MessageLine($"|     IPC count            = {result.FileSystemStatistics.ReadIpcCount}");
                        Log.Default.MessageLine($"|     I/O size             = {CreateHumanReadableByteSizeString(result.FileSystemStatistics.ReadBytes)}");
                        Log.Default.MessageLine($"|     ideal time           = {result.FileSystemStatistics.EstimatedReadTime}s.");
                        if (label == "NoCache")
                        {
                            Log.Default.MessageLine($"|     log-based time       = {result.LogBasedReadTime}s.");
                        }
                        else
                        {
                            Log.Default.MessageLine($"|     cache overhead       = {result.CacheStatistics.ReadCacheOverhead}s.");
                            if (noCacheIdealReadTime != TimeSpan.Zero)
                            {
                                var ticks = result.FileSystemStatistics.EstimatedReadTime.Ticks * noCacheLogBasedReadTime.Ticks / noCacheIdealReadTime.Ticks;
                                Log.Default.MessageLine($"|     estimated time       = {TimeSpan.FromTicks(ticks) + result.CacheStatistics.ReadCacheOverhead}s.");
                            }
                            else
                            {
                                Log.Default.MessageLine($"|     estimated time       = N/A");
                            }
                        }
                        Log.Default.MessageLine($"|   write");
                        Log.Default.MessageLine($"|     IPC count            = {result.FileSystemStatistics.WriteIpcCount}");
                        Log.Default.MessageLine($"|     I/O size             = {CreateHumanReadableByteSizeString(result.FileSystemStatistics.WrittenBytes)}");
                        Log.Default.MessageLine($"|     ideal time           = {result.FileSystemStatistics.EstimatedWriteTime}s.");
                        if (label == "NoCache")
                        {
                            Log.Default.MessageLine($"|     log-based time       = {result.LogBasedWriteTime}s.");
                        }
                        else
                        {
                            Log.Default.MessageLine($"|     cache overhead       = {result.CacheStatistics.WriteCacheOverhead}s.");
                            if (noCacheIdealWriteTime != TimeSpan.Zero)
                            {
                                var ticks = result.FileSystemStatistics.EstimatedWriteTime.Ticks * noCacheLogBasedWriteTime.Ticks / noCacheIdealWriteTime.Ticks;
                                Log.Default.MessageLine($"|     estimated time       = {TimeSpan.FromTicks(ticks) + result.CacheStatistics.WriteCacheOverhead}s.");
                            }
                            else
                            {
                                Log.Default.MessageLine($"|     estimated time       = N/A");
                            }
                        }
                        Log.Default.MessageLine($"|   cache hit ratio        = {result.CacheStatistics.CacheHitRatio:F3} ({result.CacheStatistics.CacheHitCount}/{result.CacheStatistics.CacheSearchCount})");
                        Log.Default.MessageLine($"|   readahead hit ratio    = {result.CacheStatistics.ReadAheadHitRatio:F3} ({result.CacheStatistics.ReadAheadHitCount}/{result.CacheStatistics.ReadAheadCount})");
                        Log.Default.MessageLine($"|");
                    }
                    Log.Default.MessageLine($"+--");
                    Log.Default.MessageLine();
                }
            }
        }

        private static string CreateHumanReadableByteSizeString(ulong size)
        {
            if (size < (1 << 10))
            {
                return $"{size} bytes";
            }
            else if (size < (1 << 20))
            {
                return $"{size / 1024.0:F2} KiB";
            }
            else if (size < (1 << 30))
            {
                return $"{size / 1024.0 / 1024.0:F3} MiB";
            }
            else
            {
                return $"{size / 1024.0 / 1024.0 / 1024.0:F3} GiB";
            }
        }

        private static string GetSimulatedAccessLogPath(CommandLineOption option, string label)
        {
            return Path.Combine(
                Path.GetDirectoryName(option.AccessLog),
                $"{Path.GetFileNameWithoutExtension(option.AccessLog)}-{label}.txt");
        }

        private static void SaveSimulatedAccessLog(string path, SimulationResult simulationResult)
        {
            var serializableLogs = simulationResult.SimulatedAccessLog.ToSerializableLogs();
            using (var writer = new StreamWriter(path, false, new UTF8Encoding(encoderShouldEmitUTF8Identifier: true)))
            {
                AccessLogSerializer.Serialize(writer, serializableLogs);
            }
        }
    }
}
