﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using DeviceTreeDataCollector;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;

namespace MakePartialDesc
{
    public class MakePartialDesc
    {
        /// <summary>
        /// MakePartialDesc モードのメイン処理
        /// パラメータを元に指定されたパスに partial desc ファイルを生成する
        /// </summary>
        /// <param name="inputParam">モード引数セット</param>
        /// <param name="collectedPropertyList">device tree から収集したプロパティ情報</param>
        /// <param name="outputFilePath">出力先ファイル</param>
        public void Process(InputParameter inputParam, IEnumerable<CollectedProperty> collectedPropertyList, string outputFilePath)
        {
            string memoryMapDescTexts = MakeMemoryMapDescTexts(collectedPropertyList, inputParam.MemoryMapPageAlignment);
            string enableInterruptDescTexts = MakeInterruptDescTexts(collectedPropertyList);
            GeneratePartialDescFile(memoryMapDescTexts, enableInterruptDescTexts, inputParam.TemplateFilePath, outputFilePath);
        }

        /// <summary>
        /// partial desc ファイルを生成する
        /// </summary>
        /// <param name="memoryMapDescTexts"></param>
        /// <param name="enableInterruptDescTexts"></param>
        /// <param name="templateFilePath"></param>
        /// <param name="outputFilePath"></param>
        private static void GeneratePartialDescFile(
            string memoryMapDescTexts,
            string enableInterruptDescTexts,
            string templateFilePath,
            string outputFilePath)
        {
            using (var sr = new StreamReader(templateFilePath, Encoding.UTF8, true))
            using (var sw = new StreamWriter(outputFilePath, false, Encoding.UTF8))
            {
                var outputXml = sr.ReadToEnd();
                outputXml = outputXml.Replace("__MEMORY_MAP_LIST__", memoryMapDescTexts)
                                .Replace("__ENABLED_INTERRUPT_LIST__", enableInterruptDescTexts);
                sw.Write(outputXml);
            }
        }

        /// <summary>
        /// 収集した device tree 情報から partial desc のメモリマップ部分の文字列を生成する
        /// </summary>
        /// <param name="collectedPropertyList"></param>
        /// <param name="memoryMapPageAlignment"></param>
        /// <returns></returns>
        private string MakeMemoryMapDescTexts(IEnumerable<CollectedProperty> collectedPropertyList, int memoryMapPageAlignment)
        {
            string resultTexts = string.Empty;
            var listInInterest = collectedPropertyList.Where(cp => cp.CollectParameter.Id == "MemoryMap");
            if (listInInterest.Any())
            {
                List<AddressRange> memoryMaps = new List<AddressRange>();
                foreach (var cp in listInInterest)
                {
                    var baseMemoryMaps = cp.Property.GetValues<AddressRange>();

                    if (!baseMemoryMaps.Any())
                    {
                        continue;
                    }

                    if (cp.CollectParameter.Selector.TargetIndex.Any())
                    {
                        // Selector が指定されている場合、指定されたインデックスのものだけを取り出す
                        foreach (var tidx in cp.CollectParameter.Selector.TargetIndex)
                        {
                            if (0 <= tidx && tidx <= baseMemoryMaps.Count())
                            {
                                memoryMaps.Add(baseMemoryMaps.ElementAt(tidx));
                            }
                            else
                            {
                                Console.Error.WriteLine("WARN: Attempted out-of-range access to the memory map list obtained from the device tree.\n");
                                Console.Error.WriteLine("Either specification of following in input yml might be wrong:");
                                Console.Error.WriteLine($"  - PropertyCollectParameters.Selector.TargetIndex {tidx}");
                            }
                        }
                    }
                    else
                    {
                        // Selector が指定されていなければ、全部対象にする
                        memoryMaps.AddRange(baseMemoryMaps.ToList());
                    }
                }

                var sanitizedMemoryMaps = SanitizeMemoryMaps(AlignMemoryMaps(memoryMaps, memoryMapPageAlignment));
                if (sanitizedMemoryMaps.Any())
                {
                    resultTexts += MakeMemoryMapsXmlLines(sanitizedMemoryMaps) + "\r\n";
                }
            }
            return resultTexts;
        }

        /// <summary>
        /// メモリマップ列の各要素を、指定したアライメントに沿うように拡張する（開始アドレスは切り捨て、終了アドレスは切り上げ）
        /// </summary>
        /// <param name="memoryMaps"></param>
        /// <param name="memoryMapPageAlignment"></param>
        /// <returns></returns>
        private static List<AddressRange> AlignMemoryMaps(IEnumerable<AddressRange> memoryMaps, int memoryMapPageAlignment)
        {
            var alignedMemoryMaps = new List<AddressRange>();
            foreach (var memoryMap in memoryMaps)
            {
                var alignedBeginAddress = (memoryMap.BeginAddress / (ulong)memoryMapPageAlignment) * (ulong)memoryMapPageAlignment;
                var alignedEndAddress = ((memoryMap.BeginAddress + memoryMap.Size + (ulong)memoryMapPageAlignment - 1) / (ulong)memoryMapPageAlignment) * (ulong)memoryMapPageAlignment;
                alignedMemoryMaps.Add(new AddressRange(alignedBeginAddress, alignedEndAddress - alignedBeginAddress));
            }
            return alignedMemoryMaps;
        }

        /// <summary>
        /// メモリマップ列をアドレスでソートし、重複・包含関係にあるものを削除する
        /// また、連続していて結合可能な要素があれば結合する
        /// </summary>
        /// <param name="memoryMaps"></param>
        /// <returns></returns>
        public static List<AddressRange> SanitizeMemoryMaps(IEnumerable<AddressRange> memoryMaps)
        {
            var sanitizedMemoryMaps = new List<AddressRange>(memoryMaps);

            sanitizedMemoryMaps.Sort();
            sanitizedMemoryMaps = sanitizedMemoryMaps.Distinct().ToList(); // まず完全一致する要素を削除

            var removeList = new HashSet<AddressRange>(); // 削除対象の重複指定を自動的に排除するため HashSet を使用
            for (var i = 0; i < sanitizedMemoryMaps.Count(); ++i)
            {
                var mi = sanitizedMemoryMaps.ElementAt(i);
                for (var j = 0; j < sanitizedMemoryMaps.Count(); ++j)
                {
                    if (i == j) // 自分とは比較しない
                    {
                        continue;
                    }

                    var mj = sanitizedMemoryMaps.ElementAt(j);
                    if (mi.Includes(mj))
                    {
                        // 被包含関係にある要素は削除対象に
                        removeList.Add(mj);
                    }
                }
            }

            // 被包含要素を削除
            foreach (var rm in removeList)
            {
                sanitizedMemoryMaps.Remove(rm);
            }

            // 一致 or 包含関係にある要素がなくなったので、この時点の sanitizedMemoryMaps には開始アドレスが同じ要素は存在せず、開始アドレスでソートされている

            // 隣り合っている要素で結合可能なものを結合する
            bool continueLoop;
            do
            {
                continueLoop = false;
                for (var i = 0; i < sanitizedMemoryMaps.Count() - 1; ++i)
                {
                    var m = sanitizedMemoryMaps.ElementAt(i);
                    var mn = sanitizedMemoryMaps.ElementAt(i + 1);

                    // m.BeginAddress < mn.BeginAddress と  m.BeginAddress + m.Size < mn.BeginAddress + mn.Size が保証されているはず
                    if (mn.BeginAddress <= m.BeginAddress + m.Size) // 次の要素の開始アドレスと前の要素の終了アドレスから接点判定
                    {
                        // 要素をまとめる
                        var combined = new AddressRange(m.BeginAddress, (mn.BeginAddress - m.BeginAddress) + mn.Size);
                        sanitizedMemoryMaps.Remove(m);
                        sanitizedMemoryMaps.Remove(mn);
                        sanitizedMemoryMaps.Insert(i, combined);
                        sanitizedMemoryMaps.Sort(); // 念のため

                        // ループの最初に戻してやり直す
                        continueLoop = true;
                        break;
                    }
                }
            } while (continueLoop);

            // sanitizedMemoryMaps.ForEach(x => Console.Error.WriteLine($"sanitizedMemoryMaps: {x.ToString()}"));
            return sanitizedMemoryMaps;
        }

        /// <summary>
        /// 渡された AddressRange の列から partial desc に埋め込むメモリマップの文字列を生成する
        /// permission と type は現時点では決め打ち
        /// </summary>
        /// <param name="memoryMaps"></param>
        /// <returns></returns>
        private static string MakeMemoryMapsXmlLines(IEnumerable<AddressRange> memoryMaps)
        {
            var xmlLines = new List<string>();
            foreach (var memoryMap in memoryMaps)
            {
                var beginAddress = memoryMap.BeginAddress;
                var size = memoryMap.Size;
                xmlLines.Add("            <MemoryMap>");
                xmlLines.Add($"                <BeginAddress>0x{beginAddress:x9}</BeginAddress>");
                xmlLines.Add($"                <Size>0x{size:x}</Size>");
                xmlLines.Add("                <Permission>RW</Permission>");
                xmlLines.Add("                <Type>Io</Type>");
                xmlLines.Add("            </MemoryMap>");
            }
            return string.Join("\r\n", xmlLines.ToArray());
        }

        /// <summary>
        /// 収集した device tree 情報から partial desc の割り込み部分の文字列を生成する
        /// </summary>
        /// <param name="collectedPropertyList"></param>
        /// <returns></returns>
        private static string MakeInterruptDescTexts(IEnumerable<CollectedProperty> collectedPropertyList)
        {
            string resultTexts = string.Empty;
            var listInInterest = collectedPropertyList.Where(cp => cp.CollectParameter.Id == "Interrupts");
            if (listInInterest.Any())
            {
                foreach (var cp in listInInterest)
                {
                    var selector = cp.CollectParameter.Selector;
                    var modifier = cp.CollectParameter.Modifier;
                    var interrupts = MakeInterruptNumberList(cp.Property.GetValues<int>(), selector.TargetIndex, selector.UnitSize, selector.TargetCellInUnit, modifier.IntegerValueOffset);
                    if (interrupts.Any())
                    {
                        resultTexts += MakeEnableInterruptsXmlLines(interrupts) + "\r\n";
                    }
                }
            }
            return resultTexts;
        }

        /// <summary>
        /// 収集した割り込み番号の DeviceTreeProperty を解析し、partial desc に埋め込む割り込み番号のリストを生成する
        /// </summary>
        /// <param name="interruptProperty"></param>
        /// <param name="targetIndex"></param>
        /// <param name="unitSize"></param>
        /// <param name="targetCellInUnit"></param>
        /// <param name="interruptNumberOffset"></param>
        /// <returns></returns>
        private static List<int> MakeInterruptNumberList(IEnumerable<int> interruptProperty, IEnumerable<int> targetIndex, int unitSize, int targetCellInUnit, int interruptNumberOffset)
        {
            var interruptNumberList = new List<int>();
            // Console.WriteLine($"targetCellInUnit:{targetCellInUnit}, unitSize:{unitSize}");

            if (targetIndex.Any())
            {
                // targetIndex が指定されている場合、指定されたインデックスのものだけを取り出す
                foreach (var tidx in targetIndex)
                {
                    int index = tidx * unitSize + targetCellInUnit;
                    if (0 <= index && index <= interruptProperty.Count())
                    {
                        // 値に interruptNumberOffset を加算しつつリストに追加
                        interruptNumberList.Add(interruptProperty.ElementAt(index) + interruptNumberOffset);
                    }
                    else
                    {
                        Console.Error.WriteLine("WARN: Attempted out-of-range access to the interrupt list obtained from the device tree.\n");
                        Console.Error.WriteLine("Either specification of following in input yml might be wrong:");
                        Console.Error.WriteLine($"  - PropertyCollectParameters.Selector.TargetIndex {tidx}");
                        Console.Error.WriteLine($"  - PropertyCollectParameters.Selector.UnitSize {unitSize}");
                        Console.Error.WriteLine($"  - PropertyCollectParameters.Selector.TargetCellOffsetInUnit {targetCellInUnit}");
                    }
                }
            }
            else
            {
                // targetIndex が指定されていなければ、全セル対象
                for (int index = targetCellInUnit; index < interruptProperty.Count(); index += unitSize)
                {
                    // 値に interruptNumberOffset を加算しつつリストに追加
                    interruptNumberList.Add(interruptProperty.ElementAt(index) + interruptNumberOffset);
                }
            }

            return interruptNumberList;
        }

        /// <summary>
        /// 割り込み番号のリストから partial desc に埋め込む割り込み番号の文字列を生成する
        /// </summary>
        /// <param name="interrupts"></param>
        /// <returns></returns>
        private static string MakeEnableInterruptsXmlLines(IEnumerable<int> interrupts)
        {
            var xmlLines = new List<string>();
            foreach (var interrupt in interrupts)
            {
                xmlLines.Add($"            <EnableInterrupts>{interrupt}</EnableInterrupts>");
            }
            return string.Join("\r\n", xmlLines.ToArray());
        }
    }
}
