﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;
using System.IO;

namespace DeviceTreeDataCollector.DeviceTree
{
    public class DeviceTreeNode
    {
        private readonly byte[] Fdt;
        private readonly string Path;
        private readonly int Offset;
        private readonly int AddressCellsInBytes; // "#address-cells" プロパティの値 * sizeof(Int32)
        private readonly int SizeCellsInBytes; // "#size-cells" プロパティの値 * sizeof(Int32)

        public DeviceTreeNode(byte[] fdt, string nodePath)
        {
            Fdt = fdt;
            Path = nodePath;
            Offset = LibFdt.HandleError(LibFdt.FdtPathOffset(fdt, nodePath));

            if (!nodePath.Equals("/"))
            {
                // 一番近い親の #address-cells と #size-cells の指定は、 reg などアドレス範囲を示す特定のフィールドで使用する
                var parent = LibFdt.HandleError(LibFdt.FdtParentOffset(Fdt, Offset));
                AddressCellsInBytes = LibFdt.FdtAddressCells(Fdt, parent) * sizeof(int);
                SizeCellsInBytes = LibFdt.FdtSizeCells(Fdt, parent) * sizeof(int);
            }
            else
            {
                // ルートノードの親ノードは存在しないので、デフォルト値を入れておく
                // まともな device tree ならばルートノードでこの値が参照されることはない
                AddressCellsInBytes = 2;
                SizeCellsInBytes = 2;
            }
        }

        // TODO: サブノードのイテレータを返す機能

        public bool ContainsProperty(string propertyName)
        {
            int lenp;
            var propertyOffset = LibFdt.FdtGetprop(Fdt, Offset, propertyName, out lenp);
            return (propertyOffset > 0);
        }

        public DeviceTreeProperty GetValue(string propertyName, DeviceTreeProperty.PropertyType propertyType)
        {
            var property = new DeviceTreeProperty(propertyName, propertyType);

            int propertyBytesLength;
            var propertyOffset = LibFdt.HandleError(LibFdt.FdtGetprop(Fdt, Offset, propertyName, out propertyBytesLength));
            byte[] propertyBytes = (new ArraySegment<byte>(Fdt, propertyOffset, propertyBytesLength)).ToArray();

            var bitConverter = new BitConverterWithEndian(BitConverterWithEndian.Endian.Big);
            switch (propertyType)
            {
                case DeviceTreeProperty.PropertyType.AddressRangeList:
                    foreach (int i in Enumerable.Range(0, propertyBytesLength / (AddressCellsInBytes + SizeCellsInBytes)))
                    {
                        property.Add(
                            new AddressRange(
                                bitConverter.ToUInt64(propertyBytes, i * (AddressCellsInBytes + SizeCellsInBytes), AddressCellsInBytes),
                                bitConverter.ToUInt64(propertyBytes, i * (AddressCellsInBytes + SizeCellsInBytes) + AddressCellsInBytes, SizeCellsInBytes)));
                    }
                    break;
                case DeviceTreeProperty.PropertyType.IntegerList:
                    foreach (int i in Enumerable.Range(0, propertyBytesLength / sizeof(int)))
                    {
                        property.Add(bitConverter.ToInt32(propertyBytes, i * sizeof(int)));
                    }
                    break;
                case DeviceTreeProperty.PropertyType.StringList:
                    string text = System.Text.Encoding.ASCII.GetString(propertyBytes);
                    property.Add(text);
                    break;
                case DeviceTreeProperty.PropertyType.Phandle:
                    foreach (int i in Enumerable.Range(0, propertyBytesLength / sizeof(uint)))
                    {
                        property.Add(new Phandle(bitConverter.ToUInt32(propertyBytes, i * sizeof(uint))));
                    }
                    break;
                default:
                    throw new ArgumentException();
            }
            return property;
        }
    }

    public class DeviceTreeReader
    {
        // 扱えるファイルの最大サイズ
        private readonly int MaxSupportedFileSize = 2 * 1024 * 1024;

        public readonly string FilePath;
        private readonly byte[] Fdt;

        public DeviceTreeReader(string filePath)
        {
            this.FilePath = filePath;

            using (var stream = new FileStream(filePath, FileMode.Open, FileAccess.Read))
            using (var reader = new BinaryReader(stream))
            {
                // libfdt に渡すために、dtb ファイルの内容をすべて byte の配列に載せる
                this.Fdt = reader.ReadBytes(MaxSupportedFileSize);

                // ファイル形式の簡易チェック
                if (LibFdt.FdtCheckHeader(Fdt) < 0)
                {
                    throw new InvalidDataException($"Check header failed. `{FilePath}` is not in valid dtb (fdt) format.");
                }
            }
        }

        public DeviceTreeNode FindNode(string nodePath)
        {
            return new DeviceTreeNode(Fdt, nodePath);
        }
    }
}
