﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Json;
using System.IO;

namespace MakeDataDocument
{
    public static class DataletUtility
    {
        public static IEnumerable<string> EnumerateDataFiles(string rootPath)
        {
            foreach (var filename in Directory.GetFiles(rootPath, "*.data.json", SearchOption.AllDirectories))
            {
                yield return filename;
            }
        }

        public static List<Dictionary<string, object>> MakeDataletContainerFromJsonText(string jsonText)
        {
            var settings = new DataContractJsonSerializerSettings();
            settings.UseSimpleDictionaryFormat = true;

            var serializer = new DataContractJsonSerializer(typeof(List<Dictionary<string, object>>), settings);
            var stream = new MemoryStream(Encoding.UTF8.GetBytes(jsonText));

            return (List<Dictionary<string, object>>)serializer.ReadObject(stream);
        }

        public static Dictionary<string, object> MakeDataletMetaInformationsFromJsonText(string jsonText)
        {
            var settings = new DataContractJsonSerializerSettings();
            settings.UseSimpleDictionaryFormat = true;

            var serializer = new DataContractJsonSerializer(typeof(Dictionary<string, object>), settings);
            var stream = new MemoryStream(Encoding.UTF8.GetBytes(jsonText));

            return (Dictionary<string, object>)serializer.ReadObject(stream);
        }

        public static string GetMetaFileName(string filename)
        {
            if (filename.EndsWith(".data.json"))
            {
                return filename.Replace(".data.json", ".meta.json");
            }
            else
            {
                throw new InvalidDataException(string.Format("サポートしていないファイル名です。\nFileName=\n{0}", filename));
            }
        }

        public static Dictionary<string, object> GetMetaInformation(string filename)
        {
            var metaFileName = GetMetaFileName(filename);

            if (File.Exists(metaFileName))
            {
                using (var reader = new StreamReader(metaFileName))
                {
                    return MakeDataletMetaInformationsFromJsonText(reader.ReadToEnd());
                }
            }
            else
            {
                return new Dictionary<string, object>();
            }
        }

        public static List<Dictionary<string, object>> MakeDataletContainerFromJsonFile(string path)
        {
            using (var reader = new StreamReader(path))
            {
                return MakeDataletContainerFromJsonText(reader.ReadToEnd());
            }
        }

        public static Dictionary<string, object> MergeDataletAndMeta(Dictionary<string, object> datalet, Dictionary<string, object> metaInformations)
        {
            var ret = new Dictionary<string, object>(datalet);

            foreach (var pair in metaInformations)
            {
                if (ret.ContainsKey(pair.Key))
                {
                    throw new InvalidDataException(string.Format(
                        "メタ情報と同じ値が格納されています。\nKey = {0}\nValue = {1}",
                        pair.Key,
                        ret[pair.Key]));
                }

                ret[pair.Key] = pair.Value;
            }

            return ret;
        }

        public static List<Dictionary<string, object>> MergeDataletContainerAndMeta(List<Dictionary<string, object>> dataletContainer, Dictionary<string, object> metaInformations)
        {
            return (
                from datalet in dataletContainer
                select MergeDataletAndMeta(datalet, metaInformations)).ToList();
        }

        public static IEnumerable<Dictionary<string, object>> EnumerateDataletsFromFiles(IEnumerable<string> filenames)
        {
            foreach (var filename in filenames)
            {
                var metaInformations = GetMetaInformation(filename);
                var dataletContainer = MakeDataletContainerFromJsonFile(filename);
                var mergedDatalets = MergeDataletContainerAndMeta(dataletContainer, metaInformations);

                foreach (var datalet in mergedDatalets)
                {
                    yield return datalet;
                }
            }
        }
    }
}
