﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Collections.Generic;

namespace ContentsUploader.Commands
{
    /// <summary>
    // コマンドの終了処理、終了コードを管理します。
    /// </summary>
    public class CommandStatus : System.IDisposable
    {
        public string Name { get; }
        public Stopwatch Stopwatch { get; }
        public bool IsMeasured { get; }
        public int ExitCode { get; set; }
        public bool Result { set { ExitCode = value ? 0 : 1; } }
        public bool IsSuccess { get { return ExitCode == 0; } }
        public bool IsFailure { get { return !IsSuccess; } }
        private List<System.Action> Disposer { get; }

        /// <summary>
        /// クラスのインスタンスを初期化します。
        /// </summary>
        /// <param name="commandName">コマンド名</param>
        /// <param name="exitCode">終了コードの初期値</param>
        public CommandStatus(string commandName, bool isMeasured = false, int exitCode = 1)
        {
            Name = commandName;
            Stopwatch = Stopwatch.StartNew();
            IsMeasured = isMeasured;
            ExitCode = exitCode;
            Disposer = new List<System.Action>();
        }

        /// <summary>
        /// リソースを解放します。
        /// </summary>
        public void Dispose()
        {
            if (Disposer.Count > 0)
            {
                foreach (var action in Disposer)
                {
                    DoAction(action);
                }
                Disposer.Clear();
            }
            WriteResult();
        }

        /// <summary>
        /// 後始末処理を追加します。Dispose メソッドが呼ばれた際に実行されます。
        /// </summary>
        /// <param name="action">後始末処理</param>
        public void AddDisposer(System.Action action)
        {
            Disposer.Add(action);
        }

        /// <summary>
        /// 後始末処理を実行します。例外を無視します。
        /// </summary>
        /// <param name="action">後始末処理</param>
        private void DoAction(System.Action action)
        {
            try
            {
                action();
            }
            catch (System.Exception e)
            {
                Log.WriteLine($"Exception : {e.Message}");
            }
        }

        /// <summary>
        /// コマンド結果を出力します。ExitCode が 0 以外なら、プロセスの終了コードに 1 (失敗) を設定します。
        /// </summary>
        private void WriteResult()
        {
            if (IsSuccess)
            {
                Log.WriteLine($"{Name} Done.");
            }
            else
            {
                Log.WriteLine($"{Name} Failure.");
                System.Environment.ExitCode = 1;
            }
            if (IsMeasured)
            {
                Log.WriteLineAsIs($"Elapsed Time: {Stopwatch.Elapsed}");
            }
        }
    }
}
