﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;

namespace ContentsUploader.Assistants
{
    // Rom コレクション
    public class RomCollection
    {
        public List<Id64> Roms { get; private set; }
        public List<Id64> Apps { get; private set; }
        public List<Id64> Aocs { get; private set; }
        public List<Id64> Patchs { get; private set; }

        public RomCollection()
        {
            Roms = new List<Id64>();
            Apps = new List<Id64>();
            Aocs = new List<Id64>();
            Patchs = new List<Id64>();
        }

        public bool Add(Id64 contentMetaId, ContentMetaType contentMetaType)
        {
            Roms.Add(contentMetaId);

            var added = true;
            switch (contentMetaType)
            {
            case ContentMetaType.Application:
                Apps.Add(contentMetaId);
                break;
            case ContentMetaType.AddOnContent:
                Aocs.Add(contentMetaId);
                break;
            case ContentMetaType.Patch:
            case ContentMetaType.Delta:
                Patchs.Add(contentMetaId);
                break;
            default:
                added = false;
                break;
            }
            return added;
        }

        public bool Add(NspAccessor.Content content)
        {
            return Add(content.ContentMetaId, content.Type);
        }

        public void Add(RomCollection collection)
        {
            Roms.AddRange(collection.Roms);
            Apps.AddRange(collection.Apps);
            Aocs.AddRange(collection.Aocs);
            Patchs.AddRange(collection.Patchs);
        }

        public bool Distinct()
        {
            Roms = Roms.Distinct().ToList();
            Apps = Apps.Distinct().ToList();
            Aocs = Aocs.Distinct().ToList();
            Patchs = Patchs.Distinct().ToList();

            //Roms = Roms.Distinct().OrderBy(x => x).ToList();
            //Apps = Apps.Distinct().OrderBy(x => x).ToList();
            //Aocs = Aocs.Distinct().OrderBy(x => x).ToList();
            //Patchs = Patchs.Distinct().OrderBy(x => x).ToList();

            // 総数に整合性があるなら真を返す
            var count = Apps.Count + Aocs.Count + Patchs.Count;
            return Roms.Count == count;
        }

        public bool IsEmpty()
        {
            return (Roms.Count == 0) && (Apps.Count == 0) && (Aocs.Count == 0) && (Patchs.Count == 0);
        }

        public bool IsAppContained()
        {
            return Apps.Count > 0;
        }

        public bool IsAppOnly()
        {
            return Apps.Count > 0 && Apps.Count == Roms.Count;
        }

        public bool IsAocOnly()
        {
            return Aocs.Count > 0 && Aocs.Count == Roms.Count;
        }

        public bool IsPatchOnly()
        {
            return Patchs.Count > 0 && Patchs.Count == Roms.Count;
        }

        public List<Id64> ToAppsAndAocsList()
        {
            return Apps.Union(Aocs).ToList();
        }
    }

    // Rom 登録結果
    public class RegisterRomResult : RomCollection
    {
        public Id64 ApplicationId { get; private set; }

        public RegisterRomResult() : this(Id64.Invalid) { }

        public RegisterRomResult(Id64 applicationId)
        {
            ApplicationId = applicationId;
        }

        public bool Validate()
        {
            return !IsEmpty() && Distinct() && ApplicationId.IsValid;
        }

        public bool Merge(RegisterRomResult result)
        {
            if (!ApplicationId.IsValid)
            {
                // 無効値ならマージ先の値を設定
                ApplicationId = result.ApplicationId;
            }
            else if (ApplicationId != result.ApplicationId)
            {
                // 有効値で異なる場合ならマージ不可
                return false;
            }
            Add(result);
            return true;
        }
    }

    // NsUid マップ
    public class NsUidMap : Dictionary<Id64, string>
    {
        public NsUidMap() : base() { }
        public NsUidMap(NsUidMap oher) : base(oher) { }

        public List<string> ToNsUids()
        {
            var keys = new List<Id64>(Keys);
            keys.Sort();

            var values = new List<string>();
            foreach (var key in keys)
            {
                values.Add(this[key]);
            }
            return values;
        }

        public Id64 FindKey(long nsUid)
        {
            return FindKey(nsUid.ToString());
        }

        public Id64 FindKey(string nsUid)
        {
            foreach (var pair in this)
            {
                if (pair.Value == nsUid)
                {
                    return pair.Key;
                }
            }
            return Id64.Invalid;
        }
    }
}
