﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Threading.Tasks;
using System.Security.Cryptography;
using CommandUtility;

namespace CommandUtility
{
    public class CryptUtility
    {
        public static byte[] EncryptedKey(byte[] targetKey, byte[] kek)
        {
            return XorBytes(targetKey, kek);
        }

        public static byte[] GenerateKey()
        {
            var rng = new RNGCryptoServiceProvider();
            var ret = new byte[16];
            rng.GetBytes(ret);

            return ret;
        }

        public static byte[] GenerateIV()
        {
            return GenerateKey();
        }

        public static byte[] EncryptAes128(byte[] input, byte[] key, byte[] iv)
        {
            var aes = Aes.Create();
            aes.BlockSize = 128;
            aes.KeySize = 128;
            aes.Mode = CipherMode.CBC;
            aes.Padding = PaddingMode.None;
            aes.Key = key;
            aes.IV = iv;

            using (var rawWriter = new MemoryStream())
            using (var cryptWriter = new CryptoStream(rawWriter, aes.CreateEncryptor(), CryptoStreamMode.Write))
            {
                cryptWriter.Write(input, 0, input.Length);
                cryptWriter.FlushFinalBlock();

                return rawWriter.ToArray();
            }
        }

        public static void EncryptAes128(Stream output, Stream input, byte[] key, byte[] iv)
        {
            var aes = Aes.Create();
            aes.BlockSize = 128;
            aes.KeySize = 128;
            aes.Mode = CipherMode.CBC;
            aes.Padding = PaddingMode.None;
            aes.Key = key;
            aes.IV = iv;

            using (var cryptWriter = new CryptoStream(output, aes.CreateEncryptor(), CryptoStreamMode.Write))
            {
                input.CopyTo(cryptWriter);
            }
        }

        public static void DecryptAes128(Stream output, Stream input, byte[] key, byte[] iv)
        {
            var aes = Aes.Create();
            aes.BlockSize = 128;
            aes.KeySize = 128;
            aes.Mode = CipherMode.CBC;
            aes.Padding = PaddingMode.None;
            aes.Key = key;
            aes.IV = iv;

            using (var cryptWriter = new CryptoStream(output, aes.CreateDecryptor(), CryptoStreamMode.Write))
            {
                input.CopyTo(cryptWriter);
            }
        }

        public static byte[] ShiftLeftOneBit(byte[] source)
        {
            var newBytes = new byte[source.Length];
            UInt16 carryBit = 0;

            for (int i = source.Length - 1; i >= 0; i--)
            {
                UInt16 shifted = (UInt16)(source[i] << 1);
                newBytes[i] = (byte)(shifted & 0xff | carryBit);
                carryBit = (UInt16)((shifted & 0x100) >> 8);
            }

            return newBytes;
        }

        public static byte[] XorBytes(byte[] a, byte[] b)
        {
            if (a.Length != b.Length)
            {
                throw new Exception("invalid byte array length");
            }

            var newBytes = new byte[a.Length];

            for (int i = 0; i < newBytes.Length; i++)
            {
                newBytes[i] = (byte)(a[i] ^ b[i]);
            }

            return newBytes;
        }

        public static byte[] XorBytes(byte[] a, byte[] b, int offsetB)
        {
            var newBytes = new byte[a.Length];
            a.CopyTo(newBytes, 0);

            for (int i = offsetB; i < offsetB + b.Length; i++)
            {
                newBytes[i] = (byte)(a[i] ^ b[i - offsetB]);
            }

            return newBytes;
        }

        public static IEnumerable<byte[]> SplitBlocks(byte[] data)
        {
            for (int i = 0; i < (int)((data.Length + 15) / 16); i++)
            {
                var part = new byte[16];
                Array.Copy(data, i * 16, part, 0, 16);
                yield return part;
            }
        }

        public static byte[] DecryptAesEcb128(byte[] input, byte[] key)
        {
            var aes = Aes.Create();
            aes.BlockSize = 128;
            aes.KeySize = 128;
            aes.Mode = CipherMode.ECB;
            aes.Padding = PaddingMode.None;
            aes.Key = key;

            using (var rawWriter = new MemoryStream())
            using (var cryptWriter = new CryptoStream(rawWriter, aes.CreateDecryptor(), CryptoStreamMode.Write))
            {
                cryptWriter.Write(input, 0, input.Length);
                cryptWriter.FlushFinalBlock();

                return rawWriter.ToArray();
            }
        }

        public static byte[] EncryptAesEcb128(byte[] input, byte[] key)
        {
            var aes = Aes.Create();
            aes.BlockSize = 128;
            aes.KeySize = 128;
            aes.Mode = CipherMode.ECB;
            aes.Padding = PaddingMode.None;
            aes.Key = key;

            using (var rawWriter = new MemoryStream())
            using (var cryptWriter = new CryptoStream(rawWriter, aes.CreateEncryptor(), CryptoStreamMode.Write))
            {
                cryptWriter.Write(input, 0, input.Length);
                cryptWriter.FlushFinalBlock();

                return rawWriter.ToArray();
            }
        }
    }

    public class CmacUtility
    {
        public static readonly byte[] Zero = BinaryUtility.FromHexString("00000000000000000000000000000000");
        public static readonly byte[] Rb =   BinaryUtility.FromHexString("00000000000000000000000000000087");

        public static byte[] CalculateCmac(byte[] inputData, byte[] key)
        {
            var k1 = CalculateK1(key);
            var k2 = CalculateK2(key);

            int paddingSize = CalculatePaddingSize(inputData);

            if (paddingSize == 0)
            {
                inputData = CryptUtility.XorBytes(inputData, k1, inputData.Length - 16);
            }
            else
            {
                var padding = new byte[paddingSize];
                padding[0] = 0x80;

                inputData = inputData.Concat(padding).ToArray();

                inputData = CryptUtility.XorBytes(inputData, k2, inputData.Length - 16);
            }

            byte[] result = CryptUtility.EncryptAes128(inputData, key, Zero);

            byte[] cmac = new byte[16];
            Array.Copy(result, result.Length - 16, cmac, 0, 16);

            return cmac;
        }

        private static int CalculatePaddingSize(byte[] inputData)
        {
            var len = inputData.Length;

            if (len == 0)
            {
                return 16;
            }
            else if ((len % 16) == 0)
            {
                return 0;
            }
            else
            {
                return 16 - (len % 16);
            }
        }

        public static byte[] CalculateK1(byte[] key)
        {
            var L = CryptUtility.EncryptAes128(Zero, key, Zero);

            if ((L[0] & 0x80) == 0)
            {
                return CryptUtility.ShiftLeftOneBit(L);
            }
            else
            {
                return CryptUtility.XorBytes(CryptUtility.ShiftLeftOneBit(L), Rb);
            }
        }

        public static byte[] CalculateK2(byte[] key)
        {
            var k1 = CalculateK1(key);

            if ((k1[0] & 0x80) == 0)
            {
                return CryptUtility.ShiftLeftOneBit(k1);
            }
            else
            {
                return CryptUtility.XorBytes(CryptUtility.ShiftLeftOneBit(k1), Rb);
            }
        }

        public static void Verify(byte[] input, byte[] expectedCmac, byte[] cmacKey)
        {
            var resultCmac = CalculateCmac(input, cmacKey);
            if (! Enumerable.SequenceEqual(resultCmac, expectedCmac))
            {
                throw new InvalidDataException("unmatch cmac");
            }
        }
    }
}
