﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace ElfLibrary
{
    using System;
    using System.IO;

    /// <summary>
    /// セクションの種類
    /// </summary>
    public enum ElfSectionType : ulong
    {
        NULL = 0,
        PROGBITS = 1,
        SYMTAB = 2,
        STRTAB = 3,
        RELA = 4,
        HASH = 5,
        DYNAMIC = 6,
        NOTE = 7,
        NOBITS = 8,
        REL = 9,
        SHLIB = 10,
        DYNSYM = 11,
        LOPROC = 0x70000000,
        HIPROC = 0x7fffffff,
        LOUSER = 0x80000000,
        HIUSER = 0xffffffff,
    }

    /// <summary>
    /// セクションの属性
    /// </summary>
    public enum ElfSectionAttributeFlags : ulong
    {
        NULL = 0x0,
        WRITE = 0x1,
        ALLOC = 0x2,
        EXECINSTR = 0x4,
        MASKPROC = 0xf0000000,
    }

    /// <summary>
    /// 予約されているセクションのインデックス番号
    /// </summary>
    public enum ElfSpecialSectionIndex
    {
        Undef = 0x0000,
        LoReserve = 0xff00,
        LoProc = 0xff00,
        HiProc = 0xff1f,
        Abs = 0xfff1,
        Common = 0xfff2,
        HiReserve = 0xffff,
    }

    /// <summary>
    /// ELF セクションヘッダクラス
    /// </summary>
    public abstract class ElfSectionHeader : IElfFile
    {
        public uint Name { get; set; }
        public uint Type { get; set; }
        public ulong Flags { get; set; }
        public ulong Address { get; set; }
        public ulong Offset { get; set; }
        public ulong Size { get; set; }
        public uint Link { get; set; }
        public uint Info { get; set; }
        public ulong AddressAlign { get; set; }
        public ulong EntrySize { get; set; }

        /// <summary>
        /// ELF セクションの位置を計算。
        /// </summary>
        /// <returns>セクションの終了地点のオフセット</returns>
        public ulong SetPosition(ulong offset)
        {
            if (!(1 == AddressAlign || 0 == AddressAlign % 2))
            {
                throw new InvalidOperationException();
            }
            Offset = NumericConvert.RoundUp(offset, AddressAlign);
            return Offset + Size;
        }

        /// <summary>
        /// ファイルから SectionHeader を読み込みます。
        /// </summary>
        /// <param name="reader">読み込みインスタンス</param>
        public abstract void ReadElfFile(ElfBinaryReader reader);

        /// <summary>
        /// ファイルに SectionHeader を書き込みます。
        /// </summary>
        /// <param name="writer">書き込みインスタンス</param>
        public abstract void WriteElfData(ElfBinaryWriter writer);
    }
}
