﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NOMINMAX

#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_NcaFileSystemDriver.h>
#include <nn/fssystem/fs_NcaFileSystemDriverUtility.h>
#include <nn/fssystem/utilTool/fs_NcaSparseStorage.h>
#include "BufferManager.h"
#include "DefaultMemoryResource.h"
#include "ManagedStreamStorage.h"
#include "SparseStorageStream.h"
#include "../Util/DeclareAlive.h"

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::IO;
using namespace System::Runtime::InteropServices;

private class SparseStorageStreamImpl
{
private:
    enum class SparseMode
    {
        None,
        Basic,
        Empty
    };

    static SparseMode GetSparseMode(NintendoContentArchiveReader^ targetReader, NintendoContentArchiveReader^ patchReader, int fsIndex)
    {
        switch( targetReader->GetContentType() )
        {
        case (byte)NcaHeader::ContentType::Program:
            switch( fsIndex )
            {
            case 0: // コード
                return SparseMode::Empty;

            case 1: // データ
                return SparseMode::Basic;

            default:
                break;
            }
            break;

        case (byte)NcaHeader::ContentType::Manual:
            switch( fsIndex )
            {
            case 0:
                if( targetReader->HasFsInfo(0) )
                {
                    if( patchReader->HasFsInfo(0) )
                    {
                        return SparseMode::Basic;
                    }
                    else
                    {
                        // 元ロム側をすべて参照しない
                        return SparseMode::Empty;
                    }
                }
                break;

            default:
                break;
            }
            break;

        default:
            break;
        }

        return SparseMode::None;
    }

public:
    SparseStorageStreamImpl(Int32 blockSize, Int32 eraseSize)
        : m_Allocator()
        , m_Builder(blockSize, eraseSize)
    {
    }

    void Build(NcaFsHeaderReader* pOutHeaderReader, NintendoContentArchiveReader^ targetReader, NintendoContentArchiveReader^ patchReader, Int32 fsIndex, Int64 physicalOffset)
    {
        const auto mode = GetSparseMode(targetReader, patchReader, fsIndex);
        if( mode != SparseMode::None )
        {
            NcaFileSystemDriver driver(targetReader->GetImpl(), patchReader->GetImpl(), &m_Allocator, GetBufferManager());
            NcaFileSystemDriver::SparseParam param(&m_Builder, pOutHeaderReader, (mode == SparseMode::Empty), fsIndex, physicalOffset);
            param.SetSparseStorage(targetReader->GetExternalSparseStorage());

            auto result = driver.SparsifyStorage(&param);
            if( result.IsFailure() )
            {
                throw gcnew System::Exception(
                    String::Format("Failed to SparsifyStorage NcaFileSystemDriver({0}) 0x{1:X8}.", fsIndex, result.GetInnerValueForDebug()));
            }

            // 外部データを付与
            if( pOutHeaderReader->ExistsSparseLayer() )
            {
                auto pSparseStorage = targetReader->GetExternalSparseStorage();
                if( pSparseStorage == nullptr )
                {
                    pSparseStorage = AllocateShared<utilTool::NcaSparseStorage>(*targetReader->GetImpl());
                    targetReader->SetExternalSparseStorage(pSparseStorage);
                }

                result = pSparseStorage->SetPartition(&m_Allocator, &m_Builder, *pOutHeaderReader);
                if( result.IsFailure() )
                {
                    throw gcnew System::Exception(
                        String::Format("Failed to SetPartition NcaSparseStorage({0}) 0x{1:X8}.", fsIndex, result.GetInnerValueForDebug()));
                }
            }
        }
    }

    void WriteTable(array<Byte>^ tableBuffer)
    {
        const auto tableSize = static_cast<int>(GetTableSize());
        if( tableBuffer->GetLength(0) < tableSize )
        {
            throw gcnew System::ArgumentException();
        }

        char headerBuffer[nn::fssystem::NcaBucketInfo::HeaderSize];
        fs::MemoryStorage headerStorage(headerBuffer, sizeof(headerBuffer));
        pin_ptr<unsigned char> tablePtr = &tableBuffer[0];
        fs::MemoryStorage tableStorage(tablePtr, tableSize);

        const int64_t nodeOffset = 0;
        const auto nodeSize = m_Builder.QueryTableNodeStorageSize();
        const auto entryOffset = nodeOffset + nodeSize;
        const auto entrySize = m_Builder.QueryTableEntryStorageSize();

        auto result = m_Builder.WriteTable(
            &m_Allocator,
            fs::SubStorage(&headerStorage, 0, nn::fssystem::NcaBucketInfo::HeaderSize),
            fs::SubStorage(&tableStorage, nodeOffset, nodeSize),
            fs::SubStorage(&tableStorage, entryOffset, entrySize)
        );

        tablePtr = nullptr;

        if( result.IsFailure() )
        {
            throw gcnew System::Exception(
                String::Format("Failed to WriteTable SparseStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
        }
    }

    array<Byte>^ ReadData(Int64 offset, Int32 size)
    {
        array<Byte>^ buffer = gcnew array<Byte>(size);
        pin_ptr<unsigned char> ptr = &buffer[0];

        auto result = m_Builder.ReadData(offset, ptr, size);
        ptr = nullptr;

        if( result.IsFailure() )
        {
            throw gcnew System::Exception(
                String::Format("Failed to ReadData SparseStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
        }

        return buffer;
    }

    void OutputBuildLog(Stream^ tableStream)
    {
        auto nodeStream = gcnew MemoryStream();
        auto entryStream = gcnew MemoryStream();
        {
            ManagedStreamStorage tableStorage(tableStream);
            ManagedStreamStorage nodeStorage(nodeStream);
            ManagedStreamStorage entryStorage(entryStream);

            auto result = m_Builder.OutputBuildLog(&m_Allocator, &tableStorage, &nodeStorage, &entryStorage);
            if( result.IsFailure() )
            {
                throw gcnew System::Exception(
                    String::Format("Failed to OutputBuildLog SparseStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug())
                );
            }
        }

        nodeStream->Position = 0;
        nodeStream->CopyTo(tableStream);
        entryStream->Position = 0;
        entryStream->CopyTo(tableStream);

        delete nodeStream;
        delete entryStream;
    }

    Int64 GetDataSize() const
    {
        return nn::util::align_up(m_Builder.QueryDataStorageSize(), 1 << nn::fssystem::NcaHeader::Log2SectorSize);
    }

    Int32 GetTableSize() const
    {
        return static_cast<int>(m_Builder.QueryTableNodeStorageSize() + m_Builder.QueryTableEntryStorageSize());
    }

private:
    DefaultAlignedMemoryResource m_Allocator;
    nn::fssystem::utilTool::SparseStorageBuilder m_Builder;
};

SparseStorageStream::SparseStorageStream(Int32 blockSize, Int32 eraseSize)
{
    m_pImpl = new SparseStorageStreamImpl(blockSize, eraseSize);
    GC::KeepAlive(this);
}

SparseStorageStream::~SparseStorageStream()
{
    this->!SparseStorageStream();
}

SparseStorageStream::!SparseStorageStream()
{
    delete m_pImpl;
}

NintendoContentArchiveFsHeaderInfo^ SparseStorageStream::Build(NintendoContentArchiveReader^ targetReader, NintendoContentArchiveReader^ patchReader, Int32 fsIndex, Int64 physicalOffset)
{
    auto pHeaderReader = std::unique_ptr<NcaFsHeaderReader>(new NcaFsHeaderReader);

    m_pImpl->Build(pHeaderReader.get(), targetReader, patchReader, fsIndex, physicalOffset);

    return Util::ReturnAndDeclareAlive(this, gcnew NintendoContentArchiveFsHeaderInfo(std::move(pHeaderReader), 0));
}

void SparseStorageStream::WriteTable(array<Byte>^ tableBuffer)
{
    m_pImpl->WriteTable(tableBuffer);
    GC::KeepAlive(this);
}

array<Byte>^ SparseStorageStream::ReadData(Int64 offset, Int32 size)
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->ReadData(offset, size));
}

void SparseStorageStream::OutputBuildLog(Stream^ tableStream)
{
    m_pImpl->OutputBuildLog(tableStream);
    GC::KeepAlive(this);
}

Int64 SparseStorageStream::GetDataSize()
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetDataSize());
}

Int32 SparseStorageStream::GetTableSize()
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetTableSize());
}

private class SparseStorageArchiveStreamImpl
{
public:
    explicit SparseStorageArchiveStreamImpl(std::shared_ptr<utilTool::NcaSparseStorage> pStorage)
        : m_pStorage(std::move(pStorage))
    {
        if( m_pStorage == nullptr )
        {
            throw gcnew System::ArgumentNullException();
        }
    }

    explicit SparseStorageArchiveStreamImpl(array<Byte>^ buffer)
        : m_pStorage(AllocateShared<utilTool::NcaSparseStorage>())
    {
        pin_ptr<unsigned char> ptr = &buffer[0];

        Result result = m_pStorage->Load(ptr, buffer->GetLength(0));
        if( result.IsFailure() )
        {
            throw gcnew System::Exception(
                String::Format("Failed to Load NcaSparseStorage 0x{0:X8}.", result.GetInnerValueForDebug()));
        }
    }

    array<Byte>^ ReadData(Int64 offset, Int32 size)
    {
        array<Byte>^ buffer = gcnew array<Byte>(size);
        pin_ptr<unsigned char> ptr = &buffer[0];

        m_pStorage->Read(offset, ptr, size);

        return buffer;
    }

    Int64 GetDataSize()
    {
        return m_pStorage->GetSize();
    }

    std::shared_ptr<utilTool::NcaSparseStorage> GetImpl()
    {
        return m_pStorage;
    }

private:
    std::shared_ptr<utilTool::NcaSparseStorage> m_pStorage;
};

SparseStorageArchiveStream::SparseStorageArchiveStream(String^ type, NintendoContentArchiveReader^ reader)
{
    Type = type;
    m_pImpl = new SparseStorageArchiveStreamImpl(reader->GetExternalSparseStorage());
    GC::KeepAlive(this);
}

SparseStorageArchiveStream::SparseStorageArchiveStream(String^ type, array<Byte>^ buffer)
{
    Type = type;
    m_pImpl = new SparseStorageArchiveStreamImpl(buffer);
    GC::KeepAlive(this);
}

SparseStorageArchiveStream::~SparseStorageArchiveStream()
{
    this->!SparseStorageArchiveStream();
}

SparseStorageArchiveStream::!SparseStorageArchiveStream()
{
    delete m_pImpl;
}

void SparseStorageArchiveStream::Attach(NintendoContentArchiveReader^ reader)
{
    reader->SetExternalSparseStorage(m_pImpl->GetImpl());
    GC::KeepAlive(this);
}

array<Byte>^ SparseStorageArchiveStream::ReadData(Int64 offset, Int32 size)
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->ReadData(offset, size));
}

Int64 SparseStorageArchiveStream::GetDataSize()
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetDataSize());
}

}}}
