﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fssystem/fs_PartitionFileSystemMeta.h>
#include <FileSystemInfo.h>
#include <SourceInterface.h>

#include <msclr/marshal.h>
#include <vector>

#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_Directory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitUtil.h>

#define NN_DBM_CREATE_METADATA
#include <nn/fssystem/fs_DbmHierarchicalRomFileTableTemplate.impl.h>
#include <nn/fs/fs_IStorage.h>

#include "../Util/DeclareAlive.h"

namespace nn { namespace fssystem {

// Newable 削除したもの
class MemoryStorage : public fs::IStorage
{
    NN_DISALLOW_COPY(MemoryStorage);

private:
    char* const m_Buffer;
    const int64_t m_Size;

public:
    MemoryStorage(void* buffer, int64_t size) NN_NOEXCEPT
        : m_Buffer(static_cast<char*>(buffer))
        , m_Size(size)
    {
    }

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( offset < 0 || m_Size < static_cast<int64_t>(offset + size) )
        {
            return nn::fs::ResultOutOfRange();
        }

        std::memcpy(buffer, m_Buffer + offset, size);

        NN_RESULT_SUCCESS;
    }

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( offset < 0 || m_Size < static_cast<int64_t>(offset + size) )
        {
            return nn::fs::ResultOutOfRange();
        }

        std::memcpy(m_Buffer + offset, buffer, size);

        NN_RESULT_SUCCESS;
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        *outValue = m_Size;
        NN_RESULT_SUCCESS;
    }
};

}}

using namespace nn::fssystem;

typedef HierarchicalRomFileTableTemplate<nn::fs::IStorage, nn::fs::IStorage, nn::fs::IStorage, nn::fs::IStorage> RomFileTable;

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::Collections;
using namespace System::Runtime::InteropServices;
using namespace msclr::interop;

using namespace System::IO;
using namespace System::Text;

using namespace nn::fs;

    public ref class RomFsFileSystemInfo : public FileSystemInfo
    {
    public:
        value class EntryInfo
        {
        public:
            String^ type;
            String^ name;
            UInt64  size;
            UInt64  offset;
            String^ path;
            SourceInterface^ sourceInterface;
            String^ sourceType;

            static String^ GetDirectoryOnRomFs(String^ path)
            {
                if (path == "/")
                {
                    return nullptr;
                }

                auto index = path->LastIndexOf('/');
                if (index == 0)
                {
                    return "/";
                }
                else
                {
                    System::Diagnostics::Debug::Assert(index >= 0);
                    return path->Substring(0, index);
                }
            }

            static String^ GetFileNameOnRomFs(String^ path)
            {
                return path->Substring(path->LastIndexOf('/') + 1);
            }

            property String^ DirectoryOfEntryName
            {
                String^ get()
                {
                    return GetDirectoryOnRomFs(name);
                }
            }

            property String^ FileNameOfEntryName
            {
                String^ get()
                {
                    return GetFileNameOnRomFs(name);
                }
            }
        };

        RomFsFileSystemInfo()
        {
            entries = gcnew Generic::List<EntryInfo>();
            GC::KeepAlive(this);
        }

        void AddFileEntry(EntryInfo^ entryInfo)
        {
            entries->Add(*entryInfo);
            ++fileEntryCount;
            fileEntryBytes += GetAlignedFileEntryBytes(Encoding::UTF8->GetByteCount(entryInfo->FileNameOfEntryName));
        }

        void AddDirectoryEntry(EntryInfo^ entryInfo)
        {
            entries->Add(*entryInfo);
        }

        static int GetAlignedDirectoryEntryBytes(int nameLength)
        {
            return RomFileTable::QueryDirectoryEntrySize(nameLength);
        }

        static int GetAlignedFileEntryBytes(int nameLength)
        {
            return RomFileTable::QueryFileEntrySize(nameLength);
        }

        int version;
        Generic::List<EntryInfo>^ entries;
        int directoryEntryCount;
        long directoryEntryBytes;
        int fileEntryCount;
        long fileEntryBytes;
    };

namespace {

    String^ GetDirectoryName(String^ path)
    {
        String^ parent = RomFsFileSystemInfo::EntryInfo::GetDirectoryOnRomFs(path);
        if( String::IsNullOrEmpty(parent) )
        {
            return parent;
        }
        else
        {
            return parent->Replace("\\", "/");
        }
    }

    // @brief ROM ファイルシステム用ハッシュテーブルに適切なバケット数を決定する。
    int CalculateBucketCount(int entryCount)
    {
        // 原則エントリー数と同数をバケットとして割り当てる
        int count = entryCount;

        if (count < 3)
        {
            count = 3;
        }
        else if (count <= 19)
        {
            count |= 1;
        }
        else
        {
            // 20 以上のバケットを確保する場合、適当に素数に近い数（2 .. 17 で割り切れない数）へ丸める。
            while ( !(count % 2) || !(count % 3) || !(count % 5) || !(count % 7) || !(count % 11) || !(count % 13) || !(count % 17) )
            {
                count++;
            }
        }
        return  count;
    }


    void CreateFile(RomFileTable* pTable, const RomFsFileSystemInfo::EntryInfo& entry)
    {
        RomFileTable::FileInfo info;
        info.offset.Set(entry.offset);
        info.size.Set(entry.size);

        pin_ptr<const wchar_t> pWchar = PtrToStringChars(entry.name);
        char entryNameUtf8[nn::fs::EntryNameLengthMax + 1];
        auto sizeWritten = WideCharToMultiByte(CP_UTF8, 0, pWchar, -1, entryNameUtf8, sizeof(entryNameUtf8), nullptr, nullptr);
        if (sizeWritten == 0 || entryNameUtf8[sizeWritten - 1] != '\0')
        {
            throw gcnew System::Exception(gcnew String("Failed to create a file entry: ") + entry.name);
        }

        RomFileId dummyId = 0;
        if( pTable->CreateFile(&dummyId, entryNameUtf8, info).IsFailure() )
        {
            throw gcnew System::Exception(gcnew String("Failed to create a file entry: ") + entry.name);
        }

        pWchar = nullptr;
    }

    void CreateDirectoryRecursively(RomFileTable* pTable, String^ path)
    {
        if( String::IsNullOrEmpty(path) )
        {
            return;
        }

        // 親ディレクトリを再帰的に作成
        CreateDirectoryRecursively(pTable, GetDirectoryName(path));


        // 指定ディレクトリの作成
        pin_ptr<const wchar_t> pWchar = PtrToStringChars(path);
        char entryNameUtf8[nn::fs::EntryNameLengthMax + 1];
        auto sizeWritten = WideCharToMultiByte(CP_UTF8, 0, pWchar, -1, entryNameUtf8, sizeof(entryNameUtf8), nullptr, nullptr);
        if (sizeWritten == 0 || entryNameUtf8[sizeWritten - 1] != '\0')
        {
            throw gcnew System::Exception(gcnew String("Failed to create a directory entry: ") + path);
        }

        // 存在確認
        RomFileTable::DirectoryInfo info;
        auto result = pTable->GetDirectoryInformation(&info, entryNameUtf8);
        if( nn::fs::ResultDbmNotFound::Includes(result) )
        {
            RomDirectoryId dummyId = 0;
            if( pTable->CreateDirectory(&dummyId, entryNameUtf8, info).IsFailure() )
            {
                throw gcnew System::Exception(gcnew String("Failed to create a directory entry: ") + path);
            }
        }
        else if( result.IsFailure() )
        {
            throw gcnew System::Exception(gcnew String("Failed to create a directory entry: ") + path);
        }

        pWchar = nullptr;
    }
}

    public ref class RomFsFileSystemMetaInfo
    {
    public:
        array<Byte>^ header;
        array<Byte>^ data;
        System::Int64 offsetData;
    };

    public ref class RomFsFileSystemMeta
    {
    public:
        RomFsFileSystemMetaInfo^ Create(RomFsFileSystemInfo^ fileSystemInfo)
        {
            System::Diagnostics::Debug::Assert(fileSystemInfo->directoryEntryBytes > 0);
            System::Diagnostics::Debug::Assert(fileSystemInfo->fileEntryCount == 0 ||
                fileSystemInfo->fileEntryBytes > 0);

            const int MetaDataOffsetAlignment = 8;
            const int RomFsFileSystemHeaderSize = 512;

            NN_STATIC_ASSERT(sizeof(RomFileSystemInformation) <= RomFsFileSystemHeaderSize);

            RomFileTable table;

            UInt32 offset = 0;

            // メタデータサイズでバッファを用意
            auto directoryBucketStorageSize = table.QueryDirectoryEntryBucketStorageSize(CalculateBucketCount(fileSystemInfo->directoryEntryCount));
            auto directoryEntryStorageSize = fileSystemInfo->directoryEntryBytes;
            auto fileBucketStorageSize = table.QueryFileEntryBucketStorageSize(CalculateBucketCount(fileSystemInfo->fileEntryCount));
            auto fileEntryStorageSize = fileSystemInfo->fileEntryBytes;

            auto headerSize = RomFsFileSystemHeaderSize;
            auto dataSize =
                  directoryBucketStorageSize + directoryEntryStorageSize
                + fileBucketStorageSize      + fileEntryStorageSize;

            RomFsFileSystemMetaInfo^ info = gcnew RomFsFileSystemMetaInfo();

            info->header = gcnew array<unsigned char>(headerSize);
            pin_ptr<unsigned char> ptrHeader = &info->header[0];
            RomFileSystemInformation* pHeader = reinterpret_cast<RomFileSystemInformation*>(ptrHeader);

            info->data = gcnew array<unsigned char>(dataSize);
            pin_ptr<unsigned char> ptr = &info->data[0];

            auto directoryBucketStorageOffset = offset;
            MemoryStorage directoryBucketStorage(ptr + directoryBucketStorageOffset, directoryBucketStorageSize);
            offset += directoryBucketStorageSize;

            auto directoryEntryStorageOffset = offset;
            MemoryStorage directoryEntryStorage(ptr + directoryEntryStorageOffset, directoryEntryStorageSize);
            offset += directoryEntryStorageSize;

            auto fileBucketStorageOffset = offset;
            MemoryStorage fileBucketStorage(ptr + fileBucketStorageOffset, fileBucketStorageSize);
            offset += fileBucketStorageSize;

            auto fileEntryStorageOffset = offset;
            MemoryStorage fileEntryStorage(ptr + fileEntryStorageOffset, fileEntryStorageSize);
            offset += fileEntryStorageSize;

            auto fileBodyOffset = headerSize;

            // フォーマット、初期化
            if( table.Format(
                &directoryBucketStorage, 0, directoryBucketStorageSize,
                &directoryEntryStorage,  0, directoryEntryStorageSize,
                &fileBucketStorage,      0, fileBucketStorageSize,
                &fileEntryStorage,       0, fileEntryStorageSize
                ).IsFailure() )
            {
                throw gcnew System::Exception("Failed to initialize meta data.");
            }

            if( table.Initialize(
                &directoryBucketStorage, 0, directoryBucketStorageSize,
                &directoryEntryStorage,  0, directoryEntryStorageSize,
                &fileBucketStorage,      0, fileBucketStorageSize,
                &fileEntryStorage,       0, fileEntryStorageSize
                ).IsFailure() )
            {
                throw gcnew System::Exception("Failed to initialize meta data.");
            }

            if( table.CreateRootDirectory().IsFailure() )
            {
                throw gcnew System::Exception("Failed to initialize meta data.");
            }


            // エントリを追加
            for(int i = 0; i < fileSystemInfo->entries->Count; i++)
            {
                const auto& entry = fileSystemInfo->entries[i];

                if( entry.type == "file" || entry.type == "source" )
                {
                    // 必要に応じて親ディレクトリを追加
                    CreateDirectoryRecursively(&table, GetDirectoryName(entry.name));

                    CreateFile(&table, entry);
                }
                else if ( entry.type == "directory" )
                {
                    // 指定に応じてディレクトリを追加
                    CreateDirectoryRecursively(&table, entry.name);
                }
                else
                {
                    // adf 解釈時に不正な type は弾かれているはず
                    throw gcnew System::Exception(gcnew String("Invalid entry type: ") + entry.type);
                }
            }
            auto lastEntry = fileSystemInfo->entries[fileSystemInfo->entries->Count - 1];
            auto fileBodySize = lastEntry.offset + lastEntry.size;
            auto alignedFileBodySize = nn::util::align_up(fileBodySize, MetaDataOffsetAlignment);
            info->offsetData = headerSize + alignedFileBodySize;

            // 実際に必要だったメタデータサイズと事前に計算したデータサイズが一致していることを確認する
            uint32_t actualDirectoryEntryStorageSize;
            uint32_t actualFileEntryStorageSize;
            if( table.QueryRomFileSystemSize(&actualDirectoryEntryStorageSize, &actualFileEntryStorageSize).IsFailure() ||
                directoryEntryStorageSize != actualDirectoryEntryStorageSize ||
                fileEntryStorageSize != actualFileEntryStorageSize)
            {
                throw gcnew System::Exception("Failed to verification.");
            }

            // ヘッダを生成
            std::memset(pHeader, 0, RomFsFileSystemHeaderSize);

            pHeader->size = sizeof(RomFileSystemInformation);
            pHeader->offsetBucketDirectory = directoryBucketStorageOffset;
            pHeader->sizeBucketDirectory   = directoryBucketStorageSize;
            pHeader->offsetDirectoryEntry  = directoryEntryStorageOffset;
            pHeader->sizeDirectoryEntry    = directoryEntryStorageSize;
            pHeader->offsetBucketFile      = fileBucketStorageOffset;
            pHeader->sizeBucketFile        = fileBucketStorageSize;
            pHeader->offsetFileEntry       = fileEntryStorageOffset;
            pHeader->sizeFileEntry         = fileEntryStorageSize;
            pHeader->offsetFileBody        = fileBodyOffset;

            pHeader->offsetBucketDirectory += info->offsetData;
            pHeader->offsetDirectoryEntry  += info->offsetData;
            pHeader->offsetBucketFile      += info->offsetData;
            pHeader->offsetFileEntry       += info->offsetData;

            ptr = nullptr;

            return Util::ReturnAndDeclareAlive(this, info);
        } // NOLINT(impl/function_size)
    };
}}}
