﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/utilTool/fs_AesCtrCounterExtendedStorageBuilder.h>
#include <nn/fssystem/utilTool/fs_IndirectStorageBuilder.h>
#include <nn/fssystem/utilTool/fs_BinaryRegionFile.h>
#include <nn/fssystem/utilTool/fs_BinaryMatch.h>
#include <nn/fssystem/fs_NcaHeader.h>
#include <nn/util/util_IntUtil.h>
#include "DefaultMemoryResource.h"
#include "IndirectStorageStream.h"
#include "IndirectStorageDataCacheStorageBuilder.h"
#include "ManagedStreamStorage.h"
#include "../Util/DeclareAlive.h"

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::IO;
using namespace System::Runtime::InteropServices;

using namespace nn;
using namespace nn::fssystem;
using namespace nn::fssystem::utilTool;

private class IndirectStorageStreamBase
{
public:
    IndirectStorageStreamBase(IReadOnlySource^ oldSource, IReadOnlySource^ newSource)
        : m_Allocator()
        , m_OldStorage(oldSource)
        , m_NewStorage(newSource)
        , m_ExcludeRangeOld()
        , m_ExcludeRangeNew()
        , m_IndirectBuilder()
        , m_pBinaryRegionInfo()
    {
    }

    void SetExcludeRangeForOldSource(array<IndirectStorageStream::ExcludeRange>^ ranges)
    {
        m_ExcludeRangeOld.resize(ranges->Length);
        for( int i = 0; i < ranges->Length; ++i )
        {
            m_ExcludeRangeOld[i].offset = ranges[i].offset;
            m_ExcludeRangeOld[i].size = ranges[i].size;
        }

        m_IndirectBuilder.SetExcludeRangeForOldStorage(
            m_ExcludeRangeOld.data(), static_cast<int>(m_ExcludeRangeOld.size()));
    }

    void SetExcludeRangeForNewSource(array<IndirectStorageStream::ExcludeRange>^ ranges)
    {
        m_ExcludeRangeNew.resize(ranges->Length);
        for( int i = 0; i < ranges->Length; ++i )
        {
            m_ExcludeRangeNew[i].offset = ranges[i].offset;
            m_ExcludeRangeNew[i].size = ranges[i].size;
        }

        m_IndirectBuilder.SetExcludeRangeForNewStorage(
            m_ExcludeRangeNew.data(), static_cast<int>(m_ExcludeRangeNew.size()));
    }

    void SetBinaryRegionInfo(String^ originalPath, String^ outputPath, array<Byte>^ rawHeader)
    {
        m_pBinaryRegionInfo.reset(new BinaryRegionInfo(originalPath, outputPath, rawHeader));
    }

    void BuildImpl(UInt32 shiftSize, UInt32 blockSize, UInt32 regionSize, Int64 matchSize, Int64 windowSize, bool willUseCache)
    {
        nn::Result result;

        if( m_pBinaryRegionInfo != nullptr )
        {
            result = m_pBinaryRegionInfo->Load(&m_OldStorage, 0, blockSize, regionSize);
            if( result.IsFailure() )
            {
                throw gcnew Exception(String::Format("Failed to Load BinaryRegionInfo 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
        }

        IndirectStorageDataCacheStorageBuilder cacheOldBuilder;
        IndirectStorageDataCacheStorageBuilder cacheNewBuilder;

        if (willUseCache)
        {
            result = cacheOldBuilder.Build(&m_OldStorage);
            if (result.IsFailure())
            {
                throw gcnew Exception(String::Format("Failed to Initialize IndirectStorageDataCacheStorageBuilder(Old) 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
            result = cacheNewBuilder.Build(&m_NewStorage);
            if (result.IsFailure())
            {
                throw gcnew Exception(String::Format("Failed to Initialize IndirectStorageDataCacheStorageBuilder(New) 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
            result = m_IndirectBuilder.Initialize(cacheOldBuilder.GetCachedStorage(), cacheNewBuilder.GetCachedStorage(), blockSize);
        }
        else
        {
            result = m_IndirectBuilder.Initialize(&m_OldStorage, &m_NewStorage, blockSize);
        }
        if( result.IsFailure() )
        {
            throw gcnew System::Exception(
                String::Format("Failed to Initialize IndirectStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
        }

        if( (m_pBinaryRegionInfo != nullptr) && m_pBinaryRegionInfo->IsValid() )
        {
            m_IndirectBuilder.SetBinaryRegion(m_pBinaryRegionInfo->GetRegionArray());
        }

        result = m_IndirectBuilder.Build(shiftSize, blockSize, regionSize, matchSize, windowSize);
        if( result.IsFailure() )
        {
            throw gcnew System::Exception(
                String::Format("Failed to Build IndirectStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
        }
        if (willUseCache)
        {
            // キャッシュストレージの寿命はこの関数のため、等価な寿命の長い元のストレージに戻す
            m_IndirectBuilder.SetDataStorages(&m_OldStorage, &m_NewStorage);
        }

        // int にキャストできるかチェック
        // NOTE: この制限はオーサリングツール側の制限で nca ファイルの制限ではない
        //       （IndirectStorage テーブルデータのストリーミングに対応していないため）
        const auto size = GetIndirectTableSize();
        if( !util::IsIntValueRepresentable<int>(size) )
        {
            throw gcnew System::Exception("IndirectStorage table too large.");
        }

        if( m_pBinaryRegionInfo != nullptr )
        {
            m_pBinaryRegionInfo->Save(m_IndirectBuilder.GetBinaryRegion());
        }
    }

    void WriteIndirectTable(array<Byte>^ headerBuffer, array<Byte>^ tableBuffer)
    {
        const auto headerSize = GetIndirectHeaderSize();
        const auto tableSize = static_cast<int>(GetIndirectTableSize());

        if( headerBuffer->GetLength(0) < GetIndirectHeaderSize() || tableBuffer->GetLength(0) < GetIndirectTableSize() )
        {
            throw gcnew System::ArgumentException();
        }

        pin_ptr<unsigned char> headerPtr = &headerBuffer[0];
        fs::MemoryStorage headerStorage(headerPtr, headerSize);
        pin_ptr<unsigned char> tablePtr = &tableBuffer[0];
        fs::MemoryStorage tableStorage(tablePtr, tableSize);

        const int64_t nodeOffset = 0;
        const auto nodeSize = m_IndirectBuilder.QueryTableNodeStorageSize();
        const auto entryOffset = nodeOffset + nodeSize;
        const auto entrySize = m_IndirectBuilder.QueryTableEntryStorageSize();

        auto result = m_IndirectBuilder.WriteTable(
            &m_Allocator,
            fs::SubStorage(&headerStorage, 0, headerSize),
            fs::SubStorage(&tableStorage, nodeOffset, nodeSize),
            fs::SubStorage(&tableStorage, entryOffset, entrySize)
        );

        headerPtr = nullptr;
        tablePtr = nullptr;

        if( result.IsFailure() )
        {
            throw gcnew System::Exception(
                String::Format("Failed to WriteTable IndirectStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
        }
    }

    void OutputBuildLog(Stream^ tableStream, Stream^ fragmentStream, Int64 startOffset)
    {
        auto nodeStream = gcnew MemoryStream();
        auto entryStream = gcnew MemoryStream();
        {
            ManagedStreamStorage tableStorage(tableStream);
            ManagedStreamStorage nodeStorage(nodeStream);
            ManagedStreamStorage entryStorage(entryStream);
            ManagedStreamStorage dataStorage(fragmentStream);

            auto result = m_IndirectBuilder.OutputBuildLog(&m_Allocator, &tableStorage, &nodeStorage, &entryStorage, &dataStorage, startOffset);
            if( result.IsFailure() )
            {
                throw gcnew System::Exception(
                    String::Format("Failed to OutputBuildLog 0x{0:X8}.", result.GetInnerValueForDebug())
                );
            }
        }

        nodeStream->Position = 0;
        nodeStream->CopyTo(tableStream);
        entryStream->Position = 0;
        entryStream->CopyTo(tableStream);

        delete nodeStream;
        delete entryStream;
    }

    IndirectStorageProgress GetProgress() const
    {
        IndirectStorageProgress progress;

        auto pProgress = m_IndirectBuilder.GetProgress();
        if( pProgress != __nullptr )
        {
            const auto info = pProgress->GetInfo();

            progress.phase = IndirectStorageProgress::Phase(info.phase);
            progress.total = info.total;
            progress.value = info.value;
        }
        return progress;
    }

    Int64 GetIndirectDataSize() const
    {
        return util::align_up(m_IndirectBuilder.QueryDataStorageSize(), NcaHeader::XtsBlockSize);
    }

    Int32 GetIndirectHeaderSize() const
    {
        return static_cast<int>(m_IndirectBuilder.QueryTableHeaderStorageSize());
    }

    Int64 GetIndirectTableSize() const
    {
        return m_IndirectBuilder.QueryTableNodeStorageSize() + m_IndirectBuilder.QueryTableEntryStorageSize();
    }

protected:
    ManagedSourceStorage& GetOldStorage()
    {
        return m_OldStorage;
    }

    ManagedSourceStorage& GetNewStorage()
    {
        return m_NewStorage;
    }

    DefaultAlignedMemoryResource& GetAllocator()
    {
        return m_Allocator;
    }

    utilTool::IndirectStorageBuilder& GetIndirectBuilder()
    {
        return m_IndirectBuilder;
    }

private:
    DefaultAlignedMemoryResource m_Allocator;
    ManagedSourceStorage m_OldStorage;
    ManagedSourceStorage m_NewStorage;
    std::vector<utilTool::IndirectStorageBuilder::ExcludeRange> m_ExcludeRangeOld;
    std::vector<utilTool::IndirectStorageBuilder::ExcludeRange> m_ExcludeRangeNew;
    utilTool::IndirectStorageBuilder m_IndirectBuilder;
    std::unique_ptr<BinaryRegionInfo> m_pBinaryRegionInfo;
};

private class IndirectStorageStreamImpl : public IndirectStorageStreamBase
{
public:
    IndirectStorageStreamImpl(IReadOnlySource^ oldSource, IReadOnlySource^ newSource)
        : IndirectStorageStreamBase(oldSource, newSource)
        , m_AesCtrExBuilder()
    {
    }

    void SetBinaryMatchHint(array<IndirectStorageStream::BinaryMatchHint>^ hints)
    {
        if( (hints != nullptr) && (0 < hints->GetLength(0)) )
        {
            pin_ptr<IndirectStorageStream::BinaryMatchHint> ptr;
            ptr = &hints[0];
            GetIndirectBuilder().SetBinaryMatchHint(reinterpret_cast<BinaryMatchHint*>(ptr), hints->GetLength(0));
            ptr = nullptr;
        }
    }

    void Build(UInt32 shiftSize, UInt32 blockSize, UInt32 regionSize, Int64 matchSize, Int64 windowSize, bool willUseCache)
    {
        BuildImpl(shiftSize, blockSize, regionSize, matchSize, windowSize, willUseCache);

        m_AesCtrExBuilder.Initialize(GetIndirectDataSize() + GetIndirectTableSize(), 0);
    }

    void WriteAesCtrExTable(array<Byte>^ headerBuffer, array<Byte>^ tableBuffer)
    {
        const auto headerSize = GetAesCtrExHeaderSize();
        const auto tableSize = static_cast<int>(GetAesCtrExTableSize());

        if( headerBuffer->GetLength(0) < GetAesCtrExHeaderSize() || tableBuffer->GetLength(0) < GetAesCtrExTableSize() )
        {
            throw gcnew System::ArgumentException();
        }

        pin_ptr<unsigned char> headerPtr = &headerBuffer[0];
        fs::MemoryStorage headerStorage(headerPtr, headerSize);
        pin_ptr<unsigned char> tablePtr = &tableBuffer[0];
        fs::MemoryStorage tableStorage(tablePtr, tableSize);

        const int64_t nodeOffset = 0;
        const auto nodeSize = m_AesCtrExBuilder.QueryTableNodeStorageSize();
        const auto entryOffset = nodeOffset + nodeSize;
        const auto entrySize = m_AesCtrExBuilder.QueryTableEntryStorageSize();

        auto result = m_AesCtrExBuilder.WriteTable(
            &GetAllocator(),
            fs::SubStorage(&headerStorage, 0, headerSize),
            fs::SubStorage(&tableStorage, nodeOffset, nodeSize),
            fs::SubStorage(&tableStorage, entryOffset, entrySize)
        );

        headerPtr = nullptr;
        tablePtr = nullptr;

        if( result.IsFailure() )
        {
            throw gcnew System::Exception(
                String::Format("Failed to WriteTable AesCtrCounterExtendedStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
        }
    }

    array<Byte>^ ReadData(Int64 offset, Int32 size)
    {
        array<Byte>^ buffer = gcnew array<Byte>(size);
        pin_ptr<unsigned char> ptr = &buffer[0];

        auto result = GetIndirectBuilder().ReadData(offset, ptr, size);
        ptr = nullptr;

        if( result.IsFailure() )
        {
            throw gcnew System::Exception(
                String::Format("Failed to ReadData IndirectStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
        }

        return buffer;
    }

    Int32 GetAesCtrExHeaderSize() const
    {
        return static_cast<int>(m_AesCtrExBuilder.QueryTableHeaderStorageSize());
    }

    Int64 GetAesCtrExTableSize() const
    {
        return m_AesCtrExBuilder.QueryTableNodeStorageSize() + m_AesCtrExBuilder.QueryTableEntryStorageSize();
    }

private:
    AesCtrCounterExtendedStorageBuilder m_AesCtrExBuilder;
};

IndirectStorageStream::IndirectStorageStream(IReadOnlySource^ oldSource, IReadOnlySource^ newSource, array<BinaryMatchHint>^ binaryMatchHints)
    : m_BinaryMatchHints(binaryMatchHints)
{
    m_pImpl = new IndirectStorageStreamImpl(oldSource, newSource);
    GC::KeepAlive(this);
}

IndirectStorageStream::~IndirectStorageStream()
{
    this->!IndirectStorageStream();
}

IndirectStorageStream::!IndirectStorageStream()
{
    delete m_pImpl;
}

void IndirectStorageStream::SetExcludeRangeForOldSource(array<ExcludeRange>^ ranges)
{
    m_pImpl->SetExcludeRangeForOldSource(ranges);
    GC::KeepAlive(this);
}

void IndirectStorageStream::SetExcludeRangeForNewSource(array<ExcludeRange>^ ranges)
{
    m_pImpl->SetExcludeRangeForNewSource(ranges);
    GC::KeepAlive(this);
}

void IndirectStorageStream::SetBinaryRegionInfo(String^ originalPath, String^ outputPath, array<Byte>^ rawHeader)
{
    m_pImpl->SetBinaryRegionInfo(originalPath, outputPath, rawHeader);
    GC::KeepAlive(this);
}

void IndirectStorageStream::Build(UInt32 shiftSize, UInt32 blockSize, UInt32 regionSize, Int64 matchSize, Int64 windowSize, bool willUseCache)
{
    m_pImpl->SetBinaryMatchHint(m_BinaryMatchHints);
    m_pImpl->Build(shiftSize, blockSize, regionSize, matchSize, windowSize, willUseCache);
    GC::KeepAlive(this);
}

void IndirectStorageStream::WriteIndirectTable(array<Byte>^ headerBuffer, array<Byte>^ tableBuffer)
{
    m_pImpl->WriteIndirectTable(headerBuffer, tableBuffer);
    GC::KeepAlive(this);
}

void IndirectStorageStream::WriteAesCtrExTable(array<Byte>^ headerBuffer, array<Byte>^ tableBuffer)
{
    m_pImpl->WriteAesCtrExTable(headerBuffer, tableBuffer);
    GC::KeepAlive(this);
}

array<Byte>^ IndirectStorageStream::ReadData(Int64 offset, Int32 size)
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->ReadData(offset, size));
}

void IndirectStorageStream::OutputBuildLog(Stream^ tableStream, Stream^ fragmentStream, Int64 startOffset)
{
    m_pImpl->OutputBuildLog(tableStream, fragmentStream, startOffset);
    GC::KeepAlive(this);
}

IndirectStorageProgress IndirectStorageStream::GetProgress()
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetProgress());
}

Int64 IndirectStorageStream::GetDataSize()
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetIndirectDataSize());
}

Int32 IndirectStorageStream::GetIndirectHeaderSize()
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetIndirectHeaderSize());
}

Int32 IndirectStorageStream::GetIndirectTableSize()
{
    return Util::ReturnAndDeclareAlive(this, static_cast<Int32>(m_pImpl->GetIndirectTableSize()));
}

Int32 IndirectStorageStream::GetAesCtrExHeaderSize()
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetAesCtrExHeaderSize());
}

Int32 IndirectStorageStream::GetAesCtrExTableSize()
{
    return Util::ReturnAndDeclareAlive(this, static_cast<Int32>(m_pImpl->GetAesCtrExTableSize()));
}

private class IndirectStorageDifferenceStreamImpl : public IndirectStorageStreamBase
{
public:
    IndirectStorageDifferenceStreamImpl(IReadOnlySource^ oldSource, IReadOnlySource^ newSource)
        : IndirectStorageStreamBase(oldSource, newSource)
        , m_pDifferenceStorage()
    {
    }

    void Build(UInt32 shiftSize, UInt32 blockSize, UInt32 regionSize, Int64 matchSize, Int64 windowSize)
    {
        BuildImpl(shiftSize, blockSize, regionSize, matchSize, windowSize, false);
    }

    void Import(UInt32 blockSize, array<Byte>^ headerData, array<Byte>^ tableData, Stream^ differenceStream)
    {
        auto result = GetIndirectBuilder().Initialize(&GetOldStorage(), &GetNewStorage(), blockSize);
        if( result.IsFailure() )
        {
            throw gcnew Exception(String::Format("Failed to Initialize IndirectStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
        }
        {
            BucketTree::Header header;
            BucketTreeUtility::CopyHeader(&header, headerData);

            const auto nodeOffset = 0;
            const auto nodeSize = fssystem::IndirectStorage::QueryNodeStorageSize(header.entryCount);
            const auto entryOffset = nodeOffset + nodeSize;
            const auto entrySize = fssystem::IndirectStorage::QueryEntryStorageSize(header.entryCount);

            if( tableData->GetLength(0) < entryOffset + entrySize )
            {
                throw gcnew ArgumentException(String::Format("Failed to copy table while creating indirect storage."));
            }

            pin_ptr<unsigned char> tablePtr = &tableData[0];

            fs::MemoryStorage headerStorage(&header, sizeof(header));
            fs::MemoryStorage tableStorage(tablePtr, tableData->GetLength(0));

            result = GetIndirectBuilder().Build(&GetAllocator(), fs::SubStorage(&headerStorage, 0, sizeof(header)), fs::SubStorage(&tableStorage, nodeOffset, nodeSize), fs::SubStorage(&tableStorage, entryOffset, entrySize));

            tablePtr = nullptr;
        }
        if( result.IsFailure() )
        {
            throw gcnew Exception(String::Format("Failed to Build IndirectStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
        }
        // int にキャストできるかチェック
        // NOTE: この制限はオーサリングツール側の制限で nca ファイルの制限ではない
        //       （IndirectStorage テーブルデータのストリーミングに対応していないため）
        if( !util::IsIntValueRepresentable<int>(GetIndirectTableSize()) )
        {
            throw gcnew System::Exception("IndirectStorage table too large.");
        }
        m_pDifferenceStorage.reset(new ManagedStreamStorage(differenceStream));
    }

    array<Byte>^ ReadData(Int64 offset, Int32 size)
    {
        if( m_pDifferenceStorage == nullptr )
        {
            throw gcnew InvalidOperationException();
        }
        array<Byte>^ buffer = gcnew array<Byte>(size);
        {
            pin_ptr<unsigned char> ptr = &buffer[0];
            auto result = GetIndirectBuilder().ReadData(offset, ptr, size, m_pDifferenceStorage.get());
            ptr = nullptr;
            if( result.IsFailure() )
            {
                throw gcnew Exception(String::Format("Failed to ReadData IndirectStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
        }
        return buffer;
    }

private:
    std::unique_ptr<ManagedStreamStorage> m_pDifferenceStorage;
};

IndirectStorageDifferenceStream::IndirectStorageDifferenceStream(IReadOnlySource^ oldSource, IReadOnlySource^ newSource)
{
    m_pImpl = new IndirectStorageDifferenceStreamImpl(oldSource, newSource);
    GC::KeepAlive(this);
}

IndirectStorageDifferenceStream::~IndirectStorageDifferenceStream()
{
    this->!IndirectStorageDifferenceStream();
}

IndirectStorageDifferenceStream::!IndirectStorageDifferenceStream()
{
    delete m_pImpl;
}

void IndirectStorageDifferenceStream::SetExcludeRangeForOldSource(array<IndirectStorageStream::ExcludeRange>^ ranges)
{
    m_pImpl->SetExcludeRangeForOldSource(ranges);
    GC::KeepAlive(this);
}

void IndirectStorageDifferenceStream::SetExcludeRangeForNewSource(array<IndirectStorageStream::ExcludeRange>^ ranges)
{
    m_pImpl->SetExcludeRangeForNewSource(ranges);
    GC::KeepAlive(this);
}

void IndirectStorageDifferenceStream::Build(UInt32 shiftSize, UInt32 blockSize, UInt32 regionSize, Int64 matchSize, Int64 windowSize)
{
    m_pImpl->Build(shiftSize, blockSize, regionSize, matchSize, windowSize);
    GC::KeepAlive(this);
}

void IndirectStorageDifferenceStream::Import(UInt32 blockSize, array<Byte>^ headerData, array<Byte>^ tableData, Stream^ differenceStream)
{
    m_pImpl->Import(blockSize, headerData, tableData, differenceStream);
    GC::KeepAlive(this);
}

void IndirectStorageDifferenceStream::WriteIndirectTable(array<Byte>^ headerBuffer, array<Byte>^ tableBuffer)
{
    m_pImpl->WriteIndirectTable(headerBuffer, tableBuffer);
    GC::KeepAlive(this);
}

array<Byte>^ IndirectStorageDifferenceStream::ReadData(Int64 offset, Int32 size)
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->ReadData(offset, size));
}

void IndirectStorageDifferenceStream::OutputBuildLog(Stream^ tableStream, Stream^ fragmentStream, Int64 startOffset)
{
    m_pImpl->OutputBuildLog(tableStream, fragmentStream, startOffset);
    GC::KeepAlive(this);
}

IndirectStorageProgress IndirectStorageDifferenceStream::GetProgress()
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetProgress());
}

Int64 IndirectStorageDifferenceStream::GetDataSize()
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetIndirectDataSize());
}

Int32 IndirectStorageDifferenceStream::GetIndirectHeaderSize()
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetIndirectHeaderSize());
}

Int32 IndirectStorageDifferenceStream::GetIndirectTableSize()
{
    return Util::ReturnAndDeclareAlive(this, static_cast<Int32>(m_pImpl->GetIndirectTableSize()));
}

}}}
