﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <map>
#include <nn/nn_Allocator.h>
#include <nn/nn_Macro.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fssystem/buffers/fs_FileSystemBufferManager.h>
#include <nn/fssystem/save/fs_BufferedStorage.h>
#include <nn/util/util_BitUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <msclr/marshal.h>

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

    /*!
        :private

        @brief パッチの生成中のみ使用可能なキャッシュストレージ、キャッシュバッファーを管理するクラス
               パッチの生成処理が完了次第すぐにキャッシュバッファーを解放したいので専用の BufferManager を保持する
    */
    private class IndirectStorageDataCacheStorageBuilder
    {
    public:
        IndirectStorageDataCacheStorageBuilder()
            : m_Buffer()
        {
        }

    public:
        nn::Result Build(nn::fs::IStorage* pStorage)
        {
            m_Buffer.reset(new char[BufferSize]);

            NN_RESULT_DO(m_BufferManager.Initialize(
                CacheCount,
                reinterpret_cast<uintptr_t>(m_Buffer.get()),
                BufferSize,
                BlockSize));

            int64_t size;
            NN_RESULT_DO(pStorage->GetSize(&size));

            NN_RESULT_DO(m_CacheStorage.Initialize(
                nn::fs::SubStorage(pStorage, 0, size),
                &m_BufferManager,
                BlockSize,
                CacheCount));

            NN_RESULT_SUCCESS;
        }

        nn::fs::IStorage* GetCachedStorage()
        {
            return &m_CacheStorage;
        }

    private:
        static const int CacheCount = 128;
        static const int BlockSize = 512 * 1024;
        static const size_t BufferSize = CacheCount * BlockSize;

    private:
        std::unique_ptr<char[]> m_Buffer;
        nn::fssystem::FileSystemBufferManager m_BufferManager;
        nn::fssystem::save::BufferedStorage m_CacheStorage;
    };

}}}
