﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/crypto/crypto_Aes128XtsEncryptor.h>
#include <nn/crypto/crypto_Aes128XtsDecryptor.h>
#include <nn/fssrv/fssrv_NcaCryptoConfiguration.h>
#include "IEncryptor.h"

namespace Nintendo { namespace Authoring { namespace CryptoLibrary {

using namespace System;

    public ref class Aes128XtsCryptoDriver : IXtsModeEncryptor
    {
    public:
        property int KeySize
        {
            virtual int get() { return 16; }
        };

        Aes128XtsCryptoDriver(array<Byte>^ keyData1, array<Byte>^ keyData2)
        {
            if (keyData1->Length != KeySize || keyData2->Length != KeySize)
            {
                throw gcnew ArgumentException();
            }
            m_Key1 = gcnew array<Byte>(KeySize);
            m_Key2 = gcnew array<Byte>(KeySize);
            System::Buffer::BlockCopy(keyData1, 0, m_Key1, 0, KeySize);
            System::Buffer::BlockCopy(keyData2, 0, m_Key2, 0, KeySize);
            GC::KeepAlive(this);
        }

        virtual void EncryptBlock(array<Byte>^ iv, array<Byte>^ src, int srcOffset, int srcSize, array<Byte>^ dst, int dstOffset)
        {
            {
                pin_ptr<unsigned char> pinKey1 = &m_Key1[0];
                pin_ptr<unsigned char> pinKey2 = &m_Key2[0];
                pin_ptr<unsigned char> pinSrc = &src[srcOffset];
                pin_ptr<unsigned char> pinIv = &iv[0];
                pin_ptr<unsigned char> pinDst = &dst[dstOffset];
                auto encryptedSize = nn::crypto::EncryptAes128Xts(pinDst, srcSize, pinKey1, pinKey2, KeySize, pinIv, KeySize, pinSrc, srcSize);
                pinKey1 = nullptr;
                pinKey2 = nullptr;
                pinSrc = nullptr;
                pinIv = nullptr;
                pinDst = nullptr;
                if (encryptedSize != srcSize)
                {
                    throw gcnew InvalidOperationException();
                }
            }
            GC::KeepAlive(this);
        }

        virtual void DecryptBlock(array<Byte>^ iv, array<Byte>^ src, int srcOffset, int srcSize, array<Byte>^ dst, int dstOffset)
        {
            EncryptBlock(iv, src, srcOffset, srcSize, dst, dstOffset);
            GC::KeepAlive(this);
        }

        static array<Byte>^ GetDefaultKey(int index)
        {
            switch (index)
            {
            case 0:
                return NcaBodyEncryptionKey1;
            case 1:
                return NcaBodyEncryptionKey2;
            case 2:
                return NcaHeaderEncryptionKey1;
            case 3:
                return NcaHeaderEncryptionKey2;
            default:
                throw gcnew ArgumentException();
            }
        }

        // TORIAEZU: 固定鍵
        static array<Byte>^ NcaBodyEncryptionKey1 =
        {
            0xe5, 0x64, 0xdb, 0xfe, 0x52, 0x93, 0x3a, 0x65, 0x3c, 0x8b, 0x5e, 0xf8, 0x2c, 0x9d, 0xf4, 0xb5
        };
        static array<Byte>^ NcaBodyEncryptionKey2 =
        {
            0x60, 0x7b, 0x77, 0x3e, 0x31, 0xe9, 0x6a, 0x8f, 0x8e, 0x44, 0x5c, 0x98, 0x71, 0xc0, 0x57, 0xdb
        };


        static array<Byte>^ NcaHeaderEncryptionKey1 =
        {
            0xcb, 0x9a, 0x93, 0x9f, 0x82, 0x72, 0x54, 0x4a, 0x74, 0x5d, 0x28, 0x46, 0x9d, 0xcc, 0x38, 0x12
        };
        static array<Byte>^ NcaHeaderEncryptionKey2 =
        {
            0x06, 0x31, 0x27, 0x06, 0xae, 0x62, 0x56, 0x8c, 0x5b, 0x7e, 0xe6, 0x9f, 0x7e, 0x01, 0x02, 0x24
        };

    private:
        array<Byte>^ m_Key1;
        array<Byte>^ m_Key2;
    };

}}}
