﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Xml.Linq;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class UpdatePartitionUtils
    {
        public const string UpdatePartitionXmlName = "UpdatePartition.xml";

        public class UpdatePartitionXmlReader
        {
            private XDocument m_Document;
            public UpdatePartitionXmlReader(byte[] bytes)
            {
                using (var stream = new MemoryStream(bytes))
                {
                    m_Document = XDocument.Load(stream);
                }
            }
            public ulong GetId()
            {
                var idElem = m_Document.Descendants("SystemUpdateMetaId");
                if (!idElem.Any())
                {
                    throw new ArgumentException("Invalid UpdatePartition.xml. <SystemUpdateMetaId> is not specified.");
                }
                return Convert.ToUInt64((string)idElem.Single(), 16);
            }
            public uint GetVersion()
            {
                var versionElem = m_Document.Descendants("SystemUpdateMetaVersion");
                if (!versionElem.Any())
                {
                    throw new ArgumentException("Invalid UpdatePartition.xml. <SystemUpdateMetaVersion> is not specified.");
                }
                return Convert.ToUInt32((string)versionElem.Single(), 10);
            }
            public uint GetVersionTrancatedToRelease()
            {
                return GetVersion() & 0xFFFF0000;
            }
            public byte[] GetHash()
            {
                var hashElem = m_Document.Descendants("Hash");
                if (!hashElem.Any())
                {
                    throw new ArgumentException("Invalid UpdatePartition.xml. <Hash> is not specified.");
                }
                var hashStr = hashElem.Select(x => (string)x).Single();
                Debug.Assert(hashStr.Length == XciInfo.UppHashSize * 2);
                var bytes = new byte[XciInfo.UppHashSize];
                {
                    int j = 0;
                    for (int i = 0; i < bytes.Length; i++)
                    {
                        bytes[i] = Convert.ToByte(hashStr.Substring(j, 2), 16);
                        j += 2;
                    }
                }
                return bytes;
            }
        }

        static public void CheckUpdatePartitionVersionHighEnough(NintendoSubmissionPackageReader nspReader, uint systemUpdateMetaVersion)
        {
            if (systemUpdateMetaVersion == 0)
            {
                // no update partition
                return;
            }

            foreach (var model in ArchiveReconstructionUtils.ReadContentMetaInNsp(nspReader))
            {
                uint requiredSystemVersion = 0;
                switch (model.Type)
                {
                    case NintendoContentMetaConstant.ContentMetaTypeApplication:
                        {
                            var appModel = model as ApplicationContentMetaModel;
                            requiredSystemVersion = appModel.RequiredSystemVersion;
                        }
                        break;
                    case NintendoContentMetaConstant.ContentMetaTypePatch:
                        {
                            var patchModel = model as PatchContentMetaModel;
                            if (patchModel.OriginalRequiredSystemVersion.HasValue)
                            {
                                requiredSystemVersion = patchModel.OriginalRequiredSystemVersion.Value;
                            }
                            else
                            {
                                requiredSystemVersion = patchModel.RequiredSystemVersion;
                            }
                        }
                        break;
                    default:
                        continue;
                }

                if (requiredSystemVersion > systemUpdateMetaVersion)
                {
                    throw new ArgumentException(string.Format("The version of specified update partition (= {0}) is older than that of application or patch being included (= {1}).", systemUpdateMetaVersion, requiredSystemVersion));
                }
            }
        }

        static public byte GetLatestKeyGeneration(NintendoSubmissionPackageReader nspReader)
        {
            var keyGeneration = (byte)0;
            foreach (var cnmt in ArchiveReconstructionUtils.ReadContentMetaInNsp(nspReader))
            {
                foreach (var content in cnmt.ContentList)
                {
                    if (keyGeneration < (byte)content.KeyGeneration)
                    {
                        keyGeneration = (byte)content.KeyGeneration;
                    }
                }
            }
            return keyGeneration;
        }
    }
}
