﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Xml;
using System.Xml.Linq;
using System.Xml.XPath;
using System.Xml.Serialization;
using System.Collections.Generic;
using Nintendo.ApplicationControlProperty;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class IdConverter
    {
        public static UInt64 ConvertToAocBaseId(UInt64 applicationId)
        {
            return applicationId + 0x1000;
        }
        public static UInt64 ConvertStringToId(string value, string tagName, string parentTagName)
        {
            UInt64 id;
            try
            {
                id = Convert.ToUInt64(value, 16);
            }
            catch (Exception)
            {
                throw new ArgumentException(string.Format("<{0}> should be a hexadecimal number in <{1}>.", tagName, parentTagName));
            }

            return id;
        }

        public static UInt64 ConvertToPatchId(UInt64 applicationId)
        {
            return applicationId + 0x800;
        }
        public static UInt64 ConvertToDeltaId(UInt64 applicationId)
        {
            return applicationId + 0xC00;
        }
    }

    // TODO: CoreInfo, ContentMetaBase と
    //       xxxModel は役割も機能も似ているので統合したい
    //       0.12 系の AuthoringTool の修正が落ち着いてから作業をする

    public class CoreInfo
    {
        public CoreInfo(XElement element)
        {
            ProgramId = GetProgramId(element);
            PatchId = GetPatchId(element);
        }

        private UInt64 GetProgramId(XElement element)
        {
            string[] ProgramIdAliases =
            {
                "ApplicationId",
                "ProgramId",
                "SystemProgram",
            };

            foreach (var programIdName in ProgramIdAliases)
            {
                var xmlId =
                    from id in element.Descendants(programIdName)
                    select (string)id;
                if (xmlId.Any())
                {
                    return Convert.ToUInt64(Enumerable.First(xmlId), 16);
                }
            }
            return 0;
        }

        private UInt64? GetPatchId(XElement element)
        {
            var xmlId =
                from pid in element.Descendants("PatchId")
                select (string)pid;

            return xmlId.Any() ? Convert.ToUInt64(Enumerable.First(xmlId), 16) : (UInt64?)null;
        }

        public UInt64 ProgramId { get; private set; }
        public UInt64? PatchId { get; private set; }
    }

    public abstract class ContentMetaBase
    {
        public ContentMetaBase(string contentType)
        {
            ContentType = contentType;
        }
        public ContentMetaBase(XElement element)
        {
            Initialize(element);
            Version = GetVersion(element);
            ReleaseVersion = GetReleaseVersion(element);
            PrivateVersion = GetPrivateVersion(element);
            ContentMetaAttributes = GetContentMetaAttributes(element);
            ContentType = GetContentType(element);
            Id = GetId(element);
        }

        public static UInt32? GetVersionNullable(XElement element)
        {
            var xmlVersion =
                from version in element.Descendants("Version")
                select (string)version;
            return xmlVersion.Any() ?
                (UInt32?)Convert.ToUInt32(Enumerable.First(xmlVersion), 10) :
                null;
        }

        public static UInt32? GetReleaseVersionNullable(XElement element)
        {
            var xmlReleaseVersion =
                from version in element.Descendants("ReleaseVersion")
                select (string)version;
            return xmlReleaseVersion.Any() ?
                (UInt32?)Convert.ToUInt32(Enumerable.First(xmlReleaseVersion), 10) :
                null;
        }

        public static UInt32? GetPrivateVersionNullable(XElement element)
        {
            var xmlPrivateVersion =
                from version in element.Descendants("PrivateVersion")
                select (string)version;
            return xmlPrivateVersion.Any() ?
                (UInt32?)Convert.ToUInt32(Enumerable.First(xmlPrivateVersion), 10) :
                null;
        }

        private UInt32 GetVersion(XElement element)
        {
            return GetVersionNullable(element).GetValueOrDefault(0);
        }

        private UInt16 GetReleaseVersion(XElement element)
        {
            return (UInt16)GetReleaseVersionNullable(element).GetValueOrDefault(0);
        }

        private UInt16 GetPrivateVersion(XElement element)
        {
            return (UInt16)GetPrivateVersionNullable(element).GetValueOrDefault(0);
        }

        private Byte GetContentMetaAttributes(XElement element)
        {
            var list =
                from attribute in element.Elements("ContentMetaAttribute")
                select attribute.Value;

            return list.Any() ? (Byte)ContentMetaAttributeExtension.FromStringList(list) : (Byte)ContentMetaAttribute.None;
        }

        private string GetContentType(XElement element)
        {
            return element.Name.LocalName;
        }

        public UInt32 Version { get; private set; }
        public UInt16 ReleaseVersion { get; private set; }
        public UInt16 PrivateVersion { get; private set; }
        public Byte ContentMetaAttributes { get; private set; }
        public String ContentType { get; private set; }

        public UInt64 Id { get; protected set; }
        protected abstract UInt64 GetId(XElement element);

        protected virtual void Initialize(XElement element)
        {
        }

        private string GetExpandString(XElement element)
        {
            var expand = element.Attribute("UseEnvironmentVariable");
            if (expand != null && expand.Value.ToLower() == "true")
            {
                return Environment.ExpandEnvironmentVariables(element.Value);
            }
            return element.Value;
        }

        protected string GetValue(XElement element, string name)
        {
            var values = element.Descendants(name);
            if (values.Any())
            {
                return GetExpandString(values.First());
            }
            return null;
        }

        public abstract void MergeCoreInfo(CoreInfo CoreInfo);

        public virtual bool IsProgramType { get { return false; } }
    }

    public class GeneralMeta : ContentMetaBase
    {
        public GeneralMeta(XElement element) : base(element)
        {
        }

        protected override UInt64 GetId(XElement element)
        {
            var xmlId =
                from pid in element.Descendants("Id")
                select (string)pid;

            return xmlId.Any() ? Convert.ToUInt64(Enumerable.First(xmlId), 16) : 0;
        }
        public override void MergeCoreInfo(CoreInfo CoreInfo)
        {
            throw new NotImplementedException();
        }
    }

    public class AddOnContentMeta : ContentMetaBase
    {
        public AddOnContentMeta(XElement element) : base(element)
        {
        }

        protected override void Initialize(XElement element)
        {
            RequiredApplicationReleaseVersion = GetRequiredApplicationReleaseVersion(element);
            ApplicationId = GetApplicationId(element);
            Index = GetIndex(element);
            Tag = GetTag(element);
            DataPath = GetDataPath(element);
            FilterDescriptionFilePath = GetFilterDescriptionFilePath(element);
        }

        private UInt64 GetApplicationId(XElement element)
        {
            var xmlId =
                from pid in element.Descendants("ApplicationId")
                select (string)pid;

            return Convert.ToUInt64(Enumerable.First(xmlId), 16);
        }

        private UInt32? GetIndex(XElement element)
        {
            var xmlIndex =
                from aocIndex in element.Descendants("Index")
                select (string)aocIndex;

            var index = xmlIndex.Any() ? (UInt32?)Convert.ToUInt32(Enumerable.First(xmlIndex), 10) : null;
            if (index.HasValue && !(index >= 1 && index <= 2000))
            {
                throw new ArgumentException("'Index' should be specified in the range 1 to 2000.");
            }
            return index;
        }

        private UInt16 GetRequiredApplicationReleaseVersion(XElement element)
        {
            var xmlRequiredApplicationReleaseVersion =
                from requiredApplicationVersion in element.Descendants("RequiredApplicationReleaseVersion")
                select (string)requiredApplicationVersion;
            return xmlRequiredApplicationReleaseVersion.Any() ? Convert.ToUInt16(Enumerable.First(xmlRequiredApplicationReleaseVersion), 10) : (UInt16)0;
        }

        protected override UInt64 GetId(XElement element)
        {
            var xmlAddOnContentId =
                from addOnContentId in element.Descendants("Id")
                select (string)addOnContentId;

            // AddOnContentId 直接指定されている場合はそちらを優先
            if (xmlAddOnContentId.Any())
            {
                return Convert.ToUInt64(Enumerable.First(xmlAddOnContentId), 16);
            }

            return IdConverter.ConvertToAocBaseId(ApplicationId) + (Index.HasValue? Index.Value : 0);
        }

        private string GetDataPath(XElement element)
        {
            return GetValue(element, "DataPath");
        }

        private string GetFilterDescriptionFilePath(XElement element)
        {
            return GetValue(element, "FilterDescriptionFilePath");
        }

        private string GetTag(XElement element)
        {
            var tags =
                from tag in element.Descendants("Tag")
                select (string)tag;

            return tags.Any() ? tags.First() : null;
        }

        public UInt32? Index { get; private set; }
        public UInt64 ApplicationId { get; private set; }
        public UInt16 RequiredApplicationReleaseVersion { get; private set; }
        public string Tag { get; private set; }
        public string DataPath { get; private set; }

        public string FilterDescriptionFilePath { get; private set; }

        public override void MergeCoreInfo(CoreInfo CoreInfo)
        {
            throw new NotImplementedException();
        }
    }

    public class ApplicationMeta : ContentMetaBase
    {
        public ApplicationMeta() : base("Application")
        { }
        public ApplicationMeta(XElement element) : base(element)
        {
            RequiredSystemVersion = GetRequiredSystemVersion(element);
            MetaApplicationId = GetMetaApplicationId(element);
            ProgramIndex = GetProgramIndex(element);
        }

        private UInt32? GetRequiredSystemVersion(XElement element)
        {
            var nodes = element.XPathSelectElements("//RequiredSystemVersion");
            return nodes.Any() ? Convert.ToUInt32(nodes.First().Value, 10) : (UInt32?)null;
        }

        private string GetFilterDescriptionFilePath(XElement element)
        {
            return GetValue(element, "FilterDescriptionFilePath");
        }

        private UInt64? GetMetaApplicationId(XElement element)
        {
            var nodes = element.XPathSelectElements("//ApplicationId");
            return nodes.Any() ? (UInt64?)Convert.ToUInt64(nodes.First().Value, 16) : null;
        }

        private byte? GetProgramIndex(XElement element)
        {
            var nodes = element.XPathSelectElements("//ProgramIndex");
            return nodes.Any() ? (byte?)Convert.ToByte(nodes.First().Value, 10) : null;
        }

        protected override void Initialize(XElement element)
        {
            base.Initialize(element);
            FilterDescriptionFilePath = GetFilterDescriptionFilePath(element);
        }

        protected override UInt64 GetId(XElement element)
        {
            var xmlId =
                from pid in element.Descendants("Id")
                select (string)pid;

            return xmlId.Any() ? Convert.ToUInt64(Enumerable.First(xmlId), 16) : 0;
        }

        public UInt32? RequiredSystemVersion { get; private set; }

        public UInt64 PatchId { get; private set; }

        public override void MergeCoreInfo(CoreInfo CoreInfo)
        {
            Id = MetaApplicationId.HasValue ? MetaApplicationId.Value : CoreInfo.ProgramId;
            PatchId = CoreInfo.PatchId.HasValue ? CoreInfo.PatchId.Value : IdConverter.ConvertToPatchId(Id);
            ProgramId = CoreInfo.ProgramId;
        }

        public override bool IsProgramType { get { return true; }}

        public string FilterDescriptionFilePath { get; private set; }

        public UInt64? MetaApplicationId { get; private set; }

        public byte? ProgramIndex { get; private set; }

        public UInt64 ProgramId { get; private set; }
    }

    public class PatchMeta : ContentMetaBase
    {
        public PatchMeta() : base("Patch")
        { }
        public PatchMeta(XElement element) : base(element)
        {
            RequiredSystemVersion = GetRequiredSystemVersion(element);
            MetaApplicationId = GetMetaApplicationId(element);
            ProgramIndex = GetProgramIndex(element);
        }

        private UInt32? GetRequiredSystemVersion(XElement element)
        {
            var nodes = element.XPathSelectElements("//RequiredSystemVersion");
            return nodes.Any() ? Convert.ToUInt32(nodes.First().Value, 10) : (UInt32?)null;
        }

        protected override UInt64 GetId(XElement element)
        {
            var xmlId =
                from pid in element.Descendants("Id")
                select (string)pid;

            return xmlId.Any() ? Convert.ToUInt64(Enumerable.First(xmlId), 16) : 0;
        }

        private UInt64? GetMetaApplicationId(XElement element)
        {
            var nodes = element.XPathSelectElements("//ApplicationId");
            return nodes.Any() ? (UInt64?)Convert.ToUInt64(nodes.First().Value, 16) : null;
        }

        private byte? GetProgramIndex(XElement element)
        {
            var nodes = element.XPathSelectElements("//ProgramIndex");
            return nodes.Any() ? (byte?)Convert.ToByte(nodes.First().Value, 10) : null;
        }

        public UInt32? RequiredSystemVersion { get; private set; }

        public UInt64 ApplicationId { get; private set; }

        public UInt64 ProgramId { get; private set; }

        public UInt64? MetaApplicationId { get; private set; }

        public byte? ProgramIndex { get; private set; }

        public override void MergeCoreInfo(CoreInfo coreInfo)
        {
            ApplicationId = MetaApplicationId.HasValue ? MetaApplicationId.Value : coreInfo.ProgramId;
            ProgramId = coreInfo.ProgramId;

            if(coreInfo.PatchId.HasValue)
            {
                Id = coreInfo.PatchId.Value;
            }
            else
            {
                Id = IdConverter.ConvertToPatchId(ApplicationId);
            }
        }
        public override bool IsProgramType { get { return true; }}
    }

    public class SystemUpdateMeta : ContentMetaBase
    {
        public SystemUpdateMeta(XElement element) : base(element)
        {
            Version = GetVersion(element);
            ReleaseVersion = GetReleaseVersion(element);
            PrivateVersion = GetPrivateVersion(element);
        }

        protected override UInt64 GetId(XElement element)
        {
            var id = element.XPathSelectElement("//SystemUpdate/ContentMeta/Id").Value;
            return Convert.ToUInt64(id, 16);
        }

        private UInt32 GetVersion(XElement element)
        {
            var versions = element.XPathSelectElements("//SystemUpdate/ContentMeta/Version");
            return versions.Any() ? Convert.ToUInt32(versions.First().Value, 10) : 0;
        }

        private UInt16 GetReleaseVersion(XElement element)
        {
            var versions = element.XPathSelectElements("//SystemUpdate/ContentMeta/ReleaseVersion");
            return versions.Any() ? Convert.ToUInt16(versions.First().Value, 10) : (UInt16)0;
        }

        private UInt16 GetPrivateVersion(XElement element)
        {
            var versions = element.XPathSelectElements("//SystemUpdate/ContentMeta/PrivateVersion");
            return versions.Any() ? Convert.ToUInt16(versions.First().Value, 10) : (UInt16)0;
        }

        public override void MergeCoreInfo(CoreInfo CoreInfo)
        {
            throw new NotImplementedException();
        }

        public new UInt32 Version { get; private set; }
        public new UInt16 ReleaseVersion { get; private set; }
        public new UInt16 PrivateVersion { get; private set; }
    }


    [XmlRoot("Core", IsNullable = false)]
    public class CoreModel
    {
        [XmlElement("ProgramId")]
        public string ProgramId { get; set; }
        [XmlElement("ApplicationId")]
        public string ApplicationId { get; set; }

        public void AreValidValues()
        {
            if (string.IsNullOrEmpty(ProgramId) && string.IsNullOrEmpty(ApplicationId))
            {
                throw new ArgumentException("<ProgramId> or <ApplicationId> should be specified in <Core>");
            }
            if (ProgramId != null && ApplicationId != null)
            {
                throw new ArgumentException("<ProgramId> should not be specified with <ApplicationId>");
            }

            if (ProgramId != null)
            {
                IdConverter.ConvertStringToId(ProgramId, "ProgramId", "Core");
            }
            else
            {
                IdConverter.ConvertStringToId(ApplicationId, "ApplicationId", "Core");
            }
        }
    }

    [XmlRoot("AddOnContent", IsNullable = false)]
    public class AddOnContentModel
    {
        [XmlElement("Id")]
        public string Id { get; set; }

        [XmlElement("Index")]
        public Int16? Index { get; set; }

        [XmlElement("RequiredApplicationVersion")]
        public Int32 RequiredApplicationVersion { get; set; }

        [XmlElement("RequiredApplicationReleaseVersion")]
        public UInt16 RequiredApplicationReleaseVersion { get; set; }

        [XmlElement("RequiredApplicationPrivateVersion")]
        public Int16 RequiredApplicationPrivateVersion { get; set; }

        [XmlElement("Version")]
        public Int32 Version { get; set; }

        [XmlElement("ReleaseVersion")]
        public UInt16 ReleaseVersion { get; set; }

        [XmlElement("PrivateVersion")]
        public Int16 PrivateVersion { get; set; }

        [XmlElement("ApplicationId")]
        public string ApplicationId { get; set; }

        [XmlElement("Tag")]
        public string Tag { get; set; }

        public void AreValidValues()
        {
            if (string.IsNullOrEmpty(ApplicationId))
            {
                throw new ArgumentException("<ApplicationId> should be specified in <AddOnContent>");
            }
            if (Index != null && !string.IsNullOrEmpty(Id))
            {
                throw new ArgumentException("<Index> should not be specified with <Id>");
            }

            IdConverter.ConvertStringToId(ApplicationId, "ApplicationId", "AddOnContent");
            if (!string.IsNullOrEmpty(Id))
            {
                IdConverter.ConvertStringToId(Id, "Id", "AddOnContent");
            }
        }
    }

    [XmlRoot("Patch", IsNullable = false)]
    public class PatchModel
    {
        [XmlElement("ApplicationId")]
        public string ApplicationId { get; set; }

        public void AreValidValues()
        {
            if (ApplicationId == null || ApplicationId == string.Empty)
            {
                throw new ArgumentException("<ApplicationId> should be specified in <PatchModel>");
            }

            IdConverter.ConvertStringToId(ApplicationId, "ApplicationId", "PatchModel");
        }
    }

    [XmlRoot("SystemData", IsNullable = false)]
    public class SystemDataModel
    {
        [XmlElement("Id")]
        public string Id { get; set; }

        [XmlElement("Name")]
        public string Name { get; set; }

        [XmlElement("ContentMetaAttribute")]
        public List<string> ContentMetaAttribute { get; set; }

        [XmlElement("Version")]
        public Int32? Version { get; set; }

        [XmlElement("ReleaseVersion")]
        public UInt16? ReleaseVersion { get; set; }

        [XmlElement("PrivateVersion")]
        public Int16? PrivateVersion { get; set; }

        public void AreValidValues()
        {
            IdConverter.ConvertStringToId(Id, "Id", "SystemData");

            if (Version != null && (ReleaseVersion != null || PrivateVersion != null))
            {
                throw new ArgumentException("<Version> cannot be specified with <ReleaseVersion> and <PrivateVersion>.");
            }
        }
    }

    [XmlRoot("Delta", IsNullable = false)]
    public class DeltaModel
    {
        [XmlElement("Id")]
        public string Id { get; set; }

        [XmlElement("Version")]
        public UInt32? Version { get; set; }

        [XmlElement("ReleaseVersion")]
        public UInt16? ReleaseVersion { get; set; }

        [XmlElement("PrivateVersion")]
        public Int16? PrivateVersion { get; set; }

        public void AreValidValues()
        {
            IdConverter.ConvertStringToId(Id, "Id", "Delta");

            if (Version != null && (ReleaseVersion != null || PrivateVersion != null))
            {
                throw new ArgumentException("<Version> cannot be specified with <ReleaseVersion> and <PrivateVersion>.");
            }
        }
    }


    public class MetaFileReader
    {
        const string ApplicationDirective = "Application";
        const string AddOnContentDirective = "AddOnContent";
        const string SystemProgramDirective = "SystemProgram";
        const string SystemDataDirective = "SystemData";
        const string SystemUpdateDirective = "SystemUpdate";
        const string PatchDirective = "Patch";
        const string CoreDirective = "Core";
        const string DeltaDirective = "Delta";

        const string ApplicationId = "ApplicationId";
        const string PatchId = "PatchId";
        const string ProgramId = "ProgramId";
        const string SystemProgramId = "SystemProgramId";

        const string RequiredDownloadSystemVersion = "RequiredDownloadSystemVersion";
        const string CardSpec = "CardSpec";
        const string KeyGeneration = "KeyGeneration";
        const string KeepGeneration = "KeepGeneration";

        string[] ProgramDirectives =
        {
            ApplicationDirective,
            SystemProgramDirective,
        };

        string[] ContentTypeDirectives =
        {
            ApplicationDirective,
            AddOnContentDirective,
            SystemDataDirective,
            SystemProgramDirective,
            SystemUpdateDirective,
            PatchDirective,
            DeltaDirective,
        };

        string[] NotContentMetaDirectives =
        {
            RequiredDownloadSystemVersion,
            CardSpec,
            KeyGeneration,
            KeepGeneration,
        };

        private bool IsProgramContent(string contentType)
        {
            return contentType == "Application" || contentType == "SystemProgram";
        }

        private bool IsPatchContent(string contentType)
        {
            return contentType == "Patch";
        }

        public void CreateContentMetaList(string xmlPath, string contentType)
        {
            var list = new List<ContentMetaBase>();

            var document = XDocument.Load(xmlPath);
            CoreInfo core = null;
            var metaElements = document.Descendants("Meta").Elements();
            if (metaElements.Count() == 0)
            {
                metaElements = document.Descendants("NintendoSdkMeta").Elements();
            }
            foreach (var d in metaElements)
            {
                var directiveName = d.Name.LocalName;
                switch (directiveName)
                {
                    case CoreDirective:
                        {
                            core = new CoreInfo(d);
                        }
                        break;
                    case AddOnContentDirective:
                        {
                            var meta = new AddOnContentMeta(d);
                            list.Add(meta);
                        }
                        break;
                    case ApplicationDirective:
                        {
                            if(IsProgramContent(contentType))
                            {
                                var meta = new ApplicationMeta(d);
                                list.Add(meta);
                            }
                            else if(IsPatchContent(contentType))
                            {
                                var meta = new PatchMeta(d);
                                list.Add(meta);
                            }
                        }
                        break;
                    case PatchDirective:
                        {
                            var meta = new PatchMeta(d);
                            list.Add(meta);
                        }
                        break;
                    case SystemUpdateDirective:
                        {
                            var meta = new SystemUpdateMeta(d);
                            list.Add(meta);
                        }
                        break;

                    default:
                        {
                            if(ContentTypeDirectives.Contains(directiveName))
                            {
                                var meta = new GeneralMeta(d);
                                list.Add(meta);
                            }
                            else if(NotContentMetaDirectives.Contains(directiveName))
                            {
                                break;
                            }
                            else
                            {
                                throw new ArgumentException(string.Format("Invalid directive '{0}'", directiveName));
                            }
                        }
                        break;
                }
            }

            // Core がある場合は、プログラム系に値をマージ
            if(core != null && (IsProgramContent(contentType) || IsPatchContent(contentType)))
            {
                var programs = list.Where(item => item.IsProgramType);
                if(programs.Any())
                {
                    programs.First().MergeCoreInfo(core);
                }
                else
                {
                    ContentMetaBase program;
                    if(IsProgramContent(contentType))
                    {
                        program = new ApplicationMeta();
                        program.MergeCoreInfo(core);
                    }
                    else if(IsPatchContent(contentType))
                    {
                        program = new PatchMeta();
                        program.MergeCoreInfo(core);
                    }
                    else
                    {
                        throw new ArgumentException(String.Format("Invalid content meta type: {0}", contentType));
                    }
                    list.Add(program);
                }
            }

            // INFO:
            // プログラム系と追加コンテンツ両方存在する場合は追加コンテンツを優先
            // テストデータのメタでこのようになっている箇所がある。(testNs_Data)
            // 本来はエラーにするべきだがビルドシステム側の修正等も必要で手間がかかるのでとりあえずこれで。
            if(list.Any(item => item.IsProgramType) && list.Any(item => item.GetType() == typeof(AddOnContentMeta)))
            {
                list.RemoveAll(item => item.IsProgramType);
            }

            if(list.Count == 0)
            {
                throw new ArgumentException("Invalid meta file");
            }

            m_ContentMetaList = list;
        }

        public IEnumerable<ContentMetaBase> GetContentMetaList()
        {
            return m_ContentMetaList;
        }

        const string IdNotFoundMessage = "メタファイルに ID が記述されていません";

        public MetaFileReader(string xmlPath, string contentType)
        {
            m_Document = XDocument.Load(xmlPath);
            CreateContentMetaList(xmlPath, contentType);
        }

        public UInt32? GetRequiredDownloadSystemVersion()
        {
            var requiredDownloadSystemVersion =
                from meta in m_Document.Descendants("RequiredDownloadSystemVersion")
                select (string)meta;
            return requiredDownloadSystemVersion.Any() ?
                (UInt32?) Convert.ToUInt32(Enumerable.First(requiredDownloadSystemVersion), 10) :
                null;
        }

        public UInt32? GetVersion()
        {
            var xmlVersion = ContentMetaBase.GetVersionNullable(m_Document.Root);
            var xmlReleaseVersion = ContentMetaBase.GetReleaseVersionNullable(m_Document.Root);
            var xmlPrivateVersion = ContentMetaBase.GetPrivateVersionNullable(m_Document.Root);

            if (xmlVersion != null || xmlReleaseVersion != null || xmlPrivateVersion != null)
            {
                return NintendoContentMetaBase.MakeVersion(xmlVersion.GetValueOrDefault(0),
                                                           (UInt16)xmlReleaseVersion.GetValueOrDefault(0),
                                                           (UInt16)xmlPrivateVersion.GetValueOrDefault(0)); ;
            }
            else
            {
                return null;
            }
        }

        public Tuple<int, int, byte?> GetCardSpec()
        {
            var cardSize =
                from meta in m_Document.Descendants("CardSpec")
                select (string)meta.Element("Size");
            var cardClockRate =
                from meta in m_Document.Descendants("CardSpec")
                select (string)meta.Element("ClockRate");
            var cardLaunchFlags =
                from meta in m_Document.Descendants("CardSpec")
                select (string)meta.Element("LaunchFlags");

            var size = cardSize.Any() ? Convert.ToInt32(Enumerable.First(cardSize), 10) : 0;
            var clockRate = cardClockRate.Any() ? Convert.ToInt32(Enumerable.First(cardClockRate), 10) : 0;
            var launchFlags = cardLaunchFlags.Any() ? (byte?) Convert.ToByte(Enumerable.First(cardLaunchFlags), 10) : null;
            if (launchFlags.HasValue && launchFlags.Value == 0x0)
            {
                launchFlags = null;
            }
            return Tuple.Create(size, clockRate, launchFlags);
        }

        public bool? GetKeepGeneration()
        {
            var keepGeneration =
                from meta in m_Document.Descendants("KeepGeneration")
                select (string)meta;
            return keepGeneration.Any() ?
                (bool?) Convert.ToBoolean(Enumerable.First(keepGeneration)) : null;
        }

        private XDocument m_Document;

        private IEnumerable<ContentMetaBase> m_ContentMetaList;

        public void ValidateMetaFile(string metaType)
        {
            switch (metaType)
            {
                case NintendoContentMetaConstant.ContentMetaTypeApplication:
                    {
                        var coreModel = GetDeserializedModel<CoreModel>(CoreDirective, false, true);
                        coreModel.AreValidValues();

                        var appModel = GetDeserializedModel<ApplicationControlPropertyModel>(ApplicationDirective, true, false);
                        if (appModel != null)
                        {
                            appModel.AreValidValues();
                        }
                    }
                    break;
                case NintendoContentMetaConstant.ContentMetaTypeAddOnContent:
                    {
                        var aocModel = GetDeserializedModel<AddOnContentModel>(AddOnContentDirective, false, false);
                        aocModel.AreValidValues();
                    }
                    break;
                case NintendoContentMetaConstant.ContentMetaTypePatch:
                    {
                        var coreModel = GetDeserializedModel<CoreModel>(CoreDirective, false, true);
                        coreModel.AreValidValues();

                        var appModel = GetDeserializedModel<ApplicationControlPropertyModel>(ApplicationDirective, true, false);
                        if (appModel != null)
                        {
                            appModel.AreValidValues();
                        }

                        var patchModel = GetDeserializedModel<PatchModel>(PatchDirective, false, false);
                        patchModel.AreValidValues();
                    }
                    break;
                case NintendoContentMetaConstant.ContentMetaTypeSystemUpdate:
                    {
                        var contentMetaModel = GetDeserializedModel<SystemUpdateModel>(SystemUpdateDirective, false, false);
                        contentMetaModel.AreValidValues();
                    }
                    break;
                case NintendoContentMetaConstant.ContentMetaTypeSystemData:
                    {
                        var systemDataModel = GetDeserializedModel<SystemDataModel>(SystemDataDirective, false, false);
                        systemDataModel.AreValidValues();
                    }
                    break;
                case NintendoContentMetaConstant.ContentMetaTypeSystemProgram:
                    {
                        var coreModel = GetDeserializedModel<CoreModel>(CoreDirective, false, true);
                        coreModel.AreValidValues();
                    }
                    break;
                case NintendoContentMetaConstant.ContentMetaTypeDelta:
                    {
                        var deltaModel = GetDeserializedModel<DeltaModel>(DeltaDirective, false, false);
                        deltaModel.AreValidValues();
                    }
                    break;
                default: throw new ArgumentException("'{0}' is not supported ContentMetaType.", metaType);
            }
        }


        private ModelT GetDeserializedModel<ModelT>(string elementName, bool isNullable, bool acceptUnknownTags)
        {
            var node = m_Document.Descendants(elementName);
            if (!node.Any())
            {
                if (isNullable)
                {
                    return default(ModelT);
                }
                else
                {
                    throw new ArgumentException(string.Format("<{0}> not found.", elementName));
                }
            }

            if (node.Count() > 1)
            {
                throw new ArgumentException(string.Format("<{0}> is duplicated.", elementName));
            }

            var reader = node.First().CreateReader();
            var serializer = new XmlSerializer(typeof(ModelT));
            if (!acceptUnknownTags)
            {
                serializer.UnknownElement += XmlUnknownElementHandler.DeserializationHandler;
            }
            return (ModelT)serializer.Deserialize(reader);
        }

        public static List<T> ReadMetaModelList<T>(string metaPath, string nodeName)
        {
            var document = new XmlDocument();
            document.Load(metaPath);
            var nodes = document.SelectNodes(nodeName);

            if (nodes == null)
            {
                return null;
            }

            var modelList = new List<T>();
            foreach (XmlNode node in nodes)
            {
                var reader = XmlReader.Create(new StringReader(node.OuterXml));
                XmlSerializer serializer = new System.Xml.Serialization.XmlSerializer(typeof(T));
                modelList.Add((T)serializer.Deserialize(reader));
            }

            return modelList;
        }

        public static T ReadMetaModel<T>(string metaPath, string nodeName)
        {
            var modelList = ReadMetaModelList<T>(metaPath, nodeName);

            if (modelList == null || modelList.Count() == 0)
            {
                return default(T);
            }

            return modelList.First();
        }

        public static void VerifyCoreModel(string metaFilePath)
        {
            if (metaFilePath == null)
            {
                return;
            }

            var coreModel = ReadMetaModel<CoreModel>(metaFilePath, "//Core");

            if (coreModel == null)
            {
                return;
            }
            var coreFsModel = ReadMetaModel<FsAccessControlData>(metaFilePath, "//Core//FsAccessControlData");

            UnpublishableErrorCheckData checkData = new UnpublishableErrorCheckData();
            checkData.Nmeta.Core = coreModel;
            checkData.Nmeta.CoreFsAccessControlData = coreFsModel;

            UnpublishableError.VerifyNmetaUnpublishableError(checkData, UnpublishableErrorCheck.CoreCheckList);
        }

        public static void VerifyAocModel(string metaFilePath)
        {
            if (metaFilePath == null)
            {
                return;
            }

            var aocModel = ReadMetaModelList<AddOnContentModel>(metaFilePath, "//AddOnContent");

            if (aocModel == null)
            {
                return;
            }

            var checkDataList = aocModel.Select(entry => new UnpublishableErrorCheckData() { Nmeta = { AddOnContent = entry, AddOnContentList = aocModel } }).ToList();
            UnpublishableErrorCheckData checkData = new UnpublishableErrorCheckData();
            checkData.Nmeta.AddOnContentList = aocModel;

            UnpublishableError.VerifyNmetaUnpublishableError(checkDataList, UnpublishableErrorCheck.AddOnContentCheckList);
        }
    }
}
