﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Text.RegularExpressions;
using YamlDotNet.RepresentationModel;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    using EntryFilterRule = Pair<FilterType, string>;
    using EntryFilterRuleRegex = Pair<FilterType, System.Text.RegularExpressions.Regex>;

    public enum FilterType
    {
        None, Remove, Exception, AbsoluteAdd
    }

    public class Pair<TypeA, TypeB>
    {
        public TypeA first;
        public TypeB second;

        public Pair(TypeA a, TypeB b)
        {
            first = a;
            second = b;
        }
    }

    public class FilterDescription
    {
        private static string ToolWorkDirEnvStr = "AUTHORING_TOOL_WORK_DIR";
        private static Regex RegexKeywordEnv = new Regex("@<(?<envname>[^>]*)>");
        private static string ReplaceEnvironmentVariableKeyword(string path)
        {
            MatchEvaluator meval = new MatchEvaluator( (x) => {
                if (x.Groups["envname"].Value == ToolWorkDirEnvStr)
                {
                    return System.Environment.CurrentDirectory;
                }
                else
                {
                    return System.Environment.GetEnvironmentVariable(x.Groups["envname"].Value);
                }
            });
            return RegexKeywordEnv.Replace(path, meval);
        }

        public static string FormatFilePathDelimiterForFilter(string target)
        {
            return target.Replace(@"/", @"\");
        }

        private static string FormatFilePathDelimiterForFilterRegex(string target)
        {
            return target.Replace(@"/", @"\\");
        }

        private static string NormalizePathForUri(string path)
        {
            path = Path.GetFullPath(path);
            if ((path.EndsWith("/") == false) && (path.EndsWith(@"\") == false))
            {
                path += @"\";
            }
            return path;
        }

        public static string FormatFilePathForFilter(string target, string rootDir)
        {
            string targetRelative = target;

            // 絶対パスから起点のディレクトリに対する相対パスを得る
            if (!string.IsNullOrEmpty(rootDir))
            {
                Uri rootDirUri = new Uri(NormalizePathForUri(rootDir));
                Uri targetUri = new Uri(NormalizePathForUri(target));
                if (rootDirUri.Scheme == targetUri.Scheme)
                {
                    Uri relativeUri = rootDirUri.MakeRelativeUri(targetUri);
                    targetRelative = Uri.UnescapeDataString(relativeUri.ToString());
                }
            }
            return FormatFilePathDelimiterForFilter(targetRelative).TrimEnd('\\');
        }

        public static bool IsEntryInFilterList(string target, string rootDir, List<EntryFilterRuleRegex> filterRuleRegexList)
        {
            return IsEntryInFilterList(FormatFilePathForFilter(target, rootDir), filterRuleRegexList);
        }

        public static List<EntryFilterRuleRegex> ConvertFilterRuleStringToRegex(List<EntryFilterRule> filterRuleList)
        {
            if (filterRuleList == null)
            {
                return null;
            }
            var filterRuleRegexList = new List<EntryFilterRuleRegex>();
            foreach (var filter in filterRuleList)
            {
                var rule = FormatFilePathDelimiterForFilterRegex(filter.second);
                Regex rex = new Regex(rule, RegexOptions.Compiled);
                filterRuleRegexList.Add(new EntryFilterRuleRegex(filter.first, rex));
            }
            return filterRuleRegexList;
        }

        public static bool IsEntryInFilterList(string target, List<EntryFilterRuleRegex> filterRuleRegexList)
        {
            if (filterRuleRegexList == null)
            {
                return false;
            }

            bool shouldRemove = false;
            foreach (var filter in filterRuleRegexList)
            {
                if (shouldRemove && filter.first == FilterType.Exception)
                {
                    if (filter.second.IsMatch(target))
                    {
                        shouldRemove = false;
                    }
                }
                else if (!shouldRemove && filter.first == FilterType.Remove)
                {
                    if (filter.second.IsMatch(target))
                    {
                        shouldRemove = true;
                    }
                }
            }
            return shouldRemove;
        }

        private static Regex RegexKeywordInclude = new Regex("#include *\"(?<path>[^\"]*)\"");
        private static List<EntryFilterRule> ParseFdfRecursively(string fdfPath, ref List<string> fdfPathList)
        {
            // パスの処理：相対パスの場合は fdf からの相対パスにする
            fdfPath = FormatFilePathDelimiterForFilter(fdfPath);
            if (Path.IsPathRooted(fdfPath) == false)
            {
                if (fdfPathList.Count > 0)
                {
                    // 最後の要素は最後に見た fdf ファイルパス（絶対パス）が入っている
                    fdfPath = Path.Combine(Path.GetDirectoryName(fdfPathList[fdfPathList.Count - 1]), fdfPath);
                }
            }
            fdfPath = Path.GetFullPath(fdfPath).TrimEnd('\\');

            // 再帰処理：インクルード済みだったら抜ける
            if (0 <= fdfPathList.FindIndex(path => String.Compare(path, fdfPath, StringComparison.InvariantCultureIgnoreCase) == 0))
            {
                return new List<EntryFilterRule>();
            }
            fdfPathList.Add(fdfPath);

            // ファイルに関する例外はそのまま外に出す（特にハンドリングしない）
            var filterList = new List<EntryFilterRule>();
            using (var file = new StreamReader(fdfPath))
            {
                bool isForgiveAbsolute = true;
                while (file.EndOfStream == false)
                {
                    // 空行、コメント行はスキップ
                    string line = file.ReadLine();
                    if (String.IsNullOrEmpty(line) || line.StartsWith(";"))
                    {
                        continue;
                    }

                    // 環境変数を置換
                    line = ReplaceEnvironmentVariableKeyword(line);

                    // fdf ファイルから fdf ファイルをインクルードする（#include "FDF_PATH"）
                    var match = RegexKeywordInclude.Match(line);
                    if (match.Success)
                    {
                        string includePath = match.Groups["path"].Value;
                        var filters = ParseFdfRecursively(includePath, ref fdfPathList);
                        if (filters == null)
                        {
                            // インクルードしたフィルタが空であればスキップ
                            continue;
                        }

                        filterList.AddRange(filters);
                    }

                    // 一文字目で処理を分ける（フォーマットとしてスペース等は許容しない）
                    var filterType = FilterType.None;
                    var pathStartPos = 1;
                    switch (line[0])
                    {
                        case '-':
                            filterType = FilterType.Remove;
                            isForgiveAbsolute = false;
                            break;
                        case '+':
                            filterType = FilterType.Exception;
                            isForgiveAbsolute = false;
                            break;
                        case '*':
                            if (line[1] == '+')
                            {
                                if (!isForgiveAbsolute)
                                {
                                    throw new ArgumentException("You can NOT use '+' or '-' before '*+'.");
                                }
                                filterType = FilterType.AbsoluteAdd;
                                pathStartPos = 2;
                                break;
                            }
                            continue;
                        default:
                            // TODO: ("Warning: failed to parse \"" + line + "\"");　ライブラリでのログはどうしている？
                            continue;
                    }

                    // 接頭辞以降の文字列に対し、前後のスペースをカットしてダブルクォートを削除する
                    string rule = line.Substring(pathStartPos).Trim().Trim('"');

                    filterList.Add(new EntryFilterRule(filterType, rule));
                }
                if(filterList.Count == 0)
                {
                    return null;
                }

                return filterList;
            }
        }
        public static List<EntryFilterRule> ParseFdf(string fdfPath)
        {
            var fdfPathList = new List<string>();

            // null だけチェックする
            if (fdfPath == null)
            {
                return null;
            }

            return ParseFdfRecursively(fdfPath, ref fdfPathList);
        }
    }
}
