﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.IO.Compression;
using System.Xml;
using System.Xml.Serialization;
using System.Linq;
using Nintendo.ApplicationControlProperty;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    using IconInfo = Tuple<string, string>;
    using IconInfoList = List<Tuple<string, string>>;

    public class ApplicationControl
    {
        private const string HtmlDocumentDirName = "html-document";
        private const string AccessibleUrlsDirName = "accessible-urls";

        public static string GetIconName(string language)
        {
            return string.Format("icon_" + language + ".dat");
        }

        public static string GetRootIconName(string language, bool isNxIcon)
        {
            return string.Format("{0}.{1}.jpg", isNxIcon ? "nx" : "raw", language);
        }

        public static void Generate(string metaPath, IconInfoList iconPathList, IconInfoList nxIconPathList, UInt32 nxIconMaxSize, string outputDirectoryPath, bool cleanup = true)
        {
            if(Directory.Exists(outputDirectoryPath))
            {
                if (cleanup)
                {
                    Directory.Delete(outputDirectoryPath, true);
                    // Directory.Delete は非同期なので、削除されるまで待つ
                    const int WaitForDeleteDirectoryTimeOutMilliSec = 1000 * 3;
                    const int WaitForDeleteDirectoryWaitUnitMilliSec = 100;
                    for (int waitMilliSec = 0; waitMilliSec < WaitForDeleteDirectoryTimeOutMilliSec; waitMilliSec += WaitForDeleteDirectoryWaitUnitMilliSec)
                    {
                        if (Directory.Exists(outputDirectoryPath) == false)
                        {
                            break;
                        }
                        System.Threading.Thread.Sleep(WaitForDeleteDirectoryWaitUnitMilliSec);
                    }
                    Directory.CreateDirectory(outputDirectoryPath);
                }
            }
            else
            {
                Directory.CreateDirectory(outputDirectoryPath);
            }

            using (FileStream fs = new FileStream(outputDirectoryPath + "/control.nacp", FileMode.Create, FileAccess.Write))
            {
                var bytes = MakeApplicationControlPropertyBytes(metaPath);
                fs.Write(bytes, 0, bytes.Length);
            }

            var mergedIconPathMap = IconConverter.GetMergedIconPathMap(iconPathList, nxIconPathList);

            foreach (var kv in mergedIconPathMap)
            {
                var language = kv.Key;
                var path = kv.Value.Item1;
                var nxIconPath = kv.Value.Item2;

                try
                {
                    Enum.Parse(typeof(Language), language);
                }
                catch
                {
                    throw new ArgumentException(string.Format("Undefined language '{0}' specified.", language));
                }

                var iconData = IconConverter.ConvertNxIcon(path, nxIconPath, nxIconMaxSize);
                using (var fs = File.OpenWrite(outputDirectoryPath + "/" + GetIconName(language)))
                {
                    fs.Write(iconData, 0, iconData.Length);
                }
            }
        }

        public static List<string> UpdateNintendoSubmissionPackageContentInfo(ref NintendoSubmissionPackageContentInfo info)
        {
            if (!HasApplicationControlData(info))
            {
                return new List<string>();
            }

            var model = ReadApplicationControlPropertyModel(info.MetaFilePath);
            var iconPathList = info.IconList;
            var nxIconPathList = info.NxIconList;
            var pathInfo = new ApplicationControlPropertyPathInfo(model, info.MetaFilePath);
            pathInfo.LoadPathInfo();

            SetupIconPathList(ref iconPathList, ref nxIconPathList, pathInfo);
            info.IconList = iconPathList;
            info.NxIconList = nxIconPathList;

            if (!string.IsNullOrEmpty(pathInfo.HtmlDocumentDirPath))
            {
                ReplaceContentResourceInputPath(ref info, "HtmlDocument", pathInfo.HtmlDocumentDirPath, HtmlDocumentDirName);
            }
            if (!string.IsNullOrEmpty(pathInfo.LegalInformationDirPath))
            {
                ReplaceContentResourceInputPath(ref info, "LegalInformation", pathInfo.LegalInformationDirPath);
            }
            if (!string.IsNullOrEmpty(pathInfo.AccessibleUrlsDirPath))
            {
                ReplaceContentResourceInputPath(ref info, "HtmlDocument", pathInfo.AccessibleUrlsDirPath, AccessibleUrlsDirName);
            }
            return pathInfo.DisposableDirs;
        }

        public static ApplicationControlPropertyPathInfo LoadApplicationControlPropertyPathInfo(string metaFilePath)
        {
            var model = ReadApplicationControlPropertyModel(metaFilePath);
            var pathInfo = new ApplicationControlPropertyPathInfo(model, metaFilePath);
            pathInfo.LoadPathInfo();

            return pathInfo;
        }

        private static void UpdateApplicationControlPropertyIconList(ref ApplicationControlPropertyModel model, List<Tuple<string, string>> iconFileList)
        {
            // ApplicationControlPropertyModelにiconエントリがない場合で、引数に--iconの指定がある場合は--error-unpublishableチェック用のApplicationControlPropertyModelにiconを追加
            foreach (var icon in iconFileList)
            {
                if (model.Icon == null)
                {
                    model.Icon = new List<Icon>();
                }

                if (!model.Icon.Select(entry => entry.Language).Contains(icon.Item1))
                {
                    model.Icon.Add(new Icon() { Language = icon.Item1, IconPath = new DataPath() { Path = icon.Item2 } });
                }
            }
        }

        private static void UpdateApplicationControlProperty(ref ApplicationControlPropertyModel model, NintendoSubmissionPackageContentInfo info)
        {
            UpdateApplicationControlPropertyIconList(ref model, info.IconList);
        }

        private static void VerifyCacheStorageNotDuplicate(ApplicationControlPropertyModel model)
        {
            var cacheStorageAvailableSize = Convert.ToInt64(model.CacheStorageSize, 16);
            var cacheStorageJournalSize = Convert.ToInt64(model.CacheStorageJournalSize, 16);
            var cacheStorageDataAndJournalSizeMax = Convert.ToInt64(model.CacheStorageDataAndJournalSizeMax, 16);
            var cacheStorageIndexMax = Convert.ToInt16(model.CacheStorageIndexMax, 16);

            Action<string, long, long> checkCacheStorageSizeDescription = (targetSizeStr, targetSize, partnerSize) =>
            {
                if(partnerSize != 0 && targetSize == 0)
                {
                    throw new ArgumentException($"{targetSizeStr} is necessary.");
                }
            };

            if(cacheStorageAvailableSize != 0 || cacheStorageJournalSize != 0 )
            {
                if(cacheStorageDataAndJournalSizeMax != 0 || cacheStorageIndexMax != 0)
                {
                    throw new ArgumentException("Only either CacheStorageSize/CacheStorageJournalSize or CacheStorageDataAndJournalSizeMax/CacheStorageIndexMax can be set.");
                }
            }

            checkCacheStorageSizeDescription("CacheStorageAvailableSize", cacheStorageAvailableSize, cacheStorageJournalSize);
            checkCacheStorageSizeDescription("CacheStorageJournalSize", cacheStorageJournalSize, cacheStorageAvailableSize);
            checkCacheStorageSizeDescription("CacheStorageDataAndJournalSizeMax", cacheStorageDataAndJournalSizeMax, cacheStorageIndexMax);
            checkCacheStorageSizeDescription("CacheStorageIndexMax", cacheStorageIndexMax, cacheStorageDataAndJournalSizeMax);
        }

        private static void VerifySaveDataSize(ApplicationControlPropertyModel model)
        {
            const Int64 MinSaveDataSize = 16 * 1024 * 3;// 16KB を 3 ブロック が最小
            const Int64 MaxSaveDataSize = 4L * 1024 * 1024 * 1024 - 1;// 4GB - 1 が最大
            const Int32 MaxIndex = 0x8000;// int の正値が最大
            var accountSaveDataAvailableSize = Convert.ToInt64(model.UserAccountSaveDataSize, 16);
            var accountSaveDataJournalSize = Convert.ToInt64(model.UserAccountSaveDataJournalSize, 16);
            var deviceSaveDataAvailableSize = Convert.ToInt64(model.DeviceSaveDataSize, 16);
            var deviceSaveDataJournalSize = Convert.ToInt64(model.DeviceSaveDataJournalSize, 16);
            var bcatDeliveryCacheStorageAvailableSize = Convert.ToInt64(model.BcatDeliveryCacheStorageSize, 16);
            var cacheStorageAvailableSize = Convert.ToInt64(model.CacheStorageSize, 16);
            var cacheStorageJournalSize = Convert.ToInt64(model.CacheStorageJournalSize, 16);
            var temporaryStorageAvailableSize = Convert.ToInt64(model.TemporaryStorageSize, 16);
            var cacheStorageDataAndJournalSizeMax = Convert.ToInt64(model.CacheStorageDataAndJournalSizeMax, 16);
            var cacheStorageIndexMax = Convert.ToInt16(model.CacheStorageIndexMax, 16);

            Action<string, long, long, long, long> checkSaveDataSize = (saveDataStr, saveDataSize, totalSize, minSize, maxSize) =>
            {
                if (saveDataSize != 0)
                {
                    // 下限チェック
                    if (saveDataSize < minSize)
                    {
                        throw new ArgumentException($"{saveDataStr} is too small.It must be larger than 0x{Convert.ToString(minSize, 16)}.");
                    }
                    // 上限チェック
                    if (totalSize > maxSize)
                    {
                        throw new ArgumentException($"{saveDataStr} is too large.");
                    }
                }
            };

            Action<string, string, long, long> checkSaveDataSizeWithJournal = (saveDataStr, journalStr, saveDataSize, journalSize) =>
            {
                if (saveDataSize != 0 || journalSize != 0)
                {
                    if(journalStr == null)
                    {
                        const long BcatDeliveryCacheStorageJournalSize = 2 * 1024 * 1024;
                        journalSize = BcatDeliveryCacheStorageJournalSize;
                    }
                    // 下限チェック
                    if (saveDataSize < MinSaveDataSize)
                    {
                        throw new ArgumentException($"{saveDataStr} is too small.It must be larger than 0x{Convert.ToString(MinSaveDataSize, 16)}.");
                    }
                    if (journalSize < MinSaveDataSize)
                    {
                        throw new ArgumentException($"{journalStr} is too small.It must be larger than 0x{Convert.ToString(MinSaveDataSize, 16)}.");
                    }
                    // 上限チェック
                    if (SaveDataUtils.QuerySaveDataTotalSize(saveDataSize, journalSize) > MaxSaveDataSize)
                    {
                        if(journalStr == null)
                        {
                            throw new ArgumentException($"{saveDataStr} is too large.");
                        }
                        else
                        {
                            throw new ArgumentException($"{saveDataStr} or {journalStr} is too large.");
                        }
                    }
                }
            };
            checkSaveDataSizeWithJournal("UserAccountSaveDataSize", "UserAccountSaveDataJournalSize", accountSaveDataAvailableSize, accountSaveDataJournalSize);
            checkSaveDataSizeWithJournal("DeviceSaveDataSize", "DeviceSaveDataJournalSize", deviceSaveDataAvailableSize, deviceSaveDataJournalSize);
            checkSaveDataSizeWithJournal("BcatDeliveryCacheStorageSize", null, bcatDeliveryCacheStorageAvailableSize, 0);
            checkSaveDataSizeWithJournal("CacheStorageSize", "CacheStorageJournalSize", cacheStorageAvailableSize, cacheStorageJournalSize);
            var temporaryTotalSize = (temporaryStorageAvailableSize == 0) ? 0 : SaveDataUtils.QueryTemporarySaveDataTotalSize(temporaryStorageAvailableSize);
            checkSaveDataSize("TemporaryStorageSize", temporaryStorageAvailableSize, temporaryTotalSize, MinSaveDataSize, MaxSaveDataSize);
            checkSaveDataSize("CacheStorageDataAndJournalSizeMax", cacheStorageDataAndJournalSizeMax, cacheStorageDataAndJournalSizeMax, MinSaveDataSize * 2, MaxSaveDataSize);
            checkSaveDataSize("CacheStorageIndexMax", cacheStorageIndexMax, cacheStorageIndexMax, 0, MaxIndex);
        }

        private static void VerifyApplicationControlPropertyImpl(ApplicationControlPropertyModel applicationControlPropertyModel, CoreModel coreModel, string metaFilePath)
        {
            UnpublishableErrorCheckData checkData = new UnpublishableErrorCheckData();
            checkData.Nmeta.NmetaFilePath = metaFilePath;
            checkData.ApplicationControlProperty = applicationControlPropertyModel;
            checkData.Nmeta.Core = coreModel;

            UnpublishableError.VerifyNmetaUnpublishableError(checkData, UnpublishableErrorCheck.ApplicationControlPropertyCheckList);
            VerifyCacheStorageNotDuplicate(applicationControlPropertyModel);
            VerifySaveDataSize(applicationControlPropertyModel);
        }

        public static void VerifyApplicationControlProperty(string metaFilePath, List<Tuple<string, string>> iconFileList)
        {
            var model = ReadApplicationControlPropertyModel(metaFilePath);
            UpdateApplicationControlPropertyIconList(ref model, iconFileList);

            var coreModel = MetaFileReader.ReadMetaModel<CoreModel>(metaFilePath, "//Core");
            VerifyApplicationControlPropertyImpl(model, coreModel, metaFilePath);
        }

        public static void VerifyApplicationControlProperty(string metaFilePath)
        {
            var model = ReadApplicationControlPropertyModel(metaFilePath);
            var coreModel = MetaFileReader.ReadMetaModel<CoreModel>(metaFilePath, "//Core");
            VerifyApplicationControlPropertyImpl(model, coreModel, metaFilePath);
        }

        public static void VerifyApplicationControlProperty(NintendoSubmissionPackageContentInfo info)
        {
            if (!HasApplicationControlData(info))
            {
                return;
            }

            var model = ReadApplicationControlPropertyModel(info.MetaFilePath);
            UpdateApplicationControlProperty(ref model, info);

            var coreModel = MetaFileReader.ReadMetaModel<CoreModel>(info.MetaFilePath, "//Core");
            VerifyApplicationControlPropertyImpl(model, coreModel, info.MetaFilePath);
        }

        private static bool HasApplicationControlData(NintendoSubmissionPackageContentInfo info)
        {
            return info.MetaType == NintendoContentMetaConstant.ContentMetaTypeApplication || info.MetaType == NintendoContentMetaConstant.ContentMetaTypePatch;
        }

        private static void SetupIconPathList(ref IconInfoList iconPathList,  ref IconInfoList nxIconPathList, ApplicationControlPropertyPathInfo pathInfo)
        {
            if (pathInfo.IconList.Count > 0 || pathInfo.NxIconList.Count > 0)
            {
                if (iconPathList.Count() > 0 || nxIconPathList.Count() > 0)
                {
                    Log.Warning("Meta file has <Icon>. --icon and --nx-icon options are ignored.");
                }
                iconPathList = pathInfo.IconList;
                nxIconPathList = pathInfo.NxIconList;
            }
            else
            {
                List<string> languageList = iconPathList.Select(icon => icon.Item1).ToList();
                var missingIcons = nxIconPathList.Where(nxIcon => !languageList.Contains(nxIcon.Item1));
                if (missingIcons.Count() > 0)
                {
                    string missingLanguages = string.Concat(missingIcons.Select(nxIcon => string.Format("{0}, ", nxIcon.Item1))).TrimEnd(',', ' ');
                    throw new ArgumentException(string.Format("Languages ({0}) are specified by --nx-icons, but they are not specified by --icon option.", missingLanguages));
                }
            }
        }

        public static byte[] MakeApplicationControlPropertyBytes(string metaPath)
        {
            var model = ReadApplicationControlPropertyModel(metaPath);

            var metaFileReader = new MetaFileReader(metaPath, "Application");
            var meta = metaFileReader.GetContentMetaList().First();
            if(meta.GetType() == typeof(AddOnContentMeta))
            {
                var aocMeta = (AddOnContentMeta)meta;
                return model.MakePropertyBytes(aocMeta.ApplicationId, IdConverter.ConvertToAocBaseId(aocMeta.ApplicationId));
            }
            else if(meta.GetType() == typeof(ApplicationMeta))
            {
                var applicationMeta = (ApplicationMeta)meta;
                return model.MakePropertyBytes(applicationMeta.ProgramId, IdConverter.ConvertToAocBaseId(meta.Id));
            }
            else if(meta.GetType() == typeof(PatchMeta))
            {
                var patchMeta = (PatchMeta)meta;
                return model.MakePropertyBytes(patchMeta.ProgramId, IdConverter.ConvertToAocBaseId(patchMeta.ApplicationId));
            }
            else
            {
                return model.MakePropertyBytes(meta.Id, IdConverter.ConvertToAocBaseId(meta.Id));
            }
        }

        private static ApplicationControlPropertyModel ReadApplicationControlPropertyModel(string metaPath)
        {
            var document = new XmlDocument();
            document.Load(metaPath);
            var node = document.SelectSingleNode("//Application");
            var nodeString = node == null ? "<Application></Application>" : node.OuterXml;

            var reader = XmlReader.Create(new StringReader(nodeString));

            XmlSerializer serializer = new System.Xml.Serialization.XmlSerializer(typeof(ApplicationControlPropertyModel));
            serializer.UnknownElement += XmlUnknownElementHandler.DeserializationHandler;
            return (ApplicationControlPropertyModel)serializer.Deserialize(reader);
        }

        private static void ErrorForUnpublishable(string errorMsg)
        {
            if (GlobalSettings.ErrorUnpublishable)
            {
                throw new ArgumentException(errorMsg);
            }
            else
            {
                Log.Warning(errorMsg);
            }
        }

        private static void WarningForUnpublishable(string errorMsg)
        {
            if (GlobalSettings.ErrorUnpublishable)
            {
                Log.Warning(errorMsg);
            }
        }

        private static void ReplaceContentResourceInputPath(ref NintendoSubmissionPackageContentInfo info, string contentType, string inputDirPath, string outputDirPath = null)
        {
            inputDirPath = inputDirPath.Replace("\\", "/");
            var enumerator = info.ResourceList.Where(resource => resource.ContentType == contentType);
            bool found = false;
            if (enumerator.Count() > 0)
            {
                foreach (var resource in enumerator)
                {
                    var iDirConnector = resource.PathList.Where(path => path.second == outputDirPath);
                    if (iDirConnector.Count() == 1)
                    {
                        string replaceTarget = string.IsNullOrEmpty(outputDirPath) ? contentType : outputDirPath;
                        Log.Warning(string.Format("{0} path is replaced by meta file", replaceTarget));
                        var dirConnect = iDirConnector.First();
                        dirConnect.first = inputDirPath;
                        found = true;
                    }
                    else if (iDirConnector.Count() == 0)
                    {
                        resource.PathList.Add(new Pair<string, string>(inputDirPath, outputDirPath));
                        found = true;
                    }
                    else if (iDirConnector.Count() > 1)
                    {
                        throw new ArgumentException(string.Format("Resource is duplicated. ContentType='{0}', outputDirPath='{1}'", contentType, outputDirPath != null ? outputDirPath : ""));
                    }
                }
            }
            if (!found)
            {
                var resource = new NintendoSubmissionPackageContentResource();
                resource.ContentType = contentType;
                resource.PathList.Add(new Pair<string, string>(inputDirPath, outputDirPath));
                info.ResourceList.Add(resource);
            }
        }
    }

    public class ApplicationControlPropertyXmlSource : ISource
    {
        public long Size { get; set; }

        private  ISource m_source;

        public ApplicationControlPropertyXmlSource(ApplicationControlPropertyModel model)
        {
            byte[] buf;
            {
                var nameSpace = new XmlSerializerNamespaces();
                nameSpace.Add(String.Empty, String.Empty);
                using (var memoryStream = new MemoryStream())
                {
                    var sw = new StreamWriter(memoryStream, Encoding.UTF8);
                    var serializer = new XmlSerializer(typeof(ApplicationControlPropertyModel));
                    serializer.Serialize(sw, model, nameSpace);
                    buf = memoryStream.ToArray();
                }
            }

            m_source = new MemorySource(buf, 0, buf.Length);
            Size = m_source.Size;
        }

        public ByteData PullData(long offset, int size)
        {
            return m_source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
    }

    public class ApplicationControlPropertyPathInfo
    {
        public string HtmlDocumentDirPath { get; private set; }
        public string LegalInformationDirPath { get; private set; }
        public string AccessibleUrlsDirPath { get; private set; }
        public IconInfoList IconList;
        public IconInfoList NxIconList;

        public bool IsAccessibleUrlsSpecifiedFilePath { get; private set; }
        public bool IsLegalInformationSpecifiedFilePath { get; private set; }

        public List<string> DisposableDirs { get; private set; }

        public ApplicationControlPropertyPathInfo()
        {
            DisposableDirs = new List<string>();
        }

        public ApplicationControlPropertyPathInfo(ApplicationControlPropertyModel model, string metaFilePath)
        {
            m_Model = model;
            m_MetaFilePath = metaFilePath;
            IconList = new IconInfoList();
            NxIconList = new IconInfoList();
            DisposableDirs = new List<string>();
        }

        public void LoadPathInfo()
        {
            List<string> languageList = new List<string>();
            foreach (var item in m_Model.Icon)
            {
                if (languageList.Contains(item.Language))
                {
                    throw new ArgumentException(string.Format("Icon for Language '{0}' is duplicated.", item.Language));
                }
                languageList.Add(item.Language);

                string iconFullPath = GetFullPath(item.IconPath.Path, m_MetaFilePath);
                IconList.Add(new IconInfo(item.Language, iconFullPath));
                if (!string.IsNullOrEmpty(item.NxIconPath.Path))
                {
                    string nxIconFullPath = GetFullPath(item.NxIconPath.Path, m_MetaFilePath);
                    NxIconList.Add(new IconInfo(item.Language, nxIconFullPath));
                }
            }

            if (!string.IsNullOrEmpty(m_Model.HtmlDocumentPath.Path))
            {
                HtmlDocumentDirPath = GetFullPath(m_Model.HtmlDocumentPath.Path, m_MetaFilePath);
            }
            else
            {
                HtmlDocumentDirPath = null;
            }

            if (!string.IsNullOrEmpty(m_Model.LegalInformationFilePath.Path))
            {
                string extractDir = ExtractZipFile(GetFullPath(m_Model.LegalInformationFilePath.Path, m_MetaFilePath));
                LegalInformationDirPath = extractDir;
                DisposableDirs.Add(extractDir);
                IsLegalInformationSpecifiedFilePath = true;
            }
            else
            {
                LegalInformationDirPath = null;
            }

            if (!string.IsNullOrEmpty(m_Model.AccessibleUrlsFilePath.Path))
            {
                AccessibleUrlsDirPath = GenerateAccessibleUrlsDir(m_Model.AccessibleUrlsFilePath.Path, m_MetaFilePath);
                DisposableDirs.Add(AccessibleUrlsDirPath);
                IsAccessibleUrlsSpecifiedFilePath = true;
            }
        }

        public string ValidDirPath(string optionDirPath, string infoDirPath)
        {
            if (string.IsNullOrEmpty(infoDirPath))
            {
                return optionDirPath;
            }
            else
            {
                return infoDirPath;
            }
        }

        public IconInfoList ValidIconInfoList(IconInfoList optionIconList, IconInfoList infoIconList)
        {
            if (infoIconList == null || infoIconList.Count == 0)
            {
                return optionIconList;
            }
            else
            {
                return infoIconList;
            }
        }

        public static string GetFullPath(string path, string metaFilePath)
        {
            string metaFullPath = Path.GetFullPath(metaFilePath);
            string fullPath = Path.IsPathRooted(path) ? path : Path.GetFullPath(Path.Combine(Path.GetDirectoryName(metaFullPath), path));
            return fullPath.Replace("\\", "/");
        }

        private string ExtractZipFile(string fileName)
        {
            var extension = Path.GetExtension(fileName);
            if (extension.ToLower() != ".zip")
            {
                throw new ArgumentException("<LegalInformationFilePath> should be specfied zip file path.");
            }

            string tmpDirPath = Path.GetTempPath();
            string extractDirName = Path.Combine(tmpDirPath, Path.GetRandomFileName()).Replace("\\", "/").TrimEnd('/');

            ZipFile.ExtractToDirectory(fileName, extractDirName);

            return extractDirName;
        }

        private string GenerateAccessibleUrlsDir(string filePath, string metaFilePath)
        {
            string fullPath = GetFullPath(filePath, metaFilePath);

            string tmpDirPath = Path.GetTempPath();
            string outputDirPath = Path.Combine(tmpDirPath, Path.GetRandomFileName()).Replace("\\", "/").TrimEnd('/');

            Directory.CreateDirectory(outputDirPath);

            string dstFilePath = Path.Combine(outputDirPath, "accessible-urls.txt");

            // 改行コードを \n にして、コピーする
            var lines = File.ReadAllLines(fullPath);
            using (var sw = new StreamWriter(dstFilePath))
            {
                sw.NewLine = "\n";
                foreach (var line in lines)
                {
                    sw.WriteLine(line);
                }
            }

            return outputDirPath;
        }

        private ApplicationControlPropertyModel m_Model;
        private string m_MetaFilePath;
    }
}
