﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.Runtime.InteropServices;

namespace Nintendo.ApplicationControlProperty
{
    public enum Language
    {
        AmericanEnglish = 0,
        BritishEnglish,
        Japanese,
        French,
        German,
        LatinAmericanSpanish,
        Spanish,
        Italian,
        Dutch,
        CanadianFrench,
        Portuguese,
        Russian,
        Korean,
        TraditionalChinese,
        SimplifiedChinese
    }

    public enum StartupUserAccountValue : byte
    {
        None = (byte)0x0u,
        Required = (byte)0x1u,
        RequiredWithNetworkServiceAccountAvailable = (byte)0x2u
    }

    [Flags]
    public enum AttributeFlagValue : uint
    {
        None                        = 0x0u,
        Demo                        = 0x1u << 0,
        RetailInteractiveDisplay    = 0x1u << 1,
    }

    [Flags]
    public enum ParentalControlFlagValue : uint
    {
        None = 0x0u,
        FreeCommunication = 0x1u
    }

    public enum ScreenshotValue : byte
    {
        Allow = (byte)0x0u,
        Deny = (byte)0x1u,
    }

    public enum VideoCaptureValue : byte
    {
        Disable = (byte)0x0u,
        Manual  = (byte)0x1u,
        Enable  = (byte)0x2u,
    }

    public enum Organization
    {
        CERO = 0,
        GRACGCRB,
        GSRMR,
        ESRB,
        ClassInd,
        USK,
        PEGI,
        PEGIPortugal,
        PEGIBBFC,
        Russian,
        ACB,
        OFLC
    }

    public enum DataLossConfirmationValue : byte
    {
        None = (byte)0x0u,
        Required = (byte)0x1u,
    }

    public enum PlayLogPolicyValue : byte
    {
        All = (byte)0x0u,
        LogOnly = (byte)0x1u,
        None = (byte)0x2u,
    }

    public enum LogoTypeValue : byte
    {
        LicensedByNintendo = (byte)0x0u,
        Nintendo = (byte)0x2u
    }

    public enum LogoHandlingValue : byte
    {
        Auto = (byte)0x0u,
        Manual = (byte)0x1u,
    }

    public enum AddOnContentRegistrationTypeValue : byte
    {
        AllOnLaunch = (byte)0x0u,
        OnDemand = (byte)0x1u,
    }

    public enum HdcpValue : byte
    {
        None = (byte)0x0u,
        Required = (byte)0x1u,
    }

    public enum CrashReportValue : byte
    {
        Deny = (byte) 0x0u,
        Allow = (byte) 0x1u,
    }

    public enum RuntimeAddOnContentInstallValue : byte
    {
        Deny = (byte)0x0u,
        AllowAppend = (byte)0x1u,
    }

    public enum PlayLogQueryCapabilityValue : byte
    {
        None = (byte) 0x0u,
        WhiteList = (byte) 0x1u,
        All = (byte) 0x2u,
    }

    [Flags]
    public enum RepairFlagValue : byte
    {
        None                    = 0x0,
        SuppressGameCardAccess  = 0x1 << 0,
    }

    [Flags]
    public enum RequiredNetworkServiceLicenseOnLaunchValue : byte
    {
        None                    = 0x0,
        Common                  = 0x1 << 0,
    }

    [StructLayout(LayoutKind.Sequential)]
    public struct ApplicationTitle
    {
        public const int NameLength = 512;
        public const int PublisherLength = 256;
        public const int NameStringLength = 127;
        public const int PublisherStringLength = 63;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = NameLength)]
        public byte[] Name;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = PublisherLength)]
        public byte[] Publisher;
    }

    [StructLayout(LayoutKind.Sequential)]
    public struct ApplicationControlProperty
    {
        public const int MaxLanguageCount = 16;
        public const int IsbnLength = 37;
        public const int RatingAgeCount = 32;
        public const int DisplayVersionLength = 16;
        public const int ApplicationErrorCodeCategoryLength = 8;
        public const int LocalCommunicationIdLength = 8;
        public const int Reserved_0x30F3Length = 3;
        public const int BcatPassphraseLength = 65;
        public const int ReservedForUserAccountSaveDataOperationLength = 6;
        public const int PlayLogQueryableApplicationIdCount = 16;
        public const int ReservedLength = 3564;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = MaxLanguageCount)]
        public ApplicationTitle[] Title;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = IsbnLength)]
        public byte[] Isbn;
        public byte StartupUserAccount;
        public byte reserved_0x3026;
        public byte AddOnContentRegistrationType;
        public uint AttributeFlag;
        public uint SupportedLanguageFlag;
        public uint ParentalControlFlag;
        public byte Screenshot;
        public byte VideoCapture;
        public byte DataLossConfirmation;
        public byte PlayLogPolicy;
        public ulong PresenceGroupId;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = RatingAgeCount)]
        public sbyte[] RatingAge;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = DisplayVersionLength)]
        public byte[] DisplayVersion;
        public ulong AddOnContentBaseId;
        public ulong SaveDataOwnerId;
        public long UserAccountSaveDataSize;
        public long UserAccountSaveDataJournalSize;
        public long DeviceSaveDataSize;
        public long DeviceSaveDataJournalSize;
        public long BcatDeliveryCacheStorageSize;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = ApplicationErrorCodeCategoryLength)]
        public byte[] ApplicationErrorCodeCategory;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = LocalCommunicationIdLength)]
        public ulong[] LocalCommunicationId;
        public byte LogoType;
        public byte LogoHandling;
        public byte RuntimeAddOnContentInstall;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = Reserved_0x30F3Length)]
        public byte[] reserved_0x30F3;
        public byte CrashReport;
        public byte Hdcp;
        public ulong SeedForPseudoDeviceId;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = BcatPassphraseLength)]
        public byte[] BcatPassphrase;
        public byte reserved_0x3141;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = ReservedForUserAccountSaveDataOperationLength)]
        public byte[] ReservedForUserAccountSaveDataOperation;
        public long UserAccountSaveDataSizeMax;
        public long UserAccountSaveDataJournalSizeMax;
        public long DeviceSaveDataSizeMax;
        public long DeviceSaveDataJournalSizeMax;
        public long TemporaryStorageSize;
        public long CacheStorageSize;
        public long CacheStorageJournalSize;
        public long CacheStorageDataAndJournalSizeMax;
        public ushort CacheStorageIndexMax;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = PlayLogQueryableApplicationIdCount)]
        public ulong[] PlayLogQueryableApplicationId;
        public byte PlayLogQueryCapability;
        public byte RepairFlag;
        public byte ProgramIndex;
        public byte RequiredNetworkServiceLicenseOnLaunchFlag;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = ReservedLength)]
        public byte[] Reserved;

        private string BytesToUtf8String(byte[] bytes, string name, int maxCharCount)
        {
            return BytesToString(bytes, name, Encoding.UTF8, maxCharCount);
        }

        private string BytesToString(byte[] bytes, string name, Encoding encoding, int maxCharCount)
        {
            if (bytes.Length == 0)
            {
                return string.Empty;
            }
            var str = encoding.GetString(bytes).TrimEnd('\0');
            var stringInfo = new System.Globalization.StringInfo(str);
            if (stringInfo.LengthInTextElements > maxCharCount)
            {
                throw new ArgumentException(string.Format("<{0}> に指定できる文字数は {1} です。: '{2}'", name, maxCharCount, str));
            }
            return str;
        }

        private string GetLanguageString(int value)
        {
            try
            {
                return Enum.GetName(typeof(Language), value);
            }
            catch
            {
                throw new ArgumentException(string.Format("<Language> に未定義の言語が記述されています: '{0}'", value));
            }
        }

        private List<Title> GetTitles(ApplicationTitle[] title)
        {
            var titles = new List<Title>();
            for (int i = 0; i < MaxLanguageCount; i++)
            {
                var titleEntry = new Title();
                titleEntry.Name = BytesToUtf8String(title[i].Name, "Name", ApplicationTitle.NameStringLength);
                if (String.IsNullOrWhiteSpace(titleEntry.Name))
                {
                    continue;
                }
                titleEntry.Publisher = BytesToUtf8String(title[i].Publisher, "Publisher", ApplicationTitle.PublisherStringLength);
                titleEntry.Language = GetLanguageString(i);
                titles.Add(titleEntry);
            }
            return titles;
        }

        private string GetStartupUserAccountString(byte value)
        {
            try
            {
                return Enum.GetName(typeof(StartupUserAccountValue), value);
            }
            catch
            {
                throw new ArgumentException(string.Format("<StartupUserAccount> に未定義の起動時本体アカウントが記述されています: '{0}'", value));
            }
        }

        private List<string> GetAttributes(uint value)
        {
            if (((AttributeFlagValue)value).Equals(AttributeFlagValue.None))
            {
                return new List<string>();
            }
            return ((AttributeFlagValue)value).ToString().Split(',').Select(x => x.Trim()).ToList();
        }

        private List<string> GetParentalControls(uint value)
        {
            if (((ParentalControlFlagValue)value).Equals(ParentalControlFlagValue.None))
            {
                return new List<string>();
            }
            return ((ParentalControlFlagValue)value).ToString().Split(',').Select(x => x.Trim()).ToList();
        }

        private List<string> GetSupportedLanguage(uint value)
        {
            var languages = new List<string>();
            for (int i = 0; i < MaxLanguageCount; i++)
            {
                if ((value & (0x1u << i)) != 0)
                {
                    languages.Add(GetLanguageString(i));
                }
            }
            return languages;
        }

        private string GetScreenshotString(byte value)
        {
            try
            {
                return Enum.GetName(typeof(ScreenshotValue), value);
            }
            catch
            {
                throw new ArgumentException(string.Format("<Screenshot> に未定義のスクリーンショット撮影が記述されています: '{0}'", value));
            }
        }

        private string GetVideoCaptureString(byte value)
        {
            try
            {
                return Enum.GetName(typeof(VideoCaptureValue), value);
            }
            catch
            {
                throw new ArgumentException(string.Format("<VideoCapture> に未定義の動画撮影が記述されています: '{0}'", value));
            }
        }

        private string GetOrganizationString(int value)
        {
            try
            {
                return Enum.GetName(typeof(Organization), value);
            }
            catch
            {
                throw new ArgumentException(string.Format("<Organization> に未定義のレーティング団体が記述されています: '{0}'", value));
            }
        }

        private List<Rating> GetRatings(sbyte[] ratingAge)
        {
            var ratings = new List<Rating>();
            for (int i = 0; i < RatingAgeCount; i++)
            {
                if (ratingAge[i] == -1)
                {
                    continue;
                }
                var ratingEntry = new Rating();
                ratingEntry.Organization = GetOrganizationString(i);
                ratingEntry.Age = ratingAge[i];
                ratings.Add(ratingEntry);
            }
            return ratings;
        }

        private string GetDataLossConfirmationString(byte value)
        {
            try
            {
                return Enum.GetName(typeof(DataLossConfirmationValue), value);
            }
            catch
            {
                throw new ArgumentException(string.Format("<DataLossConfirmation> に未定義のデータ消失確認が記述されています: '{0}'", value));
            }
        }

        private string GetPlayLogPolicyString(byte value)
        {
            try
            {
                return Enum.GetName(typeof(PlayLogPolicyValue), value);
            }
            catch
            {
                throw new ArgumentException(string.Format("<PlayLogPolicy> に未定義のログ記録方法が記述されています: '{0}'", value));
            }
        }

        private string GetLogoTypeString(byte value)
        {
            try
            {
                return Enum.GetName(typeof(LogoTypeValue), value);
            }
            catch
            {
                throw new ArgumentException(string.Format("<LogoType> に未定義のロゴデータのライセンス種別が記述されています: '{0}'", value));
            }
        }

        private List<string> GetLocalCommunicationIds(ulong[] ids)
        {
            var localCommunicationIds = new List<string>();
            foreach (var id in ids)
            {
                localCommunicationIds.Add("0x" + id.ToString("x16"));
            }
            return localCommunicationIds.Distinct().ToList();
        }

        private string GetLogoHandlingString(byte value)
        {
            try
            {
                return Enum.GetName(typeof(LogoHandlingValue), value);
            }
            catch
            {
                throw new ArgumentException(string.Format("<LogoHandling> に未定義の表示管理方法が記述されています: '{0}'", value));
            }
        }

        private string GetAddOnContentRegistrationTypeString(byte value)
        {
            try
            {
                return Enum.GetName(typeof(AddOnContentRegistrationTypeValue), value);
            }
            catch
            {
                throw new ArgumentException(string.Format("<AddOnContentRegistrationType> に未定義の値が記述されています: '{0}'", value));
            }
        }

        private string GetHdcpString(byte value)
        {
            try
            {
                return Enum.GetName(typeof(HdcpValue), value);
            }
            catch
            {
                throw new ArgumentException(string.Format("<Hdcp> に未定義の値が記述されています: '{0}'", value));
            }
        }

        private string GetCrashReportString(byte value)
        {
            try
            {
                return Enum.GetName(typeof(CrashReportValue), value);
            }
            catch
            {
                throw new ArgumentException(string.Format("<CrashReport> に未定義の値が記述されています: '{0}'", value));
            }
        }

        private string GetRuntimeAddOnContentInstallString(byte value)
        {
            try
            {
                return Enum.GetName(typeof(RuntimeAddOnContentInstallValue), value);
            }
            catch
            {
                throw new ArgumentException(string.Format("<RuntimeAddOnContentInstall> に未定義の値が記述されています: '{0}'", value));
            }
        }

        private List<string> GetPlayLogQueryableApplicationIdString(ulong[] ids)
        {
            var idList = new List<string>();
            foreach (var id in ids)
            {
                idList.Add("0x" + id.ToString("x16"));
            }
            return idList.Distinct().ToList();
        }

        private string GetPlayLogQueryCapabilityString(byte value)
        {
            try
            {
                return Enum.GetName(typeof(PlayLogQueryCapabilityValue), value);
            }
            catch
            {
                throw new ArgumentException(string.Format("<PlayLogQueryCapability> に未定義の値が記述されています: '{0}'", value));
            }
        }

        private List<string> GetRepairString(byte value)
        {
            if (((RepairFlagValue)value).Equals(RepairFlagValue.None))
            {
                return new List<string>();
            }
            return ((RepairFlagValue)value).ToString().Split(',').Select(x => x.Trim()).ToList();
        }

        private List<string> GetRequiredNetworkServiceLicenseOnLaunchString(byte value)
        {
            if (((RequiredNetworkServiceLicenseOnLaunchValue)value).Equals(RequiredNetworkServiceLicenseOnLaunchValue.None))
            {
                return new List<string>();
            }
            return ((RequiredNetworkServiceLicenseOnLaunchValue)value).ToString().Split(',').Select(x => x.Trim()).ToList();
        }

        public ApplicationControlPropertyModel MakePropertyModel()
        {
            var model = new ApplicationControlPropertyModel();
            model.Title = GetTitles(Title);
            model.Isbn = BytesToUtf8String(Isbn, "Isbn", IsbnLength - 1);
            model.StartupUserAccount = GetStartupUserAccountString(StartupUserAccount);
            model.Attribute = GetAttributes(AttributeFlag);
            model.ParentalControl = GetParentalControls(ParentalControlFlag);
            model.SupportedLanguage = GetSupportedLanguage(SupportedLanguageFlag);
            model.Screenshot = GetScreenshotString(Screenshot);
            model.VideoCapture = GetVideoCaptureString(VideoCapture);
            model.PresenceGroupId = "0x" + PresenceGroupId.ToString("x16");
            model.DisplayVersion = BytesToUtf8String(DisplayVersion, "DisplayVersion", DisplayVersionLength - 1);
            model.Rating = GetRatings(RatingAge);
            model.DataLossConfirmation = GetDataLossConfirmationString(DataLossConfirmation);
            model.PlayLogPolicy = GetPlayLogPolicyString(PlayLogPolicy);
            model.SaveDataOwnerId = "0x" + SaveDataOwnerId.ToString("x16");
            model.UserAccountSaveDataSize = "0x" + UserAccountSaveDataSize.ToString("x16");
            model.UserAccountSaveDataJournalSize = "0x" + UserAccountSaveDataJournalSize.ToString("x16");
            model.DeviceSaveDataSize = "0x" + DeviceSaveDataSize.ToString("x16");
            model.DeviceSaveDataJournalSize = "0x" + DeviceSaveDataJournalSize.ToString("x16");
            model.BcatDeliveryCacheStorageSize = "0x" + BcatDeliveryCacheStorageSize.ToString("x16");
            model.ApplicationErrorCodeCategory = BytesToUtf8String(ApplicationErrorCodeCategory, "ApplicationErrorCodeCategory", ApplicationErrorCodeCategoryLength - 1);
            model.AddOnContentBaseId = "0x" + AddOnContentBaseId.ToString("x16");
            model.LogoType = GetLogoTypeString(LogoType);
            model.LocalCommunicationId = GetLocalCommunicationIds(LocalCommunicationId);
            model.LogoHandling = GetLogoHandlingString(LogoHandling);
            model.SeedForPseudoDeviceId = SeedForPseudoDeviceId.ToString("x16");
            model.BcatPassphrase = BytesToUtf8String(BcatPassphrase, "BcatPassphrase", BcatPassphraseLength - 1);
            model.AddOnContentRegistrationType = GetAddOnContentRegistrationTypeString(AddOnContentRegistrationType);
            model.UserAccountSaveDataSizeMax = "0x" + UserAccountSaveDataSizeMax.ToString("x16");
            model.UserAccountSaveDataJournalSizeMax = "0x" + UserAccountSaveDataJournalSizeMax.ToString("x16");
            model.DeviceSaveDataSizeMax = "0x" + DeviceSaveDataSizeMax.ToString("x16");
            model.DeviceSaveDataJournalSizeMax = "0x" + DeviceSaveDataJournalSizeMax.ToString("x16");
            model.TemporaryStorageSize = "0x" + TemporaryStorageSize.ToString("x16");
            model.CacheStorageSize = "0x" + CacheStorageSize.ToString("x16");
            model.CacheStorageJournalSize = "0x" + CacheStorageJournalSize.ToString("x16");
            model.CacheStorageDataAndJournalSizeMax = "0x" + CacheStorageDataAndJournalSizeMax.ToString("x16");
            model.CacheStorageIndexMax = "0x" + CacheStorageIndexMax.ToString("x16");
            model.Hdcp = GetHdcpString(Hdcp);
            model.CrashReport = GetCrashReportString(CrashReport);
            model.RuntimeAddOnContentInstall = GetRuntimeAddOnContentInstallString(RuntimeAddOnContentInstall);
            model.PlayLogQueryableApplicationId = GetPlayLogQueryableApplicationIdString(PlayLogQueryableApplicationId);
            model.PlayLogQueryCapability = GetPlayLogQueryCapabilityString(PlayLogQueryCapability);
            model.Repair = GetRepairString(RepairFlag);
            model.ProgramIndex = ProgramIndex.ToString();
            model.RequiredNetworkServiceLicenseOnLaunch = GetRequiredNetworkServiceLicenseOnLaunchString(RequiredNetworkServiceLicenseOnLaunchFlag);

            // 以下はバイナリにはデータが存在しないため省略
            // model.Version;
            // model.ReleaseVersion;
            // model.PrivateVersion;
            // model.RequiredSystemVersion;
            // model.ApplicationId;
            return model;
        }
    }

    public class DataPath
    {
        [System.Xml.Serialization.XmlAttribute("UseEnvironmentVariable")]
        public bool UseEnvironmentVariables { get; set; }

        [System.Xml.Serialization.XmlText]
        public string Path
        {
            get
            {
                if(UseEnvironmentVariables)
                {
                    return Environment.ExpandEnvironmentVariables(m_Path);
                }
                else
                {
                    return m_Path;
                }
            }
            set
            {
                m_Path = value;
            }
        }

        private string m_Path;
    }

    [System.Xml.Serialization.XmlRoot("Title", IsNullable = false)]
    public class Title
    {
        [System.Xml.Serialization.XmlElement("Language")]
        public string Language { get; set; }

        [System.Xml.Serialization.XmlElement("Name")]
        public string Name { get; set; }

        [System.Xml.Serialization.XmlElement("Publisher")]
        public string Publisher { get; set; }
    }

    [System.Xml.Serialization.XmlRoot("Rating", IsNullable = false)]
    public class Rating
    {
        public string Organization { get; set; }
        public sbyte Age { get; set; }
    }

    [System.Xml.Serialization.XmlRoot("Icon", IsNullable = false)]
    public class Icon
    {
        [System.Xml.Serialization.XmlElement("Language")]
        public string Language { get; set; }

        [System.Xml.Serialization.XmlElement("IconPath")]
        public DataPath IconPath { get; set; } = new DataPath();

        [System.Xml.Serialization.XmlElement("NxIconPath")]
        public DataPath NxIconPath { get; set; } = new DataPath();

        [System.Xml.Serialization.XmlElement("RawIconHash")]
        public string RawIconHash { get; set; }

        [System.Xml.Serialization.XmlElement("NxIconHash")]
        public string NxIconHash { get; set; }
    }


    [System.Xml.Serialization.XmlRoot("Application", IsNullable = false)]
    public class ApplicationControlPropertyModel
    {
        [System.Xml.Serialization.XmlElement("Title")]
        public List<Title> Title { get; set; }

        [System.Xml.Serialization.XmlElement("Isbn")]
        public string Isbn { get; set; }

        [System.Xml.Serialization.XmlElement("StartupUserAccount")]
        public string StartupUserAccount { get; set; }

        [System.Xml.Serialization.XmlElement("Attribute")]
        public List<string> Attribute { get; set; }

        [System.Xml.Serialization.XmlElement("ParentalControl")]
        public List<string> ParentalControl { get; set; }

        [System.Xml.Serialization.XmlElement("SupportedLanguage")]
        public List<string> SupportedLanguage { get; set; }

        [System.Xml.Serialization.XmlElement("Screenshot")]
        public string Screenshot { get; set; }

        [System.Xml.Serialization.XmlElement("VideoCapture")]
        public string VideoCapture { get; set; }

        [System.Xml.Serialization.XmlElement("PresenceGroupId")]
        public string PresenceGroupId { get; set; }

        [System.Xml.Serialization.XmlElement("DisplayVersion")]
        public string DisplayVersion { get; set; }

        [System.Xml.Serialization.XmlElement("Rating")]
        public List<Rating> Rating { get; set; }

        [System.Xml.Serialization.XmlElement("DataLossConfirmation")]
        public string DataLossConfirmation { get; set; }

        [System.Xml.Serialization.XmlElement("PlayLogPolicy")]
        public string PlayLogPolicy { get; set; }

        [System.Xml.Serialization.XmlElement("SaveDataOwnerId")]
        public string SaveDataOwnerId { get; set; }

        [System.Xml.Serialization.XmlElement("UserAccountSaveDataSize")]
        public string UserAccountSaveDataSize { get; set; }

        [System.Xml.Serialization.XmlElement("UserAccountSaveDataJournalSize")]
        public string UserAccountSaveDataJournalSize { get; set; }

        [System.Xml.Serialization.XmlElement("DeviceSaveDataSize")]
        public string DeviceSaveDataSize { get; set; }

        [System.Xml.Serialization.XmlElement("DeviceSaveDataJournalSize")]
        public string DeviceSaveDataJournalSize { get; set; }

        [System.Xml.Serialization.XmlElement("BcatDeliveryCacheStorageSize")]
        public string BcatDeliveryCacheStorageSize { get; set; }

        [System.Xml.Serialization.XmlElement("ApplicationErrorCodeCategory")]
        public string ApplicationErrorCodeCategory { get; set; }

        [System.Xml.Serialization.XmlElement("AddOnContentBaseId")]
        public string AddOnContentBaseId { get; set; }

        // XML としてパース出来るようにするために定義しておく
        [System.Xml.Serialization.XmlElement("Version")]
        public string Version { get; set; }

        // XML としてパース出来るようにするために定義しておく
        [System.Xml.Serialization.XmlElement("ReleaseVersion")]
        public string ReleaseVersion { get; set; }

        // XML としてパース出来るようにするために定義しておく
        [System.Xml.Serialization.XmlElement("PrivateVersion")]
        public string PrivateVersion { get; set; }

        [System.Xml.Serialization.XmlElement("LogoType")]
        public string LogoType { get; set; }

        // XML としてパース出来るようにするために定義しておく
        [System.Xml.Serialization.XmlElement("RequiredSystemVersion")]
        public string RequiredSystemVersion { get; set; }

        [System.Xml.Serialization.XmlElement("LocalCommunicationId")]
        public List<string> LocalCommunicationId { get; set; }

        [System.Xml.Serialization.XmlElement("LogoHandling")]
        public string LogoHandling { get; set; }

        // XML としてパースできるようにするために定義しておく
        [System.Xml.Serialization.XmlElement("Icon")]
        public List<Icon> Icon { get; set; }

        // XML としてパースできるようにするために定義しておく
        [System.Xml.Serialization.XmlElement("HtmlDocumentPath")]
        public DataPath HtmlDocumentPath { get; set; } = new DataPath();

        // XML としてパースできるようにするために定義しておく
        [System.Xml.Serialization.XmlElement("LegalInformationFilePath")]
        public DataPath LegalInformationFilePath { get; set; } = new DataPath();

        // XML としてパースできるようにするために定義しておく
        [System.Xml.Serialization.XmlElement("AccessibleUrlsFilePath")]
        public DataPath AccessibleUrlsFilePath { get; set; } = new DataPath();

        [System.Xml.Serialization.XmlElement("SeedForPseudoDeviceId")]
        public string SeedForPseudoDeviceId { get; set; }

        [System.Xml.Serialization.XmlElement("BcatPassphrase")]
        public string BcatPassphrase { get; set; }

        [System.Xml.Serialization.XmlElement("AddOnContentRegistrationType")]
        public string AddOnContentRegistrationType { get; set; }

        [System.Xml.Serialization.XmlElement("UserAccountSaveDataSizeMax")]
        public string UserAccountSaveDataSizeMax { get; set; }

        [System.Xml.Serialization.XmlElement("UserAccountSaveDataJournalSizeMax")]
        public string UserAccountSaveDataJournalSizeMax { get; set; }

        [System.Xml.Serialization.XmlElement("DeviceSaveDataSizeMax")]
        public string DeviceSaveDataSizeMax { get; set; }

        [System.Xml.Serialization.XmlElement("DeviceSaveDataJournalSizeMax")]
        public string DeviceSaveDataJournalSizeMax { get; set; }

        [System.Xml.Serialization.XmlElement("TemporaryStorageSize")]
        public string TemporaryStorageSize { get; set; }

        [System.Xml.Serialization.XmlElement("CacheStorageSize")]
        public string CacheStorageSize { get; set; }

        [System.Xml.Serialization.XmlElement("CacheStorageJournalSize")]
        public string CacheStorageJournalSize { get; set; }

        [System.Xml.Serialization.XmlElement("CacheStorageDataAndJournalSizeMax")]
        public string CacheStorageDataAndJournalSizeMax { get; set; }

        [System.Xml.Serialization.XmlElement("CacheStorageIndexMax")]
        public string CacheStorageIndexMax { get; set; }

        [System.Xml.Serialization.XmlElement("Hdcp")]
        public string Hdcp { get; set; }

        [System.Xml.Serialization.XmlElement("CrashReport")]
        public string CrashReport { get; set; }

        [System.Xml.Serialization.XmlElement("RuntimeAddOnContentInstall")]
        public string RuntimeAddOnContentInstall { get; set; }

        [System.Xml.Serialization.XmlElement("PlayLogQueryableApplicationId")]
        public List<string> PlayLogQueryableApplicationId { get; set; }

        [System.Xml.Serialization.XmlElement("PlayLogQueryCapability")]
        public string PlayLogQueryCapability { get; set; }

        [System.Xml.Serialization.XmlElement("Repair")]
        public List<string> Repair { get; set; }

        [System.Xml.Serialization.XmlElement("ProgramIndex")]
        public string ProgramIndex { get; set; }

        [System.Xml.Serialization.XmlElement("RequiredNetworkServiceLicenseOnLaunch")]
        public List<string> RequiredNetworkServiceLicenseOnLaunch { get; set; }

        // XML としてパースできるようにするために定義しておく
        [System.Xml.Serialization.XmlElement("ApplicationId")]
        public string ApplicationId { get; set; }

        // XML としてパースできるようにするために定義しておく
        [System.Xml.Serialization.XmlElement("FilterDescriptionFilePath")]
        public string FilterDescriptionFilePath { get; set; }

        private ApplicationControlProperty MakeApplicationControlProperty(ulong programId, ulong defaultAddOnContentBaseId)
        {
            var property = new ApplicationControlProperty();
            property.ProgramIndex = GetIndex(ProgramIndex);
            ulong applicationId = programId - property.ProgramIndex;
            if (applicationId != GetApplicationId(ApplicationId, programId))
            {
                throw new ArgumentException("<ApplicationId> もしくは <ProgramIndex> の指定が間違っています。");
            }

            property.Title = new ApplicationTitle[ApplicationControlProperty.MaxLanguageCount];
            for (int i = 0; i < ApplicationControlProperty.MaxLanguageCount; i++)
            {
                ApplicationTitle title = new ApplicationTitle();
                title.Name = new byte[ApplicationTitle.NameLength];
                title.Publisher = new byte[ApplicationTitle.PublisherLength];
                property.Title[i] = title;
            }
            property.Isbn = new byte[ApplicationControlProperty.IsbnLength];
            property.ApplicationErrorCodeCategory = new byte[ApplicationControlProperty.ApplicationErrorCodeCategoryLength];
            property.LocalCommunicationId = new ulong[ApplicationControlProperty.LocalCommunicationIdLength];
            property.reserved_0x30F3 = new byte[ApplicationControlProperty.Reserved_0x30F3Length];
            property.ReservedForUserAccountSaveDataOperation = new byte[ApplicationControlProperty.ReservedForUserAccountSaveDataOperationLength];
            property.BcatPassphrase = new byte[ApplicationControlProperty.BcatPassphraseLength];
            property.PlayLogQueryableApplicationId = new ulong[ApplicationControlProperty.PlayLogQueryableApplicationIdCount];
            property.Reserved = new byte[ApplicationControlProperty.ReservedLength];

            foreach(var title in Title)
            {
                var index = GetTitleIndexByLanguage(title.Language);

                CopyUtf8StringBytes(title.Name, property.Title[index].Name, "Name", ApplicationTitle.NameStringLength);
                CopyUtf8StringBytes(title.Publisher, property.Title[index].Publisher, "Publisher", ApplicationTitle.PublisherStringLength);
            }

            CopyUtf8StringBytes(Isbn, property.Isbn, "Isbn", ApplicationControlProperty.IsbnLength - 1);

            property.StartupUserAccount = GetStartupUserAccount(StartupUserAccount);

            foreach (var attribute in Attribute)
            {
                property.AttributeFlag |= GetAttributeFlag(attribute);
            }

            foreach (var language in SupportedLanguage)
            {
                property.SupportedLanguageFlag |= GetSupportedLanguageFlag(language);
            }

            foreach (var parentalControl in ParentalControl)
            {
                property.ParentalControlFlag |= GetParentalControlFlag(parentalControl);
            }

            property.Screenshot = GetScreenshot(Screenshot);
            property.VideoCapture = GetVideoCapture(VideoCapture);

            property.PresenceGroupId = GetPresenceGroupId(PresenceGroupId, applicationId);

            property.RatingAge = Enumerable.Repeat<sbyte>(-1, ApplicationControlProperty.RatingAgeCount).ToArray();
            foreach (var rating in Rating)
            {
                property.RatingAge[GetRatingIndexByOrganization(rating.Organization)] = rating.Age;
            }

            property.DisplayVersion = new byte[ApplicationControlProperty.DisplayVersionLength];
            CopyUtf8StringBytes(DisplayVersion, property.DisplayVersion, "DisplayVersion", ApplicationControlProperty.DisplayVersionLength - 1);

            property.DataLossConfirmation = GetDataLossConfirmation(DataLossConfirmation);

            property.PlayLogPolicy = GetPlayLogPolicy(PlayLogPolicy);

            property.SaveDataOwnerId = GetSaveDataOwnerId(SaveDataOwnerId, applicationId);
            property.UserAccountSaveDataSize = GetSaveDataSize(UserAccountSaveDataSize);
            property.UserAccountSaveDataJournalSize = GetSaveDataSize(UserAccountSaveDataJournalSize);
            property.DeviceSaveDataSize = GetSaveDataSize(DeviceSaveDataSize);
            property.DeviceSaveDataJournalSize = GetSaveDataSize(DeviceSaveDataJournalSize);
            property.BcatDeliveryCacheStorageSize = GetBcatDeliveryCacheStorageSize(BcatDeliveryCacheStorageSize);

            CopyUtf8StringBytes(ApplicationErrorCodeCategory, property.ApplicationErrorCodeCategory, "ApplicationErrorCodeCategory", ApplicationControlProperty.ApplicationErrorCodeCategoryLength - 1);

            property.LogoType = GetLogoType(LogoType);
            property.AddOnContentBaseId = GetAddOnContentBaseId(AddOnContentBaseId, defaultAddOnContentBaseId);

            property.LocalCommunicationId = Enumerable.Repeat<ulong>(applicationId, property.LocalCommunicationId.Count()).ToArray();
            foreach (var item in LocalCommunicationId.Select((localCommunicationId, i) => new { localCommunicationId, i }))
            {
                property.LocalCommunicationId[item.i] = GetLocalCommunicationId(item.localCommunicationId);
            }

            property.LogoHandling = GetLogoHandling(LogoHandling);

            property.SeedForPseudoDeviceId = GetSeedForPseudoDeviceId(SeedForPseudoDeviceId, applicationId);

            CopyBcatPassPhrase(BcatPassphrase, property.BcatPassphrase);

            property.AddOnContentRegistrationType = GetAddOnContentRegistrationType(AddOnContentRegistrationType);

            const long SaveDataExtensionSizeUnit = 1 * 1024 * 1024;
            property.UserAccountSaveDataSizeMax = GetSaveDataExtensionSizeMax(UserAccountSaveDataSizeMax, SaveDataExtensionSizeUnit);
            property.UserAccountSaveDataJournalSizeMax = GetSaveDataExtensionSizeMax(UserAccountSaveDataJournalSizeMax, SaveDataExtensionSizeUnit);
            property.DeviceSaveDataSizeMax = GetSaveDataExtensionSizeMax(DeviceSaveDataSizeMax, SaveDataExtensionSizeUnit);
            property.DeviceSaveDataJournalSizeMax = GetSaveDataExtensionSizeMax(DeviceSaveDataJournalSizeMax, SaveDataExtensionSizeUnit);

            property.TemporaryStorageSize = GetSaveDataSize(TemporaryStorageSize);
            property.CacheStorageSize = GetSaveDataSize(CacheStorageSize);
            property.CacheStorageJournalSize = GetSaveDataSize(CacheStorageJournalSize);
            property.CacheStorageDataAndJournalSizeMax = GetSaveDataSize(CacheStorageDataAndJournalSizeMax);
            property.CacheStorageIndexMax = GetSaveDataIndexMax(CacheStorageIndexMax);

            property.Hdcp = GetHdcp(Hdcp);
            property.CrashReport = GetCrashReportValue(CrashReport);
            property.RuntimeAddOnContentInstall = GetRuntimeAddOnContentInstall(RuntimeAddOnContentInstall);
            property.PlayLogQueryableApplicationId = Enumerable.Repeat<ulong>(0, property.PlayLogQueryableApplicationId.Count()).ToArray();
            foreach (var item in PlayLogQueryableApplicationId.Select((id, i) => new { id, i }))
            {
                property.PlayLogQueryableApplicationId[item.i] = GetPlayLogQueryableApplicationId(item.id);
            }
            property.PlayLogQueryCapability = GetPlayLogQueryCapability(PlayLogQueryCapability);
            foreach (var repair in Repair)
            {
                property.RepairFlag |= GetRepairFlag(repair);
            }
            foreach (var license in RequiredNetworkServiceLicenseOnLaunch)
            {
                property.RequiredNetworkServiceLicenseOnLaunchFlag |= GetRequiredNetworkServiceLicenseOnLaunchFlag(license);
            }
            return property;
        }

        public byte[] MakePropertyBytes(ulong programId, ulong defaultAddOnContentBaseId)
        {
            var property = MakeApplicationControlProperty(programId, defaultAddOnContentBaseId);
            var size = Marshal.SizeOf(typeof(ApplicationControlProperty));

            var ptr = Marshal.AllocHGlobal(size);
            Marshal.StructureToPtr(property, ptr, false);

            byte[] bytes = new byte[size];
            Marshal.Copy(ptr, bytes, 0, size);
            Marshal.FreeHGlobal(ptr);

            Debug.Assert(bytes.Length == 16384);

            return bytes;
        }

        public static ApplicationControlPropertyModel PropertyBytesToModel(byte[] bytes)
        {
            Debug.Assert(bytes.Length == 16384);

            var size = Marshal.SizeOf(typeof(ApplicationControlProperty));

            var ptr = Marshal.AllocHGlobal(size);
            Marshal.Copy(bytes, 0, ptr, size);
            var property =  Marshal.PtrToStructure<ApplicationControlProperty>(ptr);
            var model =  property.MakePropertyModel();
            Marshal.FreeHGlobal(ptr);

            return model;
        }

        public void AreValidValues()
        {
            // MakeApplicationControlProperty でエラーがでなければ問題ない
            MakeApplicationControlProperty(0, 0);
        }

        private int GetTitleIndexByLanguage(string language)
        {
            if(String.IsNullOrWhiteSpace(language))
            {
                throw new ArgumentException("<Title> 内に <Language> がありません");
            }

            var trimmed = language.Trim();

            try
            {
                return (int)Enum.Parse(typeof(Language), trimmed);
            }
            catch(ArgumentException)
            {
                throw new ArgumentException(string.Format("<Language> に未定義の言語が記述されています: '{0}'", trimmed));
            }
        }

        private void CopyUtf8StringBytes(string value, byte[] dest, string name, int maxCharCount)
        {
            CopyStringBytes(value, dest, name, Encoding.UTF8, maxCharCount);
        }

        private void CopyStringBytes(string value, byte[] dest, string name, Encoding encoding, int maxCharCount)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return;
            }

            var trimmed = value.Trim();

            var stringInfo = new System.Globalization.StringInfo(trimmed);
            if (stringInfo.LengthInTextElements > maxCharCount)
            {
                throw new ArgumentException(string.Format("<{0}> に指定できる文字数は {1} です。: '{2}'", name, maxCharCount, trimmed));
            }

            byte[] bytes = encoding.GetBytes(trimmed);
            Array.Copy(bytes, dest, bytes.Length);
        }

        private void CopyAsciiHexStringBytes(string value, byte[] dest, string name, int maxCharCount)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return;
            }

            foreach (var c in value)
            {
                if (!(('a' <= c && c <= 'f') || ('A' <= c && c <= 'F') || ('0' <= c && c <= '9')))
                {
                    throw new ArgumentException(string.Format("<{0}> に 16 進数 (a-fA-F0-9) ではない文字が含まれています。: {1}", name, value));
                }
            }

            CopyUtf8StringBytes(value, dest, name, maxCharCount);
        }

        private void CopyBcatPassPhrase(string value, byte[] dest)
        {
            CopyAsciiHexStringBytes(value, dest, "BcatPassphrase", ApplicationControlProperty.BcatPassphraseLength - 1);
            if (!(string.IsNullOrEmpty(value) || value.Count() == ApplicationControlProperty.BcatPassphraseLength - 1))
            {
                throw new ArgumentException(string.Format("<BcatPassphrase> は {0} 文字で指定してください: '{1}' ({2} 文字)", ApplicationControlProperty.BcatPassphraseLength - 1, value, value.Count()));
            }
        }

        private byte GetStartupUserAccount(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return (byte)StartupUserAccountValue.None;
            }

            var trimmed = value.Trim();

            try
            {
                return (byte)Enum.Parse(typeof(StartupUserAccountValue), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<StartupUserAccount> に未定義の起動時本体アカウントが記述されています: '{0}'", trimmed));
            }
        }

        private uint GetAttributeFlag(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return (uint)AttributeFlagValue.None;
            }

            var trimmed = value.Trim();

            try
            {
                return (uint)Enum.Parse(typeof(AttributeFlagValue), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<Attribute> に未定義の特性が記述されています: '{0}'", trimmed));
            }
        }

        private uint GetSupportedLanguageFlag(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return 0;
            }

            var trimmed = value.Trim();

            try
            {
                return 0x1u << (int)Enum.Parse(typeof(Language), trimmed);
            }
            catch (ArgumentException)
            {
                throw new ArgumentException(string.Format("<SupportedLanguage> に未定義の言語が記述されています: '{0}'", trimmed));
            }
        }
        private uint GetParentalControlFlag(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return (uint)ParentalControlFlagValue.None;
            }

            var trimmed = value.Trim();

            try
            {
                return (uint)Enum.Parse(typeof(ParentalControlFlagValue), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<ParentalControl> に未定義のペアレンタルコントロール可能機能が記述されています: '{0}'", trimmed));
            }
        }

        private byte GetScreenshot(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return (byte)ScreenshotValue.Allow;
            }

            var trimmed = value.Trim();

            try
            {
                return (byte)Enum.Parse(typeof(ScreenshotValue), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<Screenshot> に未定義のスクリーンショット撮影が記述されています: '{0}'", trimmed));
            }
        }
        private byte GetVideoCapture(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return (byte)VideoCaptureValue.Enable;
            }

            var trimmed = value.Trim();

            try
            {
                return (byte)Enum.Parse(typeof(VideoCaptureValue), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<VideoCapture> に未定義の動画撮影が記述されています: '{0}'", trimmed));
            }
        }

        private ulong GetPresenceGroupId(string value, ulong defaultId)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return defaultId;
            }

            var trimmed = value.Trim();

            return Convert.ToUInt64(value, 16);
        }

        private int GetRatingIndexByOrganization(string organization)
        {
            if (String.IsNullOrWhiteSpace(organization))
            {
                throw new ArgumentException("<Rating> 内に <Organization> がありません");
            }

            var trimmed = organization.Trim();

            try
            {
                return (int)Enum.Parse(typeof(Organization), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<Organization> に未定義のレーティング団体が記述されています: '{0}'", trimmed));
            }
        }

        private byte GetDataLossConfirmation(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return (byte)DataLossConfirmationValue.None;
            }

            var trimmed = value.Trim();

            try
            {
                return (byte)Enum.Parse(typeof(DataLossConfirmationValue), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<DataLossConfirmation> に未定義のデータ消失確認が記述されています: '{0}'", trimmed));
            }
        }

        private byte GetPlayLogPolicy(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return (byte)PlayLogPolicyValue.All;
            }

            var trimmed = value.Trim();

            try
            {
                return (byte)Enum.Parse(typeof(PlayLogPolicyValue), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<PlayLogPolicy> に未定義のログ記録方法が記述されています: '{0}'", trimmed));
            }
        }

        private ulong GetSaveDataOwnerId(string value, ulong defaultValue)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return defaultValue;
            }
            return Convert.ToUInt64(value, 16);
        }

        private long GetSaveDataSize(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return 0;
            }
            var size = Convert.ToInt64(value, 16);
            const long alignment = 16 * 1024;
            if ((size & (alignment - 1)) != 0)
            {
                throw new ArgumentException(string.Format("セーブデータサイズ値は {0} バイトの倍数になるように指定してください: '{1}'", alignment, value));
            }
            if (size < 0)
            {
                throw new ArgumentException(string.Format("セーブデータサイズ値は正の数になるように指定してください: '{0}'", value));
            }
            return size;
        }

        private long GetSaveDataExtensionSizeMax(string value, long alignment)
        {
            var size = GetSaveDataSize(value);
            if (alignment != 0 && (size & (alignment - 1)) != 0)
            {
                throw new ArgumentException(string.Format("セーブデータ拡張上限値は {0} バイトの倍数になるように指定してください: '{1}'", alignment, value));
            }
            return size;
        }

        private long GetBcatDeliveryCacheStorageSize(string value)
        {
            var size = GetSaveDataSize(value);
            if (size == 0)
            {
                return size;
            }
            else if ((size & (0x100000 - 1)) != 0)
            {
                throw new ArgumentException(string.Format("<BcatDeliveryCacheStorageSize> が 0x100000 (1 MiB) 単位になっていません: '{0}'", value));
            }
            else if (size < 0x500000)
            {
                throw new ArgumentException(string.Format("<BcatDeliveryCacheStorageSize> は 0x500000 (5 MiB) 以上で指定してください: '{0}'", value));
            }
            return size;
        }

        private ushort GetSaveDataIndexMax(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return 0;
            }
            var index = Convert.ToInt16(value, 16);
            if (index < 0)
            {
                throw new ArgumentException(string.Format("インデックス値は正の数になるように指定してください: '{0}'", value));
            }
            return (ushort)index;
        }

        private byte GetLogoType(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return (byte)LogoTypeValue.LicensedByNintendo;
            }

            var trimmed = value.Trim();

            try
            {
                return (byte)Enum.Parse(typeof(LogoTypeValue), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<LogoType> に未定義のロゴデータのライセンス種別が記述されています: '{0}'", trimmed));
            }
        }

        private ulong GetAddOnContentBaseId(string value, ulong defaultValue)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return defaultValue;
            }
            return Convert.ToUInt64(value, 16);
        }

        private ulong GetLocalCommunicationId(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                throw new ArgumentException("<LocalCommunicationId> が指定されましたが、値が未記入です。");
            }

            var trimmed = value.Trim();

            return Convert.ToUInt64(value, 16);
        }

        private byte GetLogoHandling(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return (byte)LogoHandlingValue.Auto;
            }

            var trimmed = value.Trim();

            try
            {
                return (byte)Enum.Parse(typeof(LogoHandlingValue), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<LogoHandling> に未定義のロゴ表示管理方法が記述されています: '{0}'", trimmed));
            }
        }

        private ulong GetSeedForPseudoDeviceId(string value, ulong defaultValue)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return defaultValue;
            }

            var trimmed = value.Trim();

            return Convert.ToUInt64(value, 16);
        }

        private byte GetAddOnContentRegistrationType(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return (byte)AddOnContentRegistrationTypeValue.OnDemand;
            }

            var trimmed = value.Trim();

            try
            {
                return (byte)Enum.Parse(typeof(AddOnContentRegistrationTypeValue), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<AddOnContentRegistrationTypeValue> に未定義の値が記述されています: '{0}'", trimmed));
            }
        }

        private byte GetHdcp(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return (byte)HdcpValue.None;
            }

            var trimmed = value.Trim();

            try
            {
                return (byte)Enum.Parse(typeof(HdcpValue), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<Hdcp> に未定義の値が記述されています: '{0}'", trimmed));
            }
        }

        private byte GetCrashReportValue(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return (byte)CrashReportValue.Deny;
            }

            var trimmed = value.Trim();

            try
            {
                return (byte)Enum.Parse(typeof(CrashReportValue), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<CrashReport> に未定義の値が記述されています: '{0}'", trimmed));
            }
        }
        private byte GetRuntimeAddOnContentInstall(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return (byte)RuntimeAddOnContentInstallValue.Deny;
            }

            var trimmed = value.Trim();

            try
            {
                return (byte)Enum.Parse(typeof(RuntimeAddOnContentInstallValue), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<RuntimeAddOnContentInstall> に未定義の値が記述されています: '{0}'", trimmed));
            }
        }

        private ulong GetPlayLogQueryableApplicationId(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                throw new ArgumentException("<PlayLogQueryableApplicationId> が指定されましたが、値が未記入です。");
            }

            var trimmed = value.Trim();

            return Convert.ToUInt64(value, 16);
        }

        private byte GetPlayLogQueryCapability(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return (byte)0;
            }

            var trimmed = value.Trim();
            try
            {
                return (byte)Enum.Parse(typeof(PlayLogQueryCapabilityValue), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<PlayLogQueryCapability> に未定義の値が記述されています: '{0}'", trimmed));
            }
        }

        private byte GetRepairFlag(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return (byte)RepairFlagValue.None;
            }

            var trimmed = value.Trim();
            try
            {
                return (byte)Enum.Parse(typeof(RepairFlagValue), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<Repair> に未定義の値が記述されています: '{0}'", trimmed));
            }
        }

        private byte GetRequiredNetworkServiceLicenseOnLaunchFlag(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return (byte)RequiredNetworkServiceLicenseOnLaunchValue.None;
            }

            var trimmed = value.Trim();
            try
            {
                return (byte)Enum.Parse(typeof(RequiredNetworkServiceLicenseOnLaunchValue), trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<RequiredNetworkServiceLicenseOnLaunch> に未定義の値が記述されています: '{0}'", trimmed));
            }
        }

        private byte GetIndex(string value)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return 0;
            }

            var trimmed = value.Trim();
            try
            {
                return (byte)Byte.Parse(trimmed);
            }
            catch
            {
                throw new ArgumentException(string.Format("<Index> に未定義の値が記述されています: '{0}'", trimmed));
            }
        }

        private ulong GetApplicationId(string value, ulong defaultApplicationId)
        {
            if (String.IsNullOrWhiteSpace(value))
            {
                return defaultApplicationId;
            }

            var trimmed = value.Trim();
            return Convert.ToUInt64(value, 16);
        }
    }
}
