﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Xml.Linq;
using System.Xml.Serialization;
using System.Text;
using System.Text.RegularExpressions;
using System.Diagnostics;
using System.Runtime.CompilerServices;
using System.Runtime.InteropServices;
using Nintendo.Authoring.AuthoringLibrary;
using Nintendo.Authoring.FileSystemMetaLibrary;

[assembly: InternalsVisibleTo("AuthoringToolsExecutionTest")]
[assembly: InternalsVisibleTo("AuthoringToolsExecutionProdTest")]

namespace Nintendo.Authoring.AuthoringTool
{
    using RomFsListFileInfo = Tuple<string, Int64>;
    using DirectoryConnector = Pair<string, string>;

    internal class Program
    {
        private class FsUtil
        {
            internal static void WriteFile(IFileSystemArchiveReader fsReader, string extractedPath, string fileName, long fileSize)
            {
                using (var writerStream = new FileStream(extractedPath, FileMode.Create,
                    FileAccess.ReadWrite, FileShare.None, 4096, FileOptions.SequentialScan))
                {
                    long readSize = 0;
                    while (readSize < fileSize)
                    {
                        uint bufferSize = (uint)Math.Min(1024 * 1024, fileSize - readSize);
                        var bytes = fsReader.ReadFile(fileName, readSize, bufferSize);
                        writerStream.Write(bytes, 0, bytes.Length);
                        readSize += bytes.Length;
                    }
                }
            }

            internal static bool CompareFile(IFileSystemArchiveReader sourceFsReader, IFileSystemArchiveReader targetFsReader, string fileName, long fileSize)
            {
                long readSize = 0;
                while (readSize < fileSize)
                {
                    uint bufferSize = (uint)Math.Min(4 * 1024 * 1024, fileSize - readSize);

                    var sourceBytes = sourceFsReader.ReadFile(fileName, readSize, bufferSize);
                    var targetBytes = targetFsReader.ReadFile(fileName, readSize, bufferSize);

                    if (sourceBytes.SequenceEqual(targetBytes) == false)
                        return false;

                    readSize += sourceBytes.Length;
                }
                return true;
            }

            internal static bool CheckShouldSkipStartsWithPath(string targetPath, string startsWithSkipKeyPath)
            {
                if (targetPath == null || string.IsNullOrEmpty(startsWithSkipKeyPath))
                {
                    return false;
                }
                return !targetPath.StartsWith(startsWithSkipKeyPath);
            }

            internal static bool CheckShouldSkipComparePath(string targetPath, string startsWithSkipKeyPath)
            {
                if (targetPath == null || string.IsNullOrEmpty(startsWithSkipKeyPath))
                {
                    return false;
                }
                return targetPath != startsWithSkipKeyPath;
            }

            [DllImport("kernel32.dll", SetLastError=true)]
            public static extern bool CreateHardLink(string lpFileName, string lpExistingFileName, IntPtr lpSecurityAttributes);
            [DllImport("kernel32.dll")]
            private static extern uint FormatMessage(uint dwFlags, IntPtr lpSource, uint dwMessageId, uint dwLanguageId, StringBuilder lpBuffer, int nSize, IntPtr Arguments);

            internal static string GetLastWin32ErrorMessage(int messageId)
            {
                const uint FORMAT_MESSAGE_FROM_SYSTEM = 0x00001000;
                StringBuilder message = new StringBuilder(255);

                FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, IntPtr.Zero, (uint)messageId, 0, message, message.Capacity, IntPtr.Zero);

                return message.ToString();
            }
        }

        private static FileStream OpenReadOnlyFileStream(string path, FileOptions options = FileOptions.SequentialScan)
        {
            return new FileStream(path, FileMode.Open, FileAccess.Read, FileShare.Read, 4096, options);
        }

        private static FileStream OpenNewFileStream(string path, FileOptions options = FileOptions.SequentialScan)
        {
            using (var fs = File.Create(path)) { };
            var attributes = File.GetAttributes(path);
            attributes |= FileAttributes.SparseFile;
            File.SetAttributes(path, attributes);
            return new FileStream(path, FileMode.Open, FileAccess.ReadWrite, FileShare.None, 4096, options);
        }

        private static Dictionary<ulong, byte> GetContentMetaKeyGeneration(string nspPath, KeyConfiguration keyConfig)
        {
            var contentMetaKeyGeneration = new Dictionary<ulong, byte>();
            using (var patchNspReader = new NintendoSubmissionPackageReader(nspPath))
            {
                foreach (var cnmtXml in patchNspReader.ListFileInfo().Where(x => x.Item1.EndsWith(".cnmt.xml")))
                {
                    var cnmtXmlModel = ArchiveReconstructionUtils.ReadXml<ContentMetaModel>(patchNspReader, cnmtXml.Item1, cnmtXml.Item2);
                    var keyGeneration = cnmtXmlModel.ContentList.Single(x => x.Type == NintendoContentMetaConstant.ContentTypeMeta).KeyGeneration;
                    contentMetaKeyGeneration.Add(cnmtXmlModel.GetUInt64Id(), (byte)keyGeneration);
                }
            }
            return contentMetaKeyGeneration;
        }

        private static void VerifyMetaFile(NintendoSubmissionPackageContentInfo info)
        {
            MetaFileReader.VerifyCoreModel(info.MetaFilePath);
            MetaFileReader.VerifyAocModel(info.MetaFilePath);
            ApplicationControl.VerifyApplicationControlProperty(info);
            CardSpecXmlUtils.VerifyCardSpec(info.MetaFilePath);
            return;
        }

        private static void VerifyMetaFile(string metaFilePath, List<Tuple<string, string>> iconFileList)
        {
            MetaFileReader.VerifyCoreModel(metaFilePath);
            MetaFileReader.VerifyAocModel(metaFilePath);
            ApplicationControl.VerifyApplicationControlProperty(metaFilePath, iconFileList);
            CardSpecXmlUtils.VerifyCardSpec(metaFilePath);
            return;
        }

        internal static void CreateFs(Option option)
        {
            var config = option.Config;
            var createFsOption = option.CreateFs;
            Directory.CreateDirectory(Path.GetDirectoryName(createFsOption.OutputFile));

            string adfPath = createFsOption.InputAdfFile;
            if (createFsOption.Format != ArchiveFormatType.Invalid)
            {
                var filter = FilterDescription.ParseFdf(createFsOption.InputFdfPath);
                adfPath = Path.GetDirectoryName(createFsOption.OutputFile) +
                    "\\" + Path.GetFileName(createFsOption.OutputFile) + ".adf";
                switch (createFsOption.Format)
                {
                    case ArchiveFormatType.PartitionFs:
                        {
                            var writer = new PartitionFsAdfWriter(adfPath);
                            writer.Write(createFsOption.InputDir);
                            break;
                        }
                    case ArchiveFormatType.RomFs:
                        {
                            var writer = new RomFsAdfWriter(adfPath);
                            writer.Write(createFsOption.InputDir, filter);
                            break;
                        }
                    default:
                        throw new NotImplementedException();
                }
            }

            if (createFsOption.IsOnlyAdf)
            {
                return;
            }

            ArchiveFormatType archiveFormatType = ContentArchiveLibraryInterface.GetArchiveType(adfPath);
            if (archiveFormatType != ArchiveFormatType.PartitionFs &&
                archiveFormatType != ArchiveFormatType.RomFs)
            {
                throw new FormatException("invalid formatType is indicated by .adf file.");
            }

            using (var fs = OpenNewFileStream(createFsOption.OutputFile))
            {
                ContentArchiveLibraryInterface.CreateArchiveFromAdf(fs, null, adfPath, config);
            }

            if (!createFsOption.IsSaveAdf)
            {
                File.Delete(adfPath);
            }
        }

        internal static void CreateNca(Option option)
        {
            var config = option.Config;
            var createNcaOption = option.CreateNca;
            Directory.CreateDirectory(Path.GetDirectoryName(createNcaOption.OutputFile));

            string adfPath = createNcaOption.InputAdfFile;
            if (createNcaOption.ContentType != null)
            {
                var filter = FilterDescription.ParseFdf(createNcaOption.InputFdfPath);
                adfPath = Path.GetDirectoryName(createNcaOption.OutputFile) +
                    "\\" + Path.GetFileName(createNcaOption.OutputFile) + ".adf";
                var reader = new MetaFileReader(createNcaOption.MetaFilePath, createNcaOption.MetaType);
                Func<Tuple<ulong, byte>> getNcaProgramId = () =>
                {
                    var meta = reader.GetContentMetaList().First() as ApplicationMeta;
                    if (meta != null)
                    {
                        return Tuple.Create(meta.ProgramId, meta.ProgramIndex ?? (byte)0);
                    }
                    return Tuple.Create(reader.GetContentMetaList().First().Id, (byte)0);
                };
                var ncaProgramId = getNcaProgramId();
                NintendoContentAdfWriter writer = new NintendoContentAdfWriter(adfPath, createNcaOption.ContentType, createNcaOption.MetaFilePath, createNcaOption.DescFilePath, createNcaOption.KeyAreaEncryptionKeyIndex, createNcaOption.KeyGeneration, ncaProgramId.Item1, ncaProgramId.Item2, false, createNcaOption.NoEncryption);
                writer.Write(createNcaOption.InputDirs, filter);
            }

            if (createNcaOption.IsOnlyAdf)
            {
                return;
            }

            ArchiveFormatType archiveFormatType = ContentArchiveLibraryInterface.GetArchiveType(adfPath);
            if (archiveFormatType != ArchiveFormatType.NintendoContent)
            {
                throw new FormatException("invalid formatType is indicated by .adf file.");
            }

            using (var fs = OpenNewFileStream(createNcaOption.OutputFile))
            {
                ContentArchiveLibraryInterface.CreateArchiveFromAdf(fs, createNcaOption.MetaType, adfPath, config);
            }

            if (!createNcaOption.IsSaveAdf)
            {
                File.Delete(adfPath);
                File.Delete(Path.GetDirectoryName(adfPath) + "\\" +
                            Path.GetFileNameWithoutExtension(adfPath) + ".code.adf");
                File.Delete(Path.GetDirectoryName(adfPath) + "\\" +
                            Path.GetFileNameWithoutExtension(adfPath) + ".rom.adf");
            }
        }

        internal static void CreateNsp(Option option)
        {
            var config = option.Config;
            var createNspOption = option.CreateNsp;
            GlobalSettings.ErrorUnpublishable = createNspOption.ErrorUnpublishable;
            GlobalSettings.ShowUnpublishableErroFormatXml = createNspOption.ShowUnpublishableErroFormatXml;
            GlobalSettings.IgnoreErrorUnpublishable = createNspOption.IgnoreErrorUnpublishable;
            GlobalSettings.NoCheckDirWarning = createNspOption.NoCheckDirWarning;
            Directory.CreateDirectory(Path.GetDirectoryName(createNspOption.OutputFile));

            var topContentInfo = createNspOption.NspContentInfos[0];
            var tmpDirs = ApplicationControl.UpdateNintendoSubmissionPackageContentInfo(ref topContentInfo);
            createNspOption.NspContentInfos[0] = topContentInfo;
            VerifyMetaFile(createNspOption.NspContentInfos[0]);

            if (createNspOption.GeneratesApplicationControl)
            {
                ApplicationControl.Generate(
                    createNspOption.NspContentInfos[0].MetaFilePath,
                    createNspOption.NspContentInfos[0].IconList,
                    createNspOption.NspContentInfos[0].NxIconList,
                    createNspOption.NspContentInfos[0].NxIconMaxSize,
                    createNspOption.GetApplicationControlGeneratePath());
            }

            string adfPath = createNspOption.InputAdfFile;
            if (adfPath == null)
            {
                var filter = FilterDescription.ParseFdf(createNspOption.InputFdfPath);
                adfPath = Path.GetDirectoryName(createNspOption.OutputFile) +
                    "\\" + Path.GetFileName(createNspOption.OutputFile) + ".adf";
                NintendoSubmissionPackageAdfWriter writer = new NintendoSubmissionPackageAdfWriter(adfPath);

                var originalRomFsListFileInfo = new List<Tuple<string, Int64>>();
                if (createNspOption.OriginalNspPath != null)
                {
                    using (var nspReader = new NintendoSubmissionPackageReader(createNspOption.OriginalNspPath))
                    {
                        var contentId = nspReader.ListFileInfo().Where(x => x.Item1.EndsWith(".programinfo.xml")).Select(x => x.Item1.Replace(".programinfo.xml", string.Empty)).Single();
                        using (var ncaReader = nspReader.OpenNintendoContentArchiveReader(contentId + ".nca", new NcaKeyGenerator(config.GetKeyConfiguration())))
                        {
                            foreach (var i in ncaReader.GetExistentFsIndices())
                            {
                                if (i != (int)NintendoContentArchivePartitionType.Data)
                                {
                                    continue;
                                }
                                var fsReader = ncaReader.OpenFileSystemArchiveReader(i);
                                foreach (var fileInfo in fsReader.ListFileInfo())
                                {
                                    var fragment = fsReader.GetFileFragmentList(fileInfo.Item1).First();
                                    const long RomFsMetaHeaderSize = 512; // TORIAEZU
                                    originalRomFsListFileInfo.Add(Tuple.Create(fileInfo.Item1, fragment.Item1 - RomFsMetaHeaderSize));
                                }
                            }
                        }
                    }
                }

                writer.Write(createNspOption.NspContentInfos, filter, originalRomFsListFileInfo.Count != 0 ? originalRomFsListFileInfo : null);
            }

            if (createNspOption.IsOnlyAdf)
            {
                return;
            }

            ArchiveFormatType archiveFormatType = ContentArchiveLibraryInterface.GetArchiveType(adfPath);
            if (archiveFormatType != ArchiveFormatType.NintendoSubmissionPackage)
            {
                throw new FormatException("invalid formatType is indicated by .adf file.");
            }

            NintendoContentArchiveSource.BuildLog = new NintendoContentArchiveBuildLog(createNspOption.OutputFile, false, null);

            bool hasUnpublishableError = false;
            using (var fs = OpenNewFileStream(createNspOption.OutputFile))
            {
                ContentArchiveLibraryInterface.CreateArchiveFromAdf(fs, null, adfPath, config);

                if (topContentInfo.MetaType == NintendoContentMetaConstant.ContentMetaTypeApplication ||
                    topContentInfo.MetaType == NintendoContentMetaConstant.ContentMetaTypeAddOnContent)
                {
                    // ApplicationかAocの場合にUnpublishableErrorのチェックする（Patchはmakepatchコマンドを使用するのでここではチェックしない）

                    int errorCount = CheckUnpublishableError(fs, createNspOption.ShowUnpublishableErroFormatXml, GlobalSettings.ErrorUnpublishable, GlobalSettings.IgnoreErrorUnpublishable, config);
                    if (errorCount > 0)
                    {
                        // Unpublishableエラーが発生している場合はファイルを削除する
                        // OpenNewFileStreamでファイルを使用しているのでスコープを抜けた後に削除する。
                        hasUnpublishableError = true;
                    }
                }
            }

            if (GlobalSettings.IgnoreErrorUnpublishable == null && hasUnpublishableError)
            {
                File.Delete(createNspOption.OutputFile);
                throw new ArgumentException("Found one or more unpublishable error in nsp. The nsp will be deleted automatically.");
            }

            if (!createNspOption.IsSaveAdf)
            {
                File.Delete(adfPath);
                var files = Directory.GetFiles(Path.GetDirectoryName(adfPath), Path.GetFileNameWithoutExtension(adfPath) + ".c????.*.nca.*adf");
                foreach (var file in files)
                {
                    File.Delete(file);
                }
            }

            if (createNspOption.GeneratesApplicationControl)
            {
                DeleteDirectoryIfExisted(createNspOption.GetApplicationControlGeneratePath());
            }

            foreach (var dirPath in tmpDirs)
            {
                DeleteDirectoryIfExisted(dirPath);
            }
        }

        internal static void CreateNspMeta(Option option)
        {
            var opt = option.CreateNspMeta;

            var metaType = (ContentMetaType)Enum.Parse(typeof(ContentMetaType), opt.MetaType);

            // 管理データ
            if (metaType == ContentMetaType.Application || metaType == ContentMetaType.Patch) // TODO: 他のコンテンツメタタイプにも対応
            {
                VerifyMetaFile(opt.MetaFilePath, opt.IconFileList);
                var pathInfo = ApplicationControl.LoadApplicationControlPropertyPathInfo(opt.MetaFilePath);
                var iconFileList = pathInfo.ValidIconInfoList(opt.IconFileList, pathInfo.IconList);
                var nxIconFileList = pathInfo.ValidIconInfoList(opt.NxIconFileList, pathInfo.NxIconList);

                var controlPath = opt.OutputDirectory + "/control0.ncd/data";
                EnsureDirectory(controlPath);
                ApplicationControl.Generate(opt.MetaFilePath, iconFileList, nxIconFileList, opt.NxIconMaxSize, controlPath, false);

                foreach (var dirPath in pathInfo.DisposableDirs)
                {
                    DeleteDirectoryIfExisted(dirPath);
                }
            }

            // コンテンツメタ
            var contentMetaPath = opt.OutputDirectory + "/meta0.ncd/data";
            DeleteDirectoryIfExisted(contentMetaPath);
            Directory.CreateDirectory(contentMetaPath);
            var contentMeta = new NintendoContentMetaBase(new List<Tuple<ISource, NintendoContentDescriptor>>(), opt.MetaType, opt.MetaFilePath, null);
            using (var fs = OpenNewFileStream(contentMetaPath + "/" + contentMeta.GetEntryName()))
            {
                var bytes = contentMeta.GetBytes();
                fs.Write(bytes, 0, bytes.Length);
            }
        }

        private static void MakeNspdDataDirectoryWithFilter(string outputDir, string dataDir, string fdfFile, bool forceCopy)
        {
            DeleteDirectoryIfExisted(outputDir);
            Directory.CreateDirectory(outputDir);

            var filterRules = FilterDescription.ParseFdf(fdfFile);

            // ディレクトリーを巡回し、ファイルリストを取得
            var listFiles = new List<string>();
            {
                var baseDir = new DirectoryConnector(dataDir, null);
                var targetFileNameList = RomFsAdfWriter.GetTargetFileList(filterRules, baseDir, SearchOption.AllDirectories);
                var filterRuleRegexList = FilterDescription.ConvertFilterRuleStringToRegex(filterRules);
                foreach (var file in targetFileNameList)
                {
                    if ((filterRules != null) && FilterDescription.IsEntryInFilterList(file, baseDir.first, filterRuleRegexList))
                    {
                        continue;
                    }
                    string fileName = file.Replace("\\", "/");
                    listFiles.Add(fileName);
                }
            }

            foreach (var file in listFiles)
            {
                var fileRelPath = file.Replace(dataDir + "/", string.Empty);
                MakeHardLink(outputDir, dataDir, fileRelPath, forceCopy);
            }
        }

        internal static void CreateNspd(Option option)
        {
            var opt = option.CreateNspd;
            GlobalSettings.ErrorUnpublishable = opt.ErrorUnpublishable;
            GlobalSettings.ShowUnpublishableErroFormatXml = opt.ShowUnpublishableErroFormatXml;
            GlobalSettings.IgnoreErrorUnpublishable = opt.IgnoreErrorUnpublishable;

            var metaType = (ContentMetaType)Enum.Parse(typeof(ContentMetaType), opt.MetaType);
            ApplicationControlPropertyPathInfo pathInfo;
            if (metaType == ContentMetaType.Application || metaType == ContentMetaType.Patch)
            {
                VerifyMetaFile(opt.MetaFilePath, opt.IconFileList);
                pathInfo = ApplicationControl.LoadApplicationControlPropertyPathInfo(opt.MetaFilePath);
            }
            else
            {
                pathInfo = new ApplicationControlPropertyPathInfo();
            }
            var iconFileList = pathInfo.ValidIconInfoList(opt.IconFileList, pathInfo.IconList);
            var nxIconFileList = pathInfo.ValidIconInfoList(opt.NxIconFileList, pathInfo.NxIconList);
            var htmlDocumentDirPath = pathInfo.ValidDirPath(opt.HtmlDocumentDirectory, pathInfo.HtmlDocumentDirPath);
            var legalInfoDirPath = pathInfo.ValidDirPath(opt.LegalInformationDirectory, pathInfo.LegalInformationDirPath);
            var accessibleUrlsDirPath = pathInfo.ValidDirPath(opt.AccessibleUrlsDirectory, pathInfo.AccessibleUrlsDirPath);

            if (opt.DoClean)
            {
                DeleteIncludingJunctionDirectory(opt.OutputDirectory);
            }

            // nspd
            EnsureDirectory(opt.OutputDirectory);

            // プログラム
            if (opt.CodeDirectory != null || opt.DataDirectory != null || opt.LogoDirectory != null)
            {
                EnsureDirectory(opt.OutputDirectory + "/program0.ncd");
            }
            MakeJunctionPoint(opt.OutputDirectory + "/program0.ncd/code", opt.CodeDirectory, opt.ForciblyCopyCode);
            if (opt.DataDirectory != null)
            {
                if (opt.InputFdfPath == null)
                {
                    MakeJunctionPoint(opt.OutputDirectory + "/program0.ncd/data", opt.DataDirectory, opt.ForciblyCopyData);
                }
                else
                {
                    MakeNspdDataDirectoryWithFilter(opt.OutputDirectory + "/program0.ncd/data", opt.DataDirectory, opt.InputFdfPath, opt.ForciblyCopyWithFiltering || opt.ForciblyCopyData);
                }
            }

            // ロゴデータが未指定の場合はデフォルトのロゴを展開する。
            var logoOutputDirectory = opt.OutputDirectory + "/program0.ncd/logo";
            if (opt.LogoDirectory == null)
            {
                DeleteDirectoryIfExisted(logoOutputDirectory);
                LogoManager.ExtractDefaultLogoData(logoOutputDirectory);
            }
            else
            {
                MakeJunctionPoint(logoOutputDirectory, opt.LogoDirectory);
            }

            // Html ドキュメント
            if (htmlDocumentDirPath != null || accessibleUrlsDirPath != null)
            {
                EnsureDirectory(opt.OutputDirectory + "/htmlDocument0.ncd/data");
            }
            MakeJunctionPoint(opt.OutputDirectory + "/htmlDocument0.ncd/data/html-document", htmlDocumentDirPath, opt.ForciblyCopyHtmlDocument);
            MakeJunctionPoint(opt.OutputDirectory + "/htmlDocument0.ncd/data/accessible-urls", accessibleUrlsDirPath, pathInfo.IsAccessibleUrlsSpecifiedFilePath || opt.ForciblyCopyAccessibleUrls);

            // Legal Information
            if (legalInfoDirPath != null || opt.LegalInformationZipPath != null)
            {
                string legalInformationDirectory = opt.OutputDirectory + "/legalInformation0.ncd";
                EnsureDirectory(legalInformationDirectory);
                legalInformationDirectory = legalInformationDirectory + "/data";

                if (legalInfoDirPath != null)
                {
                    MakeJunctionPoint(legalInformationDirectory, legalInfoDirPath, pathInfo.IsLegalInformationSpecifiedFilePath || opt.ForciblyCopyLegalInformation);
                }
                else if (opt.LegalInformationZipPath != null)
                {
                    DeleteDirectoryIfExisted(legalInformationDirectory);
                    EnsureDirectory(legalInformationDirectory);
                    ZipFile.ExtractToDirectory(opt.LegalInformationZipPath, legalInformationDirectory);
                }
            }

            // 管理データ
            var controlNcd = opt.OutputDirectory + "/control0.ncd";
            var controlPath = controlNcd + "/data";
            EnsureDirectory(controlNcd);
            MakeJunctionPoint(controlPath, opt.ControlDirectory, opt.ForciblyCopyControl);
            if (!Directory.Exists(controlPath))
            {
                EnsureDirectory(controlPath);
            }
            ApplicationControl.Generate(opt.MetaFilePath, iconFileList, nxIconFileList, opt.NxIconMaxSize, controlPath, false);

            // コンテンツメタ
            var contentMetaPath = opt.OutputDirectory + "/meta0.ncd/data";
            DeleteDirectoryIfExisted(contentMetaPath);
            EnsureDirectory(contentMetaPath);
            var contentMeta = new NintendoContentMetaBase(new List<Tuple<ISource, NintendoContentDescriptor>>(), opt.MetaType, opt.MetaFilePath, null);
            using (var fs = OpenNewFileStream(contentMetaPath + "/" + contentMeta.GetEntryName()))
            {
                var bytes = contentMeta.GetBytes();
                fs.Write(bytes, 0, bytes.Length);
            }

            // nspd_root
            var nspdRootFileName = Path.GetFileNameWithoutExtension(opt.OutputDirectory) + ".nspd_root";
            var directoryName = Path.GetDirectoryName(opt.OutputDirectory);
            using (var fs = File.Create(Path.Combine(directoryName, nspdRootFileName)))
            {
            }

            foreach (var dirPath in pathInfo.DisposableDirs)
            {
                DeleteDirectoryIfExisted(dirPath);
            }
        }

        internal static void CreateNacp(Option option)
        {
            using (FileStream fs = new FileStream(option.CreateNacp.OutputFilePath, FileMode.Create, FileAccess.Write))
            {
                var bytes = ApplicationControl.MakeApplicationControlPropertyBytes(option.CreateNacp.MetaFilePath);
                fs.Write(bytes, 0, bytes.Length);
            }
        }

        internal static void MakePatch(Option option)
        {
            var config = option.Config;
            var patch = option.MakePatch;

            Directory.CreateDirectory(Path.GetDirectoryName(patch.OutputFile));

            var willOptimize = patch.PreviousFilePath != null;
            var willMakeDelta = patch.WillOutputWithDelta;
            var willMergeDelta = patch.WillMergeDeltaContent;
            var nonOptimizedPatchPath = willOptimize ? Path.ChangeExtension(patch.OutputFile, ".intermediate.nsp") : patch.OutputFile;
            var nonMergedPatchPath = willMergeDelta ? Path.ChangeExtension(patch.OutputFile, ".intermediate2.nsp") : patch.OutputFile;
            var mergedPatchPath = willMergeDelta ? patch.OutputFile : "";
            var deltaPath = willMakeDelta ? Path.ChangeExtension(patch.OutputFile, ".delta.nsp") : "";

            NintendoContentArchiveSource.BuildLog = new NintendoContentArchiveBuildLog(patch.OriginalFilePath, patch.OutputFile, patch.IsSaveBuildLog, patch.CacheDirectory);

            var removeFiles = new List<string>();
            if (willOptimize)
            {
                removeFiles.Add(nonOptimizedPatchPath);
            }
            if (willMergeDelta)
            {
                removeFiles.Add(nonMergedPatchPath);
            }
            if (patch.MinimumMatchingSize != 0)
            {
                IndirectStorageSource.SetMinimumMatchingSize(patch.MinimumMatchingSize);
            }
            IndirectStorageSource.SetStronglyOptimizeSize(patch.StronglyOptimizeSize);

            if (patch.NeedsSparse)
            {
                SparseStorageSource.SetOption(patch.SparseBlockSize, patch.SparseEraseSize);
            }

            if ((!willOptimize && willMakeDelta) || (!willMakeDelta && willMergeDelta))
            {
                throw new ArgumentException("Invalid option combination");
            }

            {
                using (var outStream = OpenNewFileStream(nonOptimizedPatchPath))
                using (var originalStream = OpenReadOnlyFileStream(patch.OriginalFilePath))
                using (var currentStream = OpenReadOnlyFileStream(patch.CurrentFilePath))
                using (var previousStream = !string.IsNullOrEmpty(patch.PreviousFilePath) ? OpenReadOnlyFileStream(patch.PreviousFilePath) : null)
                {
                    if (patch.IsNcaFile)
                    {
                        ContentArchiveLibraryInterface.MakeNcaPatch(outStream, originalStream, currentStream, patch.DescFilePath, config);
                    }
                    else
                    {
                        ContentArchiveLibraryInterface.MakeNspPatch(outStream, originalStream, currentStream, previousStream, patch.MetaFilePath, patch.DescFilePath, config);
                    }
                }
            }

            if (willOptimize)
            {
                using (var outStream = OpenNewFileStream(nonMergedPatchPath))
                using (var previousStream = OpenReadOnlyFileStream(patch.PreviousFilePath))
                using (var currentStream = OpenReadOnlyFileStream(nonOptimizedPatchPath))
                {
                    if (patch.NeedsDefragment)
                    {
                        RelocatedIndirectStorageSource.SetDefragmentSize(patch.DefragmentSize);
                    }
                    ContentArchiveLibraryInterface.MakeNspIscPatch(outStream, previousStream, currentStream, patch.DescFilePath, config);
                }
            }
            if (patch.NeedsSparse)
            {
                var nonSparsePatchPath = Path.ChangeExtension(nonMergedPatchPath, ".noncompaction.nsp");
                var sparseOriginalPath = Path.ChangeExtension(patch.OutputFile, ".original.nsp");

                File.Move(nonMergedPatchPath, nonSparsePatchPath);

                NintendoContentArchiveSource.BuildLog = new NintendoContentArchiveBuildLog(patch.OriginalFilePath, sparseOriginalPath, patch.IsSaveBuildLog, null);

                SparseStorageSource.SetOption(patch.SparseBlockSize, patch.SparseEraseSize);

                using (var outOriginalStream = OpenNewFileStream(sparseOriginalPath))
                using (var outPatchStream = OpenNewFileStream(nonMergedPatchPath))
                using (var originalStream = OpenReadOnlyFileStream(patch.OriginalFilePath))
                using (var patchStream = OpenReadOnlyFileStream(nonSparsePatchPath))
                using (var previousStream = !string.IsNullOrEmpty(patch.PreviousFilePath) ? OpenReadOnlyFileStream(patch.PreviousFilePath) : null)
                {
                    ContentArchiveLibraryInterface.SparsifyNsp(outOriginalStream, outPatchStream, originalStream, patchStream, previousStream, config);
                }

                removeFiles.Add(nonSparsePatchPath);
            }
            if (willMakeDelta)
            {
                NintendoContentArchiveSource.BuildLog = new NintendoContentArchiveBuildLog(deltaPath, willMakeDelta && patch.IsSaveBuildLog, null);

                var adfPath = Path.Combine(Path.GetDirectoryName(deltaPath), Path.GetFileName(deltaPath) + ".adf");
                var metaPath = patch.DeltaMetaPath;
                if (metaPath == null)
                {
                    metaPath = Path.Combine(Path.GetDirectoryName(deltaPath), Path.GetFileName(deltaPath) + ".nmeta");
                    DeltaFragmentContentGenerator.MakeDeltaMetaFromDestination(metaPath, nonMergedPatchPath);
                }
                var writer = new NintendoSubmissionPackageAdfWriter(adfPath);
                writer.WriteDelta(patch.PreviousFilePath, nonMergedPatchPath, metaPath, -1, DeltaArchive.DeltaCommandSizeMax, false);

                ArchiveFormatType archiveFormatType = ContentArchiveLibraryInterface.GetArchiveType(adfPath);
                if (archiveFormatType != ArchiveFormatType.NintendoSubmissionPackageForDelta)
                {
                    throw new FormatException("invalid formatType is indicated by .adf file.");
                }

                var patchContentMetaKeyGeneration = GetContentMetaKeyGeneration(nonMergedPatchPath, config.GetKeyConfiguration());

                using (var fs = OpenNewFileStream(deltaPath))
                {
                    ContentArchiveLibraryInterface.CreateDeltaFromAdf(fs, adfPath, patchContentMetaKeyGeneration, config, false);
                }
            }
            if (willMergeDelta)
            {
                using (var outStream = OpenNewFileStream(mergedPatchPath))
                using (var patchStream = OpenReadOnlyFileStream(nonMergedPatchPath))
                using (var deltaStream = OpenReadOnlyFileStream(deltaPath))
                {
                    ContentArchiveLibraryInterface.MergeDeltaContentIntoPatch(outStream, patchStream, deltaStream, config);
                }
            }
            if (!patch.KeepIntermediates)
            {
                removeFiles.ForEach(p => File.Delete(p));
            }
        }

        internal static void OptimizePatch(Option option)
        {
            var config = option.Config;
            var patch = option.OptimizePatch;

            NintendoContentArchiveSource.BuildLog = new NintendoContentArchiveBuildLog(patch.PreviousFilePath, patch.OutputPath, patch.IsSaveBuildLog, patch.CacheDirectory);

            Directory.CreateDirectory(Path.GetDirectoryName(patch.OutputPath));

            using (var outStream = OpenNewFileStream(patch.OutputPath))
            using (var previousStream = OpenReadOnlyFileStream(patch.PreviousFilePath))
            using (var currentStream = OpenReadOnlyFileStream(patch.CurrentFilePath))
            {
                IndirectStorageSource.SetStronglyOptimizeSize(patch.StronglyOptimizeSize);
                if (patch.NeedsDefragment)
                {
                    RelocatedIndirectStorageSource.SetDefragmentSize(patch.DefragmentSize);
                }
                ContentArchiveLibraryInterface.MakeNspIscPatch(outStream, previousStream, currentStream, patch.DescFilePath, config);
            }
        }

        internal static void MakeDelta(Option option)
        {
            if (option.MakeDelta.SubType == MakeDeltaOption.OutputType.Binary)
            {
                MakeDeltaBinary(option);
            }
            else
            {
                MakeDeltaNsp(option);
            }
        }

        private static void MakeDeltaBinary(Option option)
        {
            var makeDeltaOption = option.MakeDelta;

            Directory.CreateDirectory(Path.GetDirectoryName(makeDeltaOption.OutputFile));

            bool sourceIsNsp = Path.GetExtension(makeDeltaOption.SourceFilePath) == ".nsp";
            bool destinationIsNsp = Path.GetExtension(makeDeltaOption.DestinationFilePath) == ".nsp";
            if (sourceIsNsp != destinationIsNsp)
            {
                throw new ArgumentException("previous and current version archive file must be .nsp file.");
            }

            Action<Stream, ISource, ISource> writeDelta = delegate (Stream output, ISource source, ISource destionation)
            {
                var sourceInfo = DeltaInfo.ReadSourceInfo(source);

                var deltaCommands = DeltaInfo.ExtractDeltaCommands(
                    source,
                    destionation,
                    DeltaArchive.DeltaCommandSizeMax,
                    DeltaArchive.DeltaCommandCompareSize,
                    null);
                DeltaInfo.CombineNeighbor(deltaCommands, DeltaArchive.DeltaCommandMergeSeekSizeMax, makeDeltaOption.DeltaCommandSizeMax);

                ContentArchiveLibraryInterface.MakeDelta(
                    output,
                    destionation,
                    sourceInfo,
                    deltaCommands);
            };

            // warning CA2202 (System.ObjectDisposedException の回避への対応) のため、try ... finally を使う
            Stream sourceStream = null;
            Stream destinationStream = null;

            try
            {
                using (var outStream = OpenNewFileStream(makeDeltaOption.OutputFile))
                {
                    sourceStream = OpenReadOnlyFileStream(makeDeltaOption.SourceFilePath);
                    destinationStream = OpenReadOnlyFileStream(makeDeltaOption.DestinationFilePath);

                    if (sourceIsNsp && destinationIsNsp)
                    {
                        Action<Action<NintendoSubmissionPackageReader, NintendoSubmissionPackageReader>> doStreams = (action) =>
                        {
                            using (var sourceNsp = new NintendoSubmissionPackageReader(sourceStream))
                            {
                                sourceStream = null;
                                using (var destinationNsp = new NintendoSubmissionPackageReader(destinationStream))
                                {
                                    destinationStream = null;

                                    action.Invoke(sourceNsp, destinationNsp);
                                };
                            }
                        };

                        doStreams.Invoke((sourceNsp, destinationNsp) =>
                        {
                            var sourceMetas = DeltaFragmentContentGenerator.ReadPatchContentMetaInNsp(sourceNsp);
                            var destinationMetas = DeltaFragmentContentGenerator.ReadPatchContentMetaInNsp(destinationNsp);

                            foreach (var destinationMeta in destinationMetas)
                            {
                                // 更新先コンテントメタに対応する元コンテントメタを見つける
                                var sourceMeta = sourceMetas.FindAll(p => p.Id == destinationMeta.Id);

                                // 現時点では 1:1 で対応するコンテントメタがない場合はエラーとする
                                if (sourceMeta.Count() != 1)
                                {
                                    throw new ArgumentException("Invalid nsp pair");
                                }
                                // コンテントメタにひもづくコンテンツ群の、マッチングを行う
                                var matchings = DeltaFragmentContentGenerator.FindMatchings(sourceMeta[0], destinationMeta);
                                foreach (var matching in matchings)
                                {
                                    var sourceNca = matching.first != null ? new FileSystemArchvieFileSource(sourceNsp, DeltaFragmentContentGenerator.GetNcaName(matching.first)) : null;
                                    var destinationNca = new FileSystemArchvieFileSource(destinationNsp, DeltaFragmentContentGenerator.GetNcaName(matching.second));

                                    if (matching.first.Type == "Program")
                                    {
                                        writeDelta(outStream, sourceNca, destinationNca);
                                    }
                                }
                            }
                        });
                    }
                    else
                    {
                        writeDelta(
                            outStream,
                            new StreamSource(sourceStream, 0, sourceStream.Length),
                            new StreamSource(destinationStream, 0, destinationStream.Length));
                    }
                }
            }
            finally
            {
                if (sourceStream != null)
                {
                    sourceStream.Dispose();
                }
                if (destinationStream != null)
                {
                    destinationStream.Dispose();
                }
            }
        }

        private static void MakeDeltaNsp(Option option)
        {
            var config = option.Config;
            var delta = option.MakeDelta;

            NintendoContentArchiveSource.BuildLog = new NintendoContentArchiveBuildLog(delta.OutputFile, delta.WillSaveBuildLog, null);

            string metaPath = delta.MetaFilePath;
            if (metaPath == null)
            {
                metaPath = Path.Combine(Path.GetDirectoryName(delta.OutputFile), Path.GetFileName(delta.OutputFile) + ".nmeta");
                DeltaFragmentContentGenerator.MakeDeltaMetaFromDestination(metaPath, delta.DestinationFilePath);
            }

            string adfPath = Path.GetDirectoryName(delta.OutputFile) +
                    "\\" + Path.GetFileName(delta.OutputFile) + ".adf";
            var writer = new NintendoSubmissionPackageAdfWriter(adfPath);
            writer.WriteDelta(delta.SourceFilePath, delta.DestinationFilePath, metaPath, -1, delta.DeltaCommandSizeMax, delta.WillSaveDelta);

            ArchiveFormatType archiveFormatType = ContentArchiveLibraryInterface.GetArchiveType(adfPath);
            if (archiveFormatType != ArchiveFormatType.NintendoSubmissionPackageForDelta)
            {
                throw new FormatException("invalid formatType is indicated by .adf file.");
            }

            var patchContentMetaKeyGeneration = GetContentMetaKeyGeneration(delta.DestinationFilePath, config.GetKeyConfiguration());

            using (var fs = OpenNewFileStream(delta.OutputFile))
            {
                ContentArchiveLibraryInterface.CreateDeltaFromAdf(fs, adfPath, patchContentMetaKeyGeneration, config, false);
            }
        }
        internal static void MergeDeltaContent(Option option)
        {
            var config = option.Config;
            var merge = option.MergeDeltaContent;

            using (var patch = OpenReadOnlyFileStream(merge.PatchFilePath))
            using (var merging = OpenReadOnlyFileStream(merge.MergingFilePath))
            using (var output = OpenNewFileStream(merge.OutputFile))
            {
                ContentArchiveLibraryInterface.MergeDeltaContentIntoPatch(output, patch, merging, config);
            }
        }

        internal static void ExtractNsp(Option option)
        {
            EnsureDirectory(option.ExtractNsp.OutputDirectory);

            using (var readerStream = OpenReadOnlyFileStream(option.ExtractNsp.InputNspFile))
            {
                var reader = new NintendoSubmissionPackageReader(readerStream);
                foreach (var info in reader.ListFileInfo())
                {
                    var fileName = info.Item1;
                    var fileSize = info.Item2;

                    var extractedPath = option.ExtractNsp.OutputDirectory + "/" + fileName;
                    Console.WriteLine("Extracting {0}.", extractedPath);

                    FsUtil.WriteFile(reader, extractedPath, fileName, fileSize);
                }
            }
            if (option.ExtractNsp.RenameBasedOnContentType)
            {
                var directory = new DirectoryInfo(option.ExtractNsp.OutputDirectory);
                var xml = XDocument.Load(directory.GetFiles("*.cnmt.xml").First().FullName);
                var map = xml.Descendants("Content").Select(p => new Tuple<string, string>(p.Element("Type").Value, p.Element("Id").Value));
                foreach (var m in map)
                {
                    var source = (m.Item1 == "Meta") ? string.Format("{0}.cnmt.nca", m.Item2) : string.Format("{0}.nca", m.Item2);
                    var dest = string.Format("{0}.nca", m.Item1);
                    File.Move(Path.Combine(option.ExtractNsp.OutputDirectory, source), Path.Combine(option.ExtractNsp.OutputDirectory, dest));
                }

            }
        }

        internal static bool IsDirectoryName(string path)
        {
            return path.EndsWith("/");
        }

        internal static void ExtractNsave(Option option)
        {
            using (var readerStream = OpenReadOnlyFileStream(option.ExtractNsave.InputNsaveFile))
            {
                var reader = new NintendoSubmissionPackageReader(readerStream);
                foreach (var info in reader.ListFileInfo())
                {
                    var fileName = info.Item1;
                    var fileSize = info.Item2;

                    var extractedPath = option.ExtractNsave.OutputDirectory + "/" + fileName;
                    Console.WriteLine("Extracting {0}.", extractedPath);

                    // セーブデータのエクスポート用に本来ないディクレトリを作成する
                    if (!Directory.Exists(Path.GetDirectoryName(extractedPath)))
                    {
                        Directory.CreateDirectory(Path.GetDirectoryName(extractedPath));
                    }

                    if (IsDirectoryName(extractedPath))
                    {
                        continue;
                    }

                    FsUtil.WriteFile(reader, extractedPath, fileName, fileSize);
                }
            }
        }

        // 備考：targetEntryPath が null でなければ、そのパスのエントリのみ extract する
        private static bool ExtractNca(NintendoContentArchiveReader originalReader, NintendoContentArchiveReader currentReader, string outputDirectoryPath, Regex targetEntryRegex, ExtractMode mode)
        {
            return ExtractNca(originalReader, currentReader, outputDirectoryPath, targetEntryRegex, mode, "");
        }

        private static bool ExtractNca(NintendoContentArchiveReader originalReader, NintendoContentArchiveReader currentReader, string outputDirectoryPath, Regex targetEntryRegex, ExtractMode mode, string ncaName)
        {
            bool isFileExtracted = false;

            foreach (var ncaInfo in currentReader.ListFsInfo(originalReader))
            {
                var fsIndex = ncaInfo.Item1;
                var fsReader = currentReader.OpenFileSystemArchiveReader(fsIndex, originalReader);
                foreach (var fsInfo in fsReader.ListFileInfo())
                {
                    var fileName = fsInfo.Item1;
                    var fileSize = fsInfo.Item2;
                    var targetPath = string.Format("{0}{1}fs{2}/{3}", ncaName, String.IsNullOrEmpty(ncaName) ? "" : "/", fsIndex, fileName);
                    // target 指定の extract だった場合はパスが一致しないとスキップ
                    if (mode != ExtractMode.All && !targetEntryRegex.IsMatch(targetPath))
                    {
                        continue;
                    }

                    // 単独ファイルの extract ではディレクトリを掘らず、 outputDirectoryPath 直下に出力
                    var extractedPath = Path.Combine(outputDirectoryPath, Path.GetFileName(fileName));
                    if (mode != ExtractMode.TargetStringPath)
                    {
                        int startIndex = String.IsNullOrEmpty(ncaName) ? 0 : ncaName.Length + 1;
                        extractedPath = Path.Combine(outputDirectoryPath, targetPath.Substring(startIndex));
                        Directory.CreateDirectory(Path.GetDirectoryName(extractedPath));
                    }
                    Console.WriteLine("  Extracting {0}\t({1} byte)", targetPath, fileSize);
                    FsUtil.WriteFile(fsReader, extractedPath, fileName, fileSize);
                    isFileExtracted = true;
                }
            }

            return isFileExtracted;
        }

        internal static void Extract(Option option)
        {
            var config = option.Config;
            var keyConfig = config.GetKeyConfiguration();
            var keyGenerator = new NcaKeyGenerator(keyConfig);
            bool isFileExtracted = false;

            var outDirectory = option.Extract.OutputDirectory;
            var inputFileName = Path.GetFileName(option.Extract.InputFile);

            var targetEntryPath = option.Extract.TargetEntryPath;
            var targetRegex = option.Extract.TargetEntryRegex;
            if (option.Extract.Mode == ExtractMode.TargetStringPath)
            {
                targetRegex = new Regex("^" + targetEntryPath + "$");
            }

            try
            {
                Directory.CreateDirectory(outDirectory);
                using (var originalStream = string.IsNullOrEmpty(option.Extract.OriginalFile) ? null : OpenReadOnlyFileStream(option.Extract.OriginalFile))
                using (var stream = OpenReadOnlyFileStream(option.Extract.InputFile))
                {
                    // 入力が nsp ファイルのとき
                    if (Path.GetExtension(inputFileName) == ".nsp")
                    {
                        var originalReader = originalStream == null ? null : new NintendoSubmissionPackageReader(originalStream);
                        var reader = new NintendoSubmissionPackageReader(stream);

                        foreach (var info in reader.ListFileInfo())
                        {
                            var fileName = info.Item1;
                            var fileSize = info.Item2;

                            // ターゲットエントリパスのオプションが有効でフルパスの場合：ターゲットの可能性がないパスは枝切りする
                            if (option.Extract.Mode == ExtractMode.TargetStringPath && FsUtil.CheckShouldSkipStartsWithPath(targetEntryPath, fileName))
                            {
                                continue;
                            }
                            var extractedPath = Path.Combine(outDirectory, fileName);
                            if (Path.GetExtension(fileName) == ".nca")
                            {
                                Console.WriteLine("{0}: {1}\t({2} byte)", (option.Extract.Mode == ExtractMode.All) ? "Target" : "Searching", fileName, fileSize);
                                if (option.Extract.Mode == ExtractMode.TargetStringPath && targetRegex.IsMatch(fileName))
                                {
                                    // .nca そのものをオプション指定によって直接 extract する場合
                                    FsUtil.WriteFile(reader, extractedPath, fileName, fileSize);
                                    isFileExtracted = true;
                                    continue;
                                }
                                else
                                {
                                    try
                                    {
                                        // .nca 内部のものを extract する場合
                                        NintendoContentArchiveReader originalNcaReader = null;
                                        NintendoContentArchiveReader ncaReader = null;
                                        if (originalReader != null)
                                        {
                                            var nca = new NintendoSubmissionPackageComparer(originalReader, reader, keyConfig);
                                            nca.FindOriginal(fileName, out originalNcaReader, out ncaReader);
                                        }
                                        else
                                        {
                                            ncaReader = reader.OpenNintendoContentArchiveReader(fileName, keyGenerator);
                                        }
                                        TicketUtility.SetExternalKey(ref ncaReader, reader);
                                        var outputDirectoryForExtractNca = option.Extract.Mode == ExtractMode.TargetStringPath ? outDirectory : extractedPath;

                                        isFileExtracted |= ExtractNca(originalNcaReader, ncaReader, outputDirectoryForExtractNca, targetRegex, option.Extract.Mode, fileName);
                                    }
                                    catch (Exception e)
                                    {
                                        Console.WriteLine(e.Message);
                                        Console.WriteLine("Warning: failed to open nca ('{0}')", fileName);
                                    }
                                }
                            }
                            else if (option.Extract.Mode == ExtractMode.All || (option.Extract.Mode != ExtractMode.All && targetRegex.IsMatch(fileName)))
                            {
                                // .nca ではない直下のエントリ
                                Console.WriteLine("Extracting {0}\t({1} byte)", fileName, fileSize);
                                if (option.Extract.ConsoleDump && fileName.EndsWith(".xml"))
                                {
                                    var fileSource = new FileSystemArchvieFileSource(reader, fileName);
                                    using (var fs = new SourceBasedStream(fileSource))
                                    using (var sr = new StreamReader(fs, Encoding.UTF8))
                                    {
                                        while (!sr.EndOfStream)
                                        {
                                            var line = sr.ReadLine();
                                            Console.WriteLine(line);
                                        }
                                    }
                                }
                                else
                                {
                                    FsUtil.WriteFile(reader, extractedPath, fileName, fileSize);
                                }
                                isFileExtracted = true;
                            }
                        }
                    }
                    // 入力が nca ファイルのとき
                    else if (Path.GetExtension(inputFileName) == ".nca")
                    {
                        var originalNcaReader = originalStream == null ? null : new NintendoContentArchiveReader(originalStream, keyGenerator);
                        var ncaReader = new NintendoContentArchiveReader(stream, keyGenerator);
                        isFileExtracted |= ExtractNca(originalNcaReader, ncaReader, outDirectory, targetRegex, option.Extract.Mode);
                    }
                    else
                    {
                        throw new ArgumentException("input archive file must be .nca or .nsp file.");
                    }
                }
            }
            catch (ExtractEntryNotFoundException) { }

            if (isFileExtracted == false)
            {
                if (option.Extract.Mode != ExtractMode.All)
                {
                    var targetPath = targetRegex.ToString();
                    if (option.Extract.Mode == ExtractMode.TargetStringPath && targetPath.Length > 2)
                    {
                        targetPath = targetPath.Substring(1, targetPath.Length - 2);
                    }
                    Console.Error.WriteLine("[Error] Target entry '{0}' not found ({1}).", targetPath, option.Extract.Mode.ToString());
                }
                else
                {
                    Console.Error.WriteLine("[Error] Nothing to be extracted.");
                }
            }
        }

        internal static readonly char ReplacePathDelimiter = '|';
        internal static readonly char[] ReplaceRuleListDelimiter = { '\t' };
        private static void Replace(Option option)
        {
            var config = option.Config;

            var filenameExtension = Path.GetExtension(option.Replace.InputFile);
            if (filenameExtension != ".nsp" && filenameExtension != ".nca")
            {
                throw new ArgumentException("input file must be .nsp or .nca file.");
            }

            Directory.CreateDirectory(option.Replace.OutputDirectory);
            var outputPath = option.Replace.OutputDirectory + "/" + Path.GetFileNameWithoutExtension(option.Replace.InputFile) + "_replaced" + filenameExtension;

            // replace ルールはファイルから読む場合があるのでその解釈
            var targetEntryPathList = new List<string>();
            var inEntryFilePathList = new List<string>();
            if (String.IsNullOrEmpty(option.Replace.InputEntryFilePath))
            {
                // ファイルから読む場合（InputEntryFilePath がないときは TargetEntryPath がルール記述ファイル）
                using (var sr = new StreamReader(option.Replace.TargetEntryPath, System.Text.Encoding.UTF8))
                {
                    while (sr.Peek() >= 0)
                    {
                        string line = sr.ReadLine();
                        var rulePair = line.Split(ReplaceRuleListDelimiter, StringSplitOptions.RemoveEmptyEntries);
                        if (rulePair.Count() != 2)
                        {
                            throw new InvalidDataException(string.Format("rule description should be 'TARGET_PATH<TAB>INPUT_PATH' ('{0}')", line));
                        }
                        targetEntryPathList.Add(rulePair[0]);
                        inEntryFilePathList.Add(rulePair[1]);
                    }
                }
            }
            else
            {
                // 正しく指定されている場合
                targetEntryPathList.AddRange(option.Replace.TargetEntryPath.Split(ReplacePathDelimiter));
                inEntryFilePathList.AddRange(option.Replace.InputEntryFilePath.Split(ReplacePathDelimiter));
            }

            using (var inStream = OpenReadOnlyFileStream(option.Replace.InputFile))
            {
                var inEntryStreamList = new List<Stream>();
                try
                {
                    foreach (var inEntry in inEntryFilePathList)
                    {
                        if(inEntry == "null")
                        {
                            inEntryStreamList.Add(new MemoryStream());
                        }
                        else
                        {
                            inEntryStreamList.Add(OpenReadOnlyFileStream(inEntry));
                        }
                    }

                    bool hasUnpublishableError = false;
                    using (var outStream = OpenNewFileStream(outputPath))
                    {
                        bool isNsp = filenameExtension == ".nsp";

                        if (inEntryStreamList.Count != targetEntryPathList.Count)
                        {
                            throw new ArgumentException(string.Format("Number of input file and replace target do not match. ({0} vs. {1})", inEntryStreamList.Count, targetEntryPathList.Count));
                        }

                        if (isNsp)
                        {
                            ContentArchiveLibraryInterface.ModifyNintendoSubmissionPackageArchive(
                                outStream, inStream,
                                inEntryStreamList, targetEntryPathList,
                                option.Replace.DescFilePath, option.Replace.NroDirectoryPath, config);

                            int errorCount = CheckUnpublishableError(outStream, option.Replace.ShowUnpublishableErroFormatXml, option.Replace.ErrorUnpublishable, option.Replace.IgnoreErrorUnpublishable, config);
                            if (errorCount > 0)
                            {
                                // Unpublishableエラーが発生している場合はファイルを削除する
                                // OpenNewFileStreamでファイルを使用しているのでスコープを抜けた後に削除する。
                                hasUnpublishableError = true;
                            }
                        }
                        else
                        {
                            ContentArchiveLibraryInterface.ModifyNintendoContentArchive(
                                outStream, inStream,
                                inEntryStreamList, targetEntryPathList,
                                option.Replace.DescFilePath, option.Replace.NroDirectoryPath, config,
                                (int)NintendoContentArchiveUtil.GetAlignmentType(option.Replace.MetaType, option.Replace.ContentType));
                        }
                    }
                    if (option.Replace.IgnoreErrorUnpublishable == null && hasUnpublishableError)
                    {
                        File.Delete(outputPath);
                        throw new ArgumentException("Found one or more unpublishable error in nsp. The nsp will be deleted automatically.");
                    }
                }
                finally
                {
                    foreach (var inEntryStream in inEntryStreamList)
                    {
                        inEntryStream.Dispose();
                    }
                }
            }
        }
        private class RomFsListFileInfoComparer : IComparer<RomFsListFileInfo>
        {
            public int Compare(RomFsListFileInfo s1, RomFsListFileInfo s2)
            {
                return string.CompareOrdinal(s1.Item1, s2.Item1);
            }
        }

        private static void ReplaceNspMeta(Option option)
        {
            var config = option.Config;
            var commandOption = option.ReplaceNspMeta;

            var intermediateNsp = commandOption.WorkDirectory + "/intermediate.nsp";

            // 置換するデータを抽出するために新しいnmetaを使用してnspを作成
            try
            {
                // プログラムがないエラー(Issue 10-813)のメッセージの抑制の為のリストを作成
                var ignoreListPath = commandOption.WorkDirectory + "/ignore_list.txt";
                using (var sw = new StreamWriter(ignoreListPath, false, Encoding.UTF8))
                {
                    sw.WriteLine("10-813");
                }

                // creatensp 用のオプションを作成
                string[] args =
                {
                    "creatensp",
                    "-o",
                    intermediateNsp,
                    "--save-adf",
                    "--type",
                    "Application",
                    "--meta",
                    commandOption.MetaFilePath,
                    "--ignore-unpublishable-error",
                    ignoreListPath,
                };
                option.Parse(args);
                CreateNsp(option);
            }
            catch (Exception ex)
            {
                Directory.Delete(commandOption.WorkDirectory, true);
                throw ex;
            }

            string replaceRuleFile;
            // replace の準備
            ContentArchiveLibraryInterface.PrepareNspReplaceApplicationControlProperty(out replaceRuleFile, commandOption.WorkDirectory, commandOption.InputFile, intermediateNsp, option.Config);

            // replace 機能で置換
            try
            {
                List<string> args = new List<string>
                {
                    "replace",
                    commandOption.InputFile,
                    "-o",
                    commandOption.OutputDirectory,
                    replaceRuleFile,
                    "--desc",
                    commandOption.DescFilePath,
                };

                // replacenspmeta オプションの --ignore-unpublishable-error を引き継ぐ
                if (option.ReplaceNspMeta.IgnoreErrorUnpublishable != null)
                {
                    args.Add("--ignore-unpublishable-error");
                    if (option.ReplaceNspMeta.IgnoreErrorUnpublishable != string.Empty)
                    {
                        args.Add(option.ReplaceNspMeta.IgnoreErrorUnpublishable);
                    }
                }

                option.Parse(args.ToArray()); // keyconfig が設定されている可能性があるので、 option を使い回す
                Replace(option);
            }
            catch (Exception ex)
            {
                Directory.Delete(commandOption.WorkDirectory, true);
                throw ex;
            }

            Directory.Delete(commandOption.WorkDirectory, true);
        }

        private static void ListNca(NintendoContentArchiveReader originalReader, NintendoContentArchiveReader reader, string ncaName, Regex targetEntryRegex, bool patchedOnly, bool showPatchFragment)
        {
            int maxPathLength = 10;
            ncaName = string.IsNullOrEmpty(ncaName) ? "" : ncaName + "/";
            var deletedFileInfoList = new List<Tuple<string, Int64, int>>();
            foreach (var ncaInfo in reader.ListFsInfo(originalReader))
            {
                var fsIndex = ncaInfo.Item1;
                NintendoContentArchiveFsHeaderInfo fsHeaderInfo = null;
                var fsReader = reader.OpenFileSystemArchiveReader(fsIndex, ref fsHeaderInfo, originalReader);
                // originalFileListInfo の調整
                List<Tuple<string, Int64>> originalFileListInfo = null;
                if (originalReader != null && originalReader.HasFsInfo(fsIndex))
                {
                     var originalFsReader = originalReader.OpenFileSystemArchiveReader(fsIndex);
                     originalFileListInfo = originalFsReader.ListFileInfo();
                     originalFileListInfo.Sort(new RomFsListFileInfoComparer());
                }

                IndirectStorageArchiveReader indirectReader = null;
                if (fsHeaderInfo.GetPatchInfo() != null && patchedOnly)
                {
                    indirectReader = new IndirectStorageArchiveReader(originalReader, reader, fsIndex);
                }
                var fileListInfo = fsReader.ListFileInfo();
                if (fileListInfo != null)
                {
                    fileListInfo.Sort(new RomFsListFileInfoComparer());
                }
                int foundIndex = 0;
#if false
                Console.WriteLine("{0}fs{1}\nSecure Value: {2}\nGeneration: {3}", ncaName, fsIndex, fsHeaderInfo.GetSecureValue(), fsHeaderInfo.GetGeneration());
#endif
                foreach (var fsInfo in fileListInfo)
                {
                    var fileName = fsInfo.Item1;
                    var fileSize = fsInfo.Item2;

                    List<IndirectStorageEntry> fragments = null;
                    bool isFileInPatch = false;
                    if (indirectReader != null)
                    {
                        var range = fsReader.GetFileFragmentList(fileName).Single();
                        fragments = indirectReader.GetFragmentListOnDifference((long)fsHeaderInfo.GetHashTargetOffset() + range.Item1, range.Item2);
                        isFileInPatch = fragments.Where(x => x.index == IndirectStorageIndex.Patch).Any();
                    }

                    // 各ファイル元 ROM との一致確認を行いつつ数え上げ
                    if (originalFileListInfo != null && foundIndex < originalFileListInfo.Count)
                    {
                        var index = originalFileListInfo.BinarySearch(foundIndex, originalFileListInfo.Count - foundIndex, fsInfo, new RomFsListFileInfoComparer());
                        if (index >= 0) // Add されたファイルではない
                        {
                            if (patchedOnly && (index > foundIndex + 1)) // この区間にあるファイルは消された
                            {
                                for (var deletedIndex = foundIndex + 1; deletedIndex < index; deletedIndex++)
                                {
                                    deletedFileInfoList.Add(Tuple.Create(originalFileListInfo[deletedIndex].Item1, originalFileListInfo[deletedIndex].Item2, fsIndex));
                                }
                            }
                            foundIndex = index;
                        }
                        else // Add されたファイルである
                        {
                            isFileInPatch = true;
                        }
                    }
                    else // Add されたファイルである
                    {
                        isFileInPatch = true;
                    }

                    if(indirectReader != null && !isFileInPatch)
                    {
                        continue;
                    }

                    var path = string.Format("{0}fs{1}/{2}", ncaName, fsIndex, fileName);
                    if (targetEntryRegex != null && !targetEntryRegex.IsMatch(path))
                    {
                        continue;
                    }
                    maxPathLength = Math.Max(maxPathLength, path.Length);
                    Console.WriteLine("{0}\t({1} byte)", path.PadRight(maxPathLength + 1), fileSize);

                    if (showPatchFragment && isFileInPatch && fragments != null)
                    {
                        char[] trimmed = { ' ', ',' };
                        var originalStr = "[Original]\n(offset, size) = ";
                        bool foundOriginal = false;
                        bool foundPatch = false;
                        var patchStr = "[Patch]\n(offset, size) = ";
                        foreach (var fragment in fragments)
                        {
                            if (fragment.index == IndirectStorageIndex.Original)
                            {
                                originalStr += string.Format("({0}, {1}), ", fragment.offset, fragment.size);
                                foundOriginal = true;
                            }
                            else if (fragment.index == IndirectStorageIndex.Patch)
                            {
                                patchStr += string.Format("({0}, {1}), ", fragment.offset, fragment.size);
                                foundPatch = true;
                            }
                        }
                        if (foundOriginal)
                        {
                            Console.WriteLine(originalStr.TrimEnd(trimmed));
                        }
                        if (foundPatch)
                        {
                            Console.WriteLine(patchStr.TrimEnd(trimmed));
                        }
                    }
                }

                if (patchedOnly && (originalFileListInfo != null) && (foundIndex + 1 < originalFileListInfo.Count))
                {
                    for (var deletedIndex = foundIndex + 1; deletedIndex < originalFileListInfo.Count; deletedIndex++)
                    {
                        deletedFileInfoList.Add(Tuple.Create(originalFileListInfo[deletedIndex].Item1, originalFileListInfo[deletedIndex].Item2, fsIndex));
                    }
                }
            }
            if (patchedOnly && deletedFileInfoList.Count > 0)
            {
                Console.WriteLine("Listing removed files from original...");
                foreach (var deletedFileInfo in deletedFileInfoList)
                {
                    var fileName = deletedFileInfo.Item1;
                    var fileSize = deletedFileInfo.Item2;
                    var fsIndex  = deletedFileInfo.Item3;
                    var path = string.Format("fs{0}/{1}", fsIndex, fileName);
                    if (targetEntryRegex != null && !targetEntryRegex.IsMatch(path))
                    {
                        continue;
                    }

                    maxPathLength = Math.Max(maxPathLength, path.Length);
                    Console.WriteLine("{0}\t({1} byte)", path.PadRight(maxPathLength + 1), fileSize);
                }
                Console.WriteLine("Done.");
            }
        }

        private static void DiffPatchedNca(NintendoContentArchiveReader originalReader, NintendoContentArchiveReader prevReader, NintendoContentArchiveReader reader, string ncaName)
        {
            int maxPathLength = 10;
            ncaName = string.IsNullOrEmpty(ncaName) ? "" : ncaName + "/";

            foreach (var ncaInfo in reader.ListFsInfo(originalReader))
            {
                var fsIndex = ncaInfo.Item1;
                NintendoContentArchiveFsHeaderInfo fsHeaderInfo = null;
                var fsReader = reader.OpenFileSystemArchiveReader(fsIndex, ref fsHeaderInfo, originalReader);
                var fileListInfo = fsReader.ListFileInfo();
                if (fileListInfo != null)
                {
                    fileListInfo.Sort(new RomFsListFileInfoComparer());
                }

                List<Tuple<string, Int64>> prevFileListInfo = null;
                NintendoContentArchiveFsHeaderInfo prevFsHeaderInfo = null;
                var prevFsReader = prevReader.OpenFileSystemArchiveReader(fsIndex, ref prevFsHeaderInfo, originalReader);
                prevFileListInfo = prevFsReader.ListFileInfo();
                if (prevFileListInfo != null)
                {
                    prevFileListInfo.Sort(new RomFsListFileInfoComparer());
                }

                int foundIndex = 0;
                foreach (var fsInfo in fileListInfo)
                {
                    var fileName = fsInfo.Item1;
                    var fileSize = fsInfo.Item2;
                    Int64 prevFileSize = 0;
                    bool isFileDifferent = false;
                    bool isAddedFile = false;

                    // 各ファイル元 ROM との一致確認を行いつつ数え上げ
                    if (prevFileListInfo != null && foundIndex < prevFileListInfo.Count)
                    {
                        var index = prevFileListInfo.BinarySearch(foundIndex, prevFileListInfo.Count - foundIndex, fsInfo, new RomFsListFileInfoComparer());
                        if (index >= 0) // Add されたファイルではない
                        {
                            if (index > foundIndex + 1) // この区間にあるファイルは消された
                            {
                                for (var deletedIndex = foundIndex + 1; deletedIndex < index; deletedIndex++)
                                {
                                    var delPath = string.Format("{0}fs{1}/{2}", ncaName, fsIndex, prevFileListInfo[deletedIndex].Item1);
                                    maxPathLength = Math.Max(maxPathLength, delPath.Length);
                                    Console.WriteLine("- {0}\t({1} byte)", delPath.PadRight(maxPathLength + 1), prevFileListInfo[deletedIndex].Item2);
                                }
                            }

                            prevFileSize = prevFileListInfo[index].Item2;
                            if (fileSize != prevFileSize)
                            {
                                isFileDifferent = true;
                            }
                            else
                            {
                                if (FsUtil.CompareFile(fsReader, prevFsReader, fileName, fileSize) == false)
                                {
                                    isFileDifferent = true;
                                }
                            }
                            foundIndex = index;
                        }
                        else // Add されたファイルである
                        {
                            isAddedFile = true;
                        }
                    }
                    else // Add されたファイルである
                    {
                        isAddedFile = true;
                    }

                    if(!isFileDifferent && !isAddedFile)
                    {
                        continue;
                    }

                    var path = string.Format("{0}fs{1}/{2}", ncaName, fsIndex, fileName);
                    maxPathLength = Math.Max(maxPathLength, path.Length);

                    string fileBytes;
                    if ((isAddedFile == false) && (fileSize != prevFileSize))
                    {
                        fileBytes = string.Format("({0} -> {1} byte)", prevFileSize, fileSize);
                    }
                    else
                    {
                        fileBytes = string.Format("({0} byte)", fileSize);
                    }

                    Console.WriteLine("{0} {1}\t{2}", (isAddedFile) ? "+" : "!", path.PadRight(maxPathLength + 1), fileBytes);
                }

                if ((prevFileListInfo != null) && (foundIndex + 1 < prevFileListInfo.Count))
                {
                    for (var deletedIndex = foundIndex + 1; deletedIndex < prevFileListInfo.Count; deletedIndex++)
                    {
                        var delPath = string.Format("{0}fs{1}/{2}", ncaName, fsIndex, prevFileListInfo[deletedIndex].Item1);
                        maxPathLength = Math.Max(maxPathLength, delPath.Length);
                        Console.WriteLine("- {0}\t({1} byte)", delPath.PadRight(maxPathLength + 1), prevFileListInfo[deletedIndex].Item2);
                    }
                }
            }
        }

        internal static void List(Option option)
        {
            var config = option.Config;
            var keyGenerator = new NcaKeyGenerator(config.GetKeyConfiguration());
            var targetRegex = option.List.TargetEntryRegex;

            Action<NintendoSubmissionPackageReader> listNspCommonImpl = (NintendoSubmissionPackageReader nspReader) =>
            {
                foreach (var info in nspReader.ListFileInfo().Where(x => targetRegex == null || targetRegex.IsMatch(x.Item1)))
                {
                    var fileName = info.Item1;
                    var fileSize = info.Item2;
                    Console.WriteLine("{0, -45}\t({1} byte)", fileName, fileSize);
                }
                Console.WriteLine("---------------------------------------------");
            };

            Action<NintendoContentArchiveReader, NintendoContentArchiveReader, string> listNcaImpl = (NintendoContentArchiveReader ncaReader, NintendoContentArchiveReader originalNcaReader, string ncaName) =>
            {
                try
                {
                    ListNca(originalNcaReader, ncaReader, ncaName, targetRegex, option.List.PatchedOnly, option.List.ShowPatchFragment);
                }
                catch (Exception e)
                {
                    if (ncaName == null)
                    {
                        throw e;
                    }
                    else
                    {
                        Console.WriteLine(e.Message);
                        Console.WriteLine("Failed to open nca ('{0}'). Skipped it.", ncaName);
                    }
                }
            };

            Action<Stream> listXciImpl = (Stream xciStream) =>
            {
                byte[] rootHash;
                {
                    Console.WriteLine("[headerInfo]");
                    var xciHeaderOffset = (long)XciInfo.CardKeyAreaPageCount * XciInfo.PageSize;
                    var xciHeaderStream = new SubStream(xciStream, xciHeaderOffset, (long)XciInfo.CardHeaderPageCount * XciInfo.PageSize);
                    var xciHeaderData = new byte[xciHeaderStream.Length];
                    {
                        xciHeaderStream.Read(xciHeaderData, 0, (int)xciHeaderStream.Length);
                    }
                    XciUtils.DumpHeader(XciUtils.DecryptHeader(xciHeaderData));
                    rootHash = XciUtils.GetRootPartitionHash(xciHeaderData);
                }

                var bodyOffset = (long)XciInfo.NormalAreaStartPageAddress * XciInfo.PageSize;
                var xciBodyStream = new SubStream(xciStream, bodyOffset, xciStream.Length - bodyOffset);
                var rootReader = new XciReader(xciBodyStream, rootHash);
                foreach (var rootEntry in rootReader.ListFileInfo())
                {
                    var rootEntryName = rootEntry.Item1;
                    Console.WriteLine("[{0}]", rootEntryName);
                    var partitionReader = rootReader.OpenXciPartitionReader(rootEntryName);
                    foreach (var entry in partitionReader.ListFileInfo())
                    {
                        var ncaName = entry.Item1;
                        var ncaSize = entry.Item2;
                        Console.WriteLine("{0, -45}\t({1} byte)", ncaName, ncaSize);
                    }
                }
            };

            var inputFileName = Path.GetFileName(option.List.InputFile);
            if (Path.GetExtension(inputFileName) == ".nsp" || Path.GetExtension(inputFileName) == ".nspu")
            {
                using (var originalNspReader = string.IsNullOrEmpty(option.List.OriginalFile) ? null : new NintendoSubmissionPackageReader(option.List.OriginalFile))
                {
                    if (originalNspReader == null)
                    {
                        using (var nspReader = new NintendoSubmissionPackageReader(option.List.InputFile))
                        {
                            listNspCommonImpl(nspReader);
                            foreach (var info in nspReader.ListFileInfo().FindAll(x => Path.GetExtension(x.Item1) == ".nca"))
                            {
                                var ncaName = info.Item1;
                                var ncaReader = nspReader.OpenNintendoContentArchiveReader(ncaName, keyGenerator);
                                TicketUtility.SetExternalKey(ref ncaReader, nspReader);
                                listNcaImpl(ncaReader, null, ncaName);
                            }
                        }
                    }
                    else
                    {
                        using (var patchedNspReader = new PatchedNintendoSubmissionPackageReader(option.List.InputFile, originalNspReader))
                        {
                            listNspCommonImpl(patchedNspReader);
                            foreach (var info in patchedNspReader.ListFileInfo().FindAll(x => Path.GetExtension(x.Item1) == ".nca"))
                            {
                                var ncaName = info.Item1;
                                var ncaReaders = patchedNspReader.OpenPatchableNintendoContentArchiveReader(ncaName, keyGenerator);
                                listNcaImpl(ncaReaders.Item1, ncaReaders.Item2, ncaName);
                            }
                        }
                    }
                }
            }
            else if (Path.GetExtension(inputFileName) == ".nca")
            {
                using (var originalStream = string.IsNullOrEmpty(option.List.OriginalFile) ? null : OpenReadOnlyFileStream(option.List.OriginalFile))
                using (var stream = OpenReadOnlyFileStream(option.List.InputFile))
                {
                    var ncaReader = new NintendoContentArchiveReader(stream, keyGenerator);
                    var originalNcaReader = originalStream == null ? null : new NintendoContentArchiveReader(originalStream, keyGenerator);
                    TicketUtility.SetExternalKey(ref ncaReader, option.List.TicketFile);
                    listNcaImpl(ncaReader, originalNcaReader, null);
                }
            }
            else if (Path.GetExtension(inputFileName) == ".xci" || Path.GetExtension(inputFileName) == ".xcie" || Path.GetExtension(inputFileName) == ".xcir")
            {
                using (var stream = OpenReadOnlyFileStream(option.List.InputFile))
                {
                    listXciImpl(stream);
                }
            }
            else
            {
                throw new ArgumentException("input archive file must be .nca or .nsp file.");
            }
        }

        internal static void DiffPatch(Option option)
        {
            var config = option.Config;
            var keyGenerator = new NcaKeyGenerator(config.GetKeyConfiguration());

            using (var originalNspReader = new NintendoSubmissionPackageReader(option.DiffPatch.OriginalFile))
            {
                using (var prevPatchedNspReader = new PatchedNintendoSubmissionPackageReader(option.DiffPatch.PreviousPatchFile, originalNspReader))
                {
                    using (var patchedNspReader = new PatchedNintendoSubmissionPackageReader(option.DiffPatch.PatchFile, originalNspReader))
                    {
                        foreach (var info in patchedNspReader.ListFileInfo().FindAll(x => Path.GetExtension(x.Item1) == ".nca"))
                        {
                            string contentType;
                            UInt64 contentMetaId;
                            byte contentIdOffset;
                            var ncaName = info.Item1;
                            var ncaReaders = patchedNspReader.OpenPatchableNintendoContentArchiveReader(ncaName, keyGenerator, out contentType, out contentIdOffset, out contentMetaId);

                            var contentId = ArchiveReconstructionUtils.GetSpecifiedContentId(prevPatchedNspReader, contentMetaId, contentType, contentIdOffset);
                            if (contentId != null)
                            {
                                var prevNcaName = string.Format("{0}{1}", contentId, (contentType == "Meta") ? ".cnmt.nca" : ".nca");
                                var prevNcaReaders = prevPatchedNspReader.OpenPatchableNintendoContentArchiveReader(prevNcaName, keyGenerator);

                                try
                                {
                                    DiffPatchedNca(ncaReaders.Item2, prevNcaReaders.Item1, ncaReaders.Item1, ncaName);
                                }
                                catch (Exception e)
                                {
                                    if (ncaName == null)
                                    {
                                        throw e;
                                    }
                                    else
                                    {
                                        Console.WriteLine(e.Message);
                                        Console.WriteLine("Failed to open nca ('{0}'). Skipped it.", ncaName);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        private static ContentMetaModel GetContentMetaModel(NintendoSubmissionPackageReader nspReader)
        {
            var entry = nspReader.ListFileInfo().Find(s => s.Item1.EndsWith(".cnmt.xml"));
            return ArchiveReconstructionUtils.ReadModel(nspReader, entry.Item1, entry.Item2);
        }

        private static NintendoContentArchiveReader GetContentArchiveReader(NintendoSubmissionPackageReader nspReader, ContentMetaModel metaModel, string contentType, KeyConfiguration config)
        {
            var content = metaModel.ContentList.Find(m => m.Type == contentType);
            if (content != null)
            {
                return GetContentArchiveReader(nspReader, content.Id + ".nca", config);
            }
            return null;
        }

        private static NintendoContentArchiveReader GetContentArchiveReader(NintendoSubmissionPackageReader nspReader, string ncaFileName, KeyConfiguration config)
        {
            var ncaReader = nspReader.OpenNintendoContentArchiveReader(ncaFileName, new NcaKeyGenerator(config));
            try
            {
                TicketUtility.SetExternalKey(ref ncaReader, nspReader);
            }
            catch
            {
                ncaReader.Dispose();
                throw;
            }
            return ncaReader;
        }

        internal static void Verify(Option option)
        {
            var verify = option.Verify;
            var config = option.Config;
            var keyConfig = config.GetKeyConfiguration();

            // 単一 nsp のチェック
            if (string.IsNullOrEmpty(verify.PreviousFilePath))
            {
                using (var nspReader = new NintendoSubmissionPackageReader(verify.InputFilePath))
                {
                    foreach (var fileInfo in nspReader.ListFileInfo().Where(x => x.Item1.EndsWith(".nca")))
                    {
                        using (var ncaReader = GetContentArchiveReader(nspReader, fileInfo.Item1, keyConfig))
                        {
                            NintendoContentArchiveReader newNcaReader = ncaReader;
                            TicketUtility.SetExternalKey(ref newNcaReader, nspReader);

                            newNcaReader.Verify();

                            if(newNcaReader != ncaReader)
                            {
                                newNcaReader.Dispose();
                            }
                        }
                    }
                }
            }
            // パッチ間のチェック
            else
            {
                // NOTE: InputFilePath の nsp はすでに別で検証済みとして単一チェックはしない
                using (var currentNspReader = new NintendoSubmissionPackageReader(verify.InputFilePath))
                using (var previousNspReader = new NintendoSubmissionPackageReader(verify.PreviousFilePath))
                {
                    var previousModel = GetContentMetaModel(previousNspReader);
                    var currentModel = GetContentMetaModel(currentNspReader);

                    // Program の調査
                    using (var previousReader = GetContentArchiveReader(previousNspReader, previousModel, "Program", keyConfig))
                    using (var currentReader = GetContentArchiveReader(currentNspReader, currentModel, "Program", keyConfig))
                    {
                        if (previousReader != null && currentReader != null)
                        {
                            currentReader.Verify(previousReader);
                        }
                    }

                    // HtmlDocument の調査
                    using (var previousReader = GetContentArchiveReader(previousNspReader, previousModel, "HtmlDocument", keyConfig))
                    using (var currentReader = GetContentArchiveReader(currentNspReader, currentModel, "HtmlDocument", keyConfig))
                    {
                        if (previousReader != null && currentReader != null)
                        {
                            currentReader.Verify(previousReader);
                        }
                    }
                }
            }
        }

        internal static void MergeNsp(Option option)
        {
            var streams = option.MergeNsp.InputFiles.Select(x => (Stream)OpenReadOnlyFileStream(x)).ToList();
            try
            {
                using (var outStream = OpenNewFileStream(option.MergeNsp.OutputFile))
                {
                    ContentArchiveLibraryInterface.MergeNintendoSubmissionPackageArchive(outStream, streams);
                }
            }
            finally
            {
                foreach (var stream in streams)
                {
                    if (stream != null)
                    {
                        ((IDisposable)stream).Dispose();
                    }
                }
            }
        }

        internal static void SplitNsp(Option option)
        {
            EnsureDirectory(option.SplitNsp.OutputDirectory);

            using (var inputStream = OpenReadOnlyFileStream(option.SplitNsp.InputNspFile))
            {
                var nspReader = new NintendoSubmissionPackageReader(inputStream);
                bool found = false;
                foreach (var cnmtXml in nspReader.ListFileInfo().Where(x => x.Item1.EndsWith(".cnmt.xml")))
                {
                    var cnmtXmlModel = ArchiveReconstructionUtils.ReadXml<ContentMetaModel>(nspReader, cnmtXml.Item1, cnmtXml.Item2);
                    if (option.SplitNsp.TargetContentMetaId.HasValue &&
                        cnmtXmlModel.GetUInt64Id() != option.SplitNsp.TargetContentMetaId.Value)
                    {
                        continue;
                    }
                    if (option.SplitNsp.TargetVersion.HasValue &&
                        cnmtXmlModel.Version.Value != option.SplitNsp.TargetVersion.Value)
                    {
                        continue;
                    }

                    found = true;

                    using (var outputStream = OpenNewFileStream(Path.Combine(option.SplitNsp.OutputDirectory, string.Format("{0}_v{1}.nsp", cnmtXmlModel.Id, cnmtXmlModel.Version))))
                    {
                        ContentArchiveLibraryInterface.SplitNintendoSubmissionPackageArchive(outputStream, inputStream, cnmtXmlModel.GetUInt64Id(), cnmtXmlModel.Version.Value);
                    }
                }
                if (!found)
                {
                    throw new ArgumentException(string.Format("There is no content that matches --id or --version"));
                }
            }
        }

        private static string GetProdEncryptionOutputFileNameBase(ProdEncryptionOption option)
        {
            string baseFileName;
            if (!string.IsNullOrEmpty(option.InputMultiApplicationCardIndicatorNspFile))
            {
                baseFileName = option.InputMultiApplicationCardIndicatorNspFile;
            }
            else
            {
                baseFileName = option.InputNspFile;
            }
            return option.OutputDirectory + "/" + Path.GetFileNameWithoutExtension(baseFileName);
        }

        internal static void ProdEncryption(Option option)
        {
            var config = option.Config;

            if (option.ProdEncryption.CreateXci)
            {
                var outputXciPath = GetProdEncryptionOutputFileNameBase(option.ProdEncryption) + "_prod.xci";
                var outputXciXmlPath = GetProdEncryptionOutputFileNameBase(option.ProdEncryption) + "_prod.xci.result.xml";
                var outputDebugPath = GetProdEncryptionOutputFileNameBase(option.ProdEncryption) + "_prod.xci.debug.cnmt.xml";

                var outputXciePath = GetProdEncryptionOutputFileNameBase(option.ProdEncryption) + "_prod.xcie";
                var outputXcieXmlPath = GetProdEncryptionOutputFileNameBase(option.ProdEncryption) + "_prod.xcie.result.xml";

                if (option.ProdEncryption.UseKeyForRepairTool)
                {
                    config.GetKeyConfiguration().KeyIndexForEncryptedXci = EncryptedXciKeyIndex.Xcir;
                    outputXciePath = outputXciePath.Replace("xcie", "xcir");
                    outputXcieXmlPath = outputXcieXmlPath.Replace("xcie", "xcir");
                }

                using (var outStream = OpenNewFileStream(outputXciPath))
                using (var outXmlStream = OpenNewFileStream(outputXciXmlPath))
                using (var outDebugStream = config.DebugConfig.EnableContentMetaBinaryExport ? OpenNewFileStream(outputDebugPath) : null)
                using (var inUppStream = option.ProdEncryption.InputUppNspFile != null ? OpenReadOnlyFileStream(option.ProdEncryption.InputUppNspFile) : null)
                {
                    if (string.IsNullOrEmpty(option.ProdEncryption.InputMultiApplicationCardIndicatorNspFile))
                    {
                        using (var inStream = OpenReadOnlyFileStream(option.ProdEncryption.InputNspFile))
                        using (var inPatchStream = option.ProdEncryption.InputPatchNspFile != null ? OpenReadOnlyFileStream(option.ProdEncryption.InputPatchNspFile) : null)
                        using (var inAocStream = option.ProdEncryption.InputAocNspFile != null ? OpenReadOnlyFileStream(option.ProdEncryption.InputAocNspFile) : null)
                        {
                            ContentArchiveLibraryInterface.ProdEncryptNintendoSubmissionPackageArchiveForXci(outStream, outXmlStream, inStream, inUppStream, inPatchStream, inAocStream, option.ProdEncryption.LaunchFlags, option.ProdEncryption.NoPaddingXci, option.ProdEncryption.KeepGeneration, config, outDebugStream);
                        }
                    }
                    else
                    {
                        var streams = option.ProdEncryption.InputNspFiles.Select(x => (Stream)OpenReadOnlyFileStream(x)).ToList();
                        try
                        {
                            using (var indicatorStream = OpenReadOnlyFileStream(option.ProdEncryption.InputMultiApplicationCardIndicatorNspFile))
                            {
                                ContentArchiveLibraryInterface.ProdEncryptNintendoSubmissionPackageArchiveForXci(outStream, outXmlStream, indicatorStream, streams, inUppStream, option.ProdEncryption.LaunchFlags, option.ProdEncryption.NoPaddingXci, option.ProdEncryption.KeepGeneration, config, outDebugStream);
                            }
                        }
                        finally
                        {
                            foreach (var stream in streams)
                            {
                                if (stream != null)
                                {
                                    ((IDisposable)stream).Dispose();
                                }
                            }
                        }
                    }
                    if (option.ProdEncryption.CheckIntegrity)
                    {
                        // ビット化け検知 & 完全性検証
                        var integrityChecker = new IntegrityChecker(config.GetKeyConfiguration());
                        integrityChecker.CheckProdEncryptedXciArchive(outStream, outXmlStream);
                    }
                }

                if (option.ProdEncryption.CreateXcie)
                {
                    using (var inStream = OpenReadOnlyFileStream(outputXciPath))
                    using (var inXmlStream = OpenReadOnlyFileStream(outputXciXmlPath))
                    using (var outStream = OpenNewFileStream(outputXciePath, FileOptions.SequentialScan))
                    using (var outXmlStream = OpenNewFileStream(outputXcieXmlPath))
                    {
                        ContentArchiveLibraryInterface.EncryptXci(outStream, outXmlStream, inStream, inXmlStream, config);
                    }
                }
            }
            else
            {
                var outputNspPath = GetProdEncryptionOutputFileNameBase(option.ProdEncryption) + "_prod.nsp";
                var outputNspXmlPath = GetProdEncryptionOutputFileNameBase(option.ProdEncryption) + "_prod.nsp.result.xml";

                var outputNspuPath = GetProdEncryptionOutputFileNameBase(option.ProdEncryption) + "_prod.nspu";
                var outputNspuXmlPath = GetProdEncryptionOutputFileNameBase(option.ProdEncryption) + "_prod.nspu.result.xml";


                uint? requiredSystemVersionSubstitute = option.ProdEncryption.RequiredSystemVersion;
                if (option.ProdEncryption.InputUppNspFile != null)
                {
                    // Upp が指定されている場合は、Upp の Version を元に requiredSystemVersionSubstitute を設定
                    using (var uppReader = new NintendoSubmissionPackageReader(option.ProdEncryption.InputUppNspFile))
                    {
                        var xml = uppReader.ReadFile(UpdatePartitionUtils.UpdatePartitionXmlName, 0, uppReader.GetFileSize(UpdatePartitionUtils.UpdatePartitionXmlName));
                        var xmlReader = new UpdatePartitionUtils.UpdatePartitionXmlReader(xml);
                        requiredSystemVersionSubstitute = xmlReader.GetVersionTrancatedToRelease();
                    }
                }

                var desiredSafeKeyGeneration = option.ProdEncryption.DesiredSafeKeyGen;
                if (option.ProdEncryption.InputUppForDesiredSafeKeyGenNspFile != null)
                {
                    // Upp が指定されている場合は、Upp の最新の鍵世代を設定
                    using (var uppReader = new NintendoSubmissionPackageReader(option.ProdEncryption.InputUppForDesiredSafeKeyGenNspFile))
                    {
                        desiredSafeKeyGeneration = UpdatePartitionUtils.GetLatestKeyGeneration(uppReader);
                    }
                }

                using (var inStream = OpenReadOnlyFileStream(option.ProdEncryption.InputNspFile))
                using (var outStream = OpenNewFileStream(outputNspPath))
                using (var outXmlStream = OpenNewFileStream(outputNspXmlPath))
                {
                    ContentArchiveLibraryInterface.ProdEncryptNintendoSubmissionPackageArchive(outStream, outXmlStream, inStream, requiredSystemVersionSubstitute, option.ProdEncryption.ExcludeContents, option.ProdEncryption.KeyGenerations, desiredSafeKeyGeneration, config);
                    if (option.ProdEncryption.CheckIntegrity)
                    {
                        // ビット化け検知 & 完全性検証
                        var integrityChecker = new IntegrityChecker(config.GetKeyConfiguration());
                        integrityChecker.CheckProdEncryptedNintendoSubmissionPackageArchive(outStream);
                    }
                }

                bool needToCreateNspu = false;
                using (var inStream = OpenReadOnlyFileStream(outputNspPath))
                {
                    var reader = new NintendoSubmissionPackageReader(inStream);
                    if (reader.ListFileInfo().Where(x => Regex.IsMatch(x.Item1, @"[0-9a-fA-F]*\.tik")).Any())
                    {
                        needToCreateNspu = true;
                    }
                }

                if (option.ProdEncryption.CreateNspu && needToCreateNspu)
                {
                    using (var inStream = OpenReadOnlyFileStream(outputNspPath))
                    using (var inXmlStream = OpenReadOnlyFileStream(outputNspXmlPath))
                    using (var outStream = OpenNewFileStream(outputNspuPath, FileOptions.SequentialScan))
                    using (var outXmlStream = OpenNewFileStream(outputNspuXmlPath))
                    {
                        ContentArchiveLibraryInterface.RemoveTicket(outStream, outXmlStream, inStream, inXmlStream);
                    }
                }
            }
        }

        private static string GetProdEncryptionPatchOutputFileNameBase(ProdEncryptionPatchOption option)
        {
            return option.OutputDirectory + "/" + Path.GetFileNameWithoutExtension(option.InputNspFile);
        }

        internal static void ProdEncryptionPatch(Option option)
        {
            var config = option.Config;

            var outputNspPath = GetProdEncryptionPatchOutputFileNameBase(option.ProdEncryptionPatch) + "_prod.nsp";
            var outputNspXmlPath = GetProdEncryptionPatchOutputFileNameBase(option.ProdEncryptionPatch) + "_prod.nsp.result.xml";
            var outputNspuPath = GetProdEncryptionPatchOutputFileNameBase(option.ProdEncryptionPatch) + "_prod.nspu";
            var outputNspuXmlPath = GetProdEncryptionPatchOutputFileNameBase(option.ProdEncryptionPatch) + "_prod.nspu.result.xml";

            uint? requiredSystemVersionSubstitute = option.ProdEncryptionPatch.RequiredSystemVersion;
            if (option.ProdEncryptionPatch.InputUppNspFile != null)
            {
                // Upp が指定されている場合は、Upp の Version を元に requiredSystemVersionSubstitute を設定
                using (var uppReader = new NintendoSubmissionPackageReader(option.ProdEncryptionPatch.InputUppNspFile))
                {
                    var xml = uppReader.ReadFile(UpdatePartitionUtils.UpdatePartitionXmlName, 0, uppReader.GetFileSize(UpdatePartitionUtils.UpdatePartitionXmlName));
                    var xmlReader = new UpdatePartitionUtils.UpdatePartitionXmlReader(xml);
                    requiredSystemVersionSubstitute = xmlReader.GetVersionTrancatedToRelease();
                }
            }

            var desiredSafeKeyGeneration = option.ProdEncryptionPatch.DesiredSafeKeyGen;
            if (option.ProdEncryptionPatch.InputUppForDesiredSafeKeyGenNspFile != null)
            {
                // Upp が指定されている場合は、Upp の最新の鍵世代を設定
                using (var uppReader = new NintendoSubmissionPackageReader(option.ProdEncryptionPatch.InputUppForDesiredSafeKeyGenNspFile))
                {
                    desiredSafeKeyGeneration = UpdatePartitionUtils.GetLatestKeyGeneration(uppReader);
                }
            }

            using (var inPatchStream = OpenReadOnlyFileStream(option.ProdEncryptionPatch.InputNspFile))
            using (var inOriginalStream = OpenReadOnlyFileStream(option.ProdEncryptionPatch.InputOriginalNspFile))
            using (var outStream = OpenNewFileStream(outputNspPath))
            using (var outXmlStream = OpenNewFileStream(outputNspXmlPath))
            {
                if (option.ProdEncryptionPatch.InputPreviousProdNspFile != null)
                {
                    using (var inPreviousProdStream = OpenReadOnlyFileStream(option.ProdEncryptionPatch.InputPreviousProdNspFile))
                    {
                        ContentArchiveLibraryInterface.ProdEncryptPatch(outStream, outXmlStream, inPatchStream, inOriginalStream, inPreviousProdStream, requiredSystemVersionSubstitute, desiredSafeKeyGeneration, option.ProdEncryptionPatch.SizeThresholdForKeyGenerationUpdate, config);
                    }
                }
                else
                {
                    ContentArchiveLibraryInterface.ProdEncryptPatch(outStream, outXmlStream, inPatchStream, inOriginalStream, null, requiredSystemVersionSubstitute, desiredSafeKeyGeneration, option.ProdEncryptionPatch.SizeThresholdForKeyGenerationUpdate, config);
                }

                if (option.ProdEncryptionPatch.InputPreviousProdNspFile == null && option.ProdEncryptionPatch.CheckIntegrity)
                {
                    using (var inOriginalProdStream = OpenReadOnlyFileStream(option.ProdEncryptionPatch.InputOriginalProdNspFile))
                    {
                        // ビット化け検知 & 完全性検証
                        var integrityChecker = new IntegrityChecker(config.GetKeyConfiguration());
                        integrityChecker.CheckProdEncryptedNintendoSubmissionPackageArchive(outStream, inOriginalProdStream);
                    }
                }
            }

            var deltaNspPath = Path.ChangeExtension(outputNspPath, ".delta.nsp");
            var deltaNotMergedNspPath = Path.ChangeExtension(outputNspPath, ".deltaNotMerged.nsp");

            if (option.ProdEncryptionPatch.InputPreviousProdNspFile != null)
            {
                // TORIAEZU: デルタ生成・マージで利用するため製品化後のパッチのコンテンツメタの鍵世代を取得
                var patchContentMetaKeyGeneration = GetContentMetaKeyGeneration(outputNspPath, config.GetKeyConfiguration());

                if (File.Exists(deltaNotMergedNspPath))
                {
                    File.Delete(deltaNotMergedNspPath);
                }
                File.Move(outputNspPath, deltaNotMergedNspPath);

                var adfPath = Path.Combine(Path.GetDirectoryName(deltaNspPath), Path.GetFileName(deltaNspPath) + ".adf");
                var metaPath = Path.Combine(Path.GetDirectoryName(deltaNspPath), Path.GetFileName(deltaNspPath) + ".nmeta");
                DeltaFragmentContentGenerator.MakeDeltaMetaFromDestination(metaPath, deltaNotMergedNspPath);

                var writer = new NintendoSubmissionPackageAdfWriter(adfPath);
                writer.WriteDelta(option.ProdEncryptionPatch.InputPreviousProdNspFile, deltaNotMergedNspPath, metaPath, -1, DeltaArchive.DeltaCommandSizeMax, false);

                using (var outputDeltaStream = OpenNewFileStream(deltaNspPath))
                {
                    ContentArchiveLibraryInterface.CreateDeltaFromAdf(outputDeltaStream, adfPath, patchContentMetaKeyGeneration, config, true);
                }

                using (var outputDeltaMergedStream = OpenNewFileStream(outputNspPath))
                using (var patchStream = OpenReadOnlyFileStream(deltaNotMergedNspPath))
                using (var deltaStream = OpenReadOnlyFileStream(deltaNspPath))
                using (var outXmlStream = OpenNewFileStream(outputNspXmlPath))
                {
                    ContentArchiveLibraryInterface.MergeDeltaContentIntoPatch(outputDeltaMergedStream, outXmlStream, patchStream, deltaStream, config);
                }

                File.Delete(adfPath);
                File.Delete(metaPath);
                File.Delete(deltaNspPath);
                File.Delete(deltaNotMergedNspPath);

                if (option.ProdEncryptionPatch.CheckIntegrity)
                {
                    using (var outStream = OpenReadOnlyFileStream(outputNspPath))
                    using (var inOriginalProdStream = OpenReadOnlyFileStream(option.ProdEncryptionPatch.InputOriginalProdNspFile))
                    {
                        // ビット化け検知 & 完全性検証
                        var integrityChecker = new IntegrityChecker(config.GetKeyConfiguration());
                        integrityChecker.CheckProdEncryptedNintendoSubmissionPackageArchive(outStream, inOriginalProdStream);
                    }
                }
            }

            if (option.ProdEncryptionPatch.CreateNspu)
            {
                using (var inStream = OpenReadOnlyFileStream(outputNspPath))
                using (var inXmlStream = OpenReadOnlyFileStream(outputNspXmlPath))
                using (var outStream = OpenNewFileStream(outputNspuPath, FileOptions.SequentialScan))
                using (var outXmlStream = OpenNewFileStream(outputNspuXmlPath))
                {
                    ContentArchiveLibraryInterface.RemoveTicket(outStream, outXmlStream, inStream, inXmlStream);
                }
            }
        }

        internal static void SparsifyNsp(Option option)
        {
            var config = option.Config;
            var sparse = option.SparsifyNsp;

            Directory.CreateDirectory(Path.GetDirectoryName(sparse.OutputOriginalFile));

            NintendoContentArchiveSource.BuildLog = new NintendoContentArchiveBuildLog(sparse.OriginalFilePath, sparse.OutputOriginalFile, sparse.IsSaveBuildLog, null);

            SparseStorageSource.SetOption(sparse.BlockSize, sparse.MinimumEraseSize);

            using (var outOriginalStream = OpenNewFileStream(sparse.OutputOriginalFile))
            using (var outPatchStream = OpenNewFileStream(sparse.OutputPatchFile))
            using (var originalStream = OpenReadOnlyFileStream(sparse.OriginalFilePath))
            using (var patchStream = OpenReadOnlyFileStream(sparse.PatchFilePath))
            using (var previousStream = !string.IsNullOrEmpty(sparse.PreviousFilePath) ? OpenReadOnlyFileStream(sparse.PreviousFilePath) : null)
            {
                ContentArchiveLibraryInterface.SparsifyNsp(outOriginalStream, outPatchStream, originalStream, patchStream, previousStream, config);
            }
        }

        internal static void ValidateMetaFile(Option option)
        {
            var reader = new MetaFileReader(option.ValidateMetaFile.MetaFilePath, option.ValidateMetaFile.MetaType);
            reader.ValidateMetaFile(option.ValidateMetaFile.MetaType);
            Console.WriteLine("Success to validate the meta file.");
        }

        internal static void ConvertIcon(Option option)
        {
            var inputFilePath = option.ConvertIcon.InputPath;
            var outputDirectory = option.ConvertIcon.outputDirectory;

            var fileName = Path.GetFileNameWithoutExtension(inputFilePath);
            var rawIconOutPath = Path.Combine(outputDirectory, fileName + ".raw.jpg");
            var nxIconOutPath = Path.Combine(outputDirectory, fileName + ".nx.jpg");

            var iconSet = IconConverter.Convert(inputFilePath, null, option.ConvertIcon.NxIconMaxSize);
            using (var fs = File.OpenWrite(rawIconOutPath))
            {
                fs.Write(iconSet.Item1, 0, iconSet.Item1.Length);
            }
            using (var fs = File.OpenWrite(nxIconOutPath))
            {
                fs.Write(iconSet.Item2, 0, iconSet.Item2.Length);
            }
        }

        internal static void ShowVersion()
        {
            var baseName = Path.GetFileNameWithoutExtension(System.Reflection.Assembly.GetExecutingAssembly().Location);
            var asm = System.AppDomain.CurrentDomain.GetAssemblies().Where(x => x.GetName().Name == baseName);
            var version = NxAddonVersion.GetCurrentVersion() + ":" + asm.Single().GetName().Version.ToString().Replace(".", "_");
            Console.WriteLine(version);
        }

        private static Action<string, T> ShowNullableProperty<T>()
        {
            return (format, value) =>
            {
                if (value != null)
                {
                    Console.WriteLine(format, value);
                }
            };
        }

        private static void OutputXml<T>(T model)
        {
            var nameSpace = new XmlSerializerNamespaces();
            nameSpace.Add(String.Empty, String.Empty);
            var sw = Console.Out;
            var serializer = new XmlSerializer(typeof(T));
            serializer.Serialize(sw, model, nameSpace);
        }

        internal static void GetAllXmlModels(Option option)
        {
            ContentArchiveLibraryInterface.GetAllXmlModels();
        }

        internal static void GetNspProperty(Option option)
        {
            var config = option.Config;

            using (var inStream = OpenReadOnlyFileStream(option.GetNspProperty.InputNspFile))
            {
                if (option.GetNspProperty.ShowApplyDifference)
                {
                    var property = NintendoContentMetaPropertyInternal.Get(new NintendoSubmissionPackageReader(inStream));
                    if (option.GetNspProperty.Format == GetNspPropertyOption.XmlFormat)
                    {
                        OutputXml(property);
                    }
                    else
                    {
                        var showBool = ShowNullableProperty<bool?>();
                        Console.WriteLine("ContentMetaProperties:");
                        showBool("  ApplyDifference: {0}", property.ApplyDifference);
                        showBool("  ApplyDifferenceBefore3NUP: {0}", property.ApplyDifferenceBefore3NUP);
                    }
                }
                else
                {
                    NintendoContentMetaPropertyList propertyList;
                    ContentArchiveLibraryInterface.GetContentMetaProperty(out propertyList, inStream, config);
                    if (option.GetNspProperty.Format == GetNspPropertyOption.XmlFormat)
                    {
                        OutputXml(propertyList);
                    }
                    else // GetNspPropertyOption.YmlFormat
                    {
                        var showLong = ShowNullableProperty<long?>();
                        Console.WriteLine("ContentMetaProperties:");
                        foreach (var property in propertyList.Properties)
                        {
                            Console.WriteLine("  Property:");
                            Console.WriteLine("    Id: {0}", property.Id);
                            Console.WriteLine("    Type: {0}", property.Type);
                            Console.WriteLine("    SizeInfo:");
                            Console.WriteLine("      DownLoad:                       {0, 12}", property.Size.DownLoad);
                            showLong("      MinimumSaveData:                {0, 12}", property.Size.MinimumSaveData);
                            showLong("      EachUserAccountSaveData:        {0, 12}", property.Size.EachUserAccountSaveData);
                            showLong("      ApplicationAreaUsedOnCard:      {0, 12}", property.Size.ApplicationAreaUsedOnCard);
                            showLong("      ApplicationAreaAvailableOnCard: {0, 12}", property.Size.ApplicationAreaAvailableOnCard);
                        }
                    }
                }
            }
        }

        internal static void GetUnpublishableError(Option option)
        {
            var config = option.Config;

            using (var inStream = OpenReadOnlyFileStream(option.GetUnpublishableError.InputNspFile))
            {
                bool isShowXml = (option.GetUnpublishableError.Format == GetUnpublishableErrorOption.XmlFormat);
                int errorCount = CheckUnpublishableError(inStream, isShowXml, true, null, config);
            }
        }

        internal static void EncryptXci(Option option)
        {
            var outputXciePath = option.EncryptXci.OutputDirectory + "/" + Path.GetFileNameWithoutExtension(option.EncryptXci.InputXciFile) + ".xcie";

            if (option.EncryptXci.UseKeyForRepairTool)
            {
                option.Config.GetKeyConfiguration().KeyIndexForEncryptedXci = EncryptedXciKeyIndex.Xcir;
                outputXciePath = outputXciePath.Replace("xcie", "xcir");
            }

            using (var inStream = OpenReadOnlyFileStream(option.EncryptXci.InputXciFile))
            using (var outStream = OpenNewFileStream(outputXciePath, FileOptions.SequentialScan))
            using (var outXmlStream = new MemoryStream())
            {
                ContentArchiveLibraryInterface.EncryptXci(outStream, outXmlStream, inStream, null, option.Config);
            }
        }

        internal static void ConvertAddOnContentForCard(Option option)
        {
            var streams = option.ConvertAddOnContentForCard.InputNspFiles.Select(x => (Stream)OpenReadOnlyFileStream(x)).ToList();
            try
            {
                using (var baseStream = OpenReadOnlyFileStream(option.ConvertAddOnContentForCard.InputBaseNspFile))
                using (var outStream = OpenNewFileStream(option.ConvertAddOnContentForCard.OutputFile))
                {
                    ContentArchiveLibraryInterface.ConvertAddOnContentForCard(outStream, streams, baseStream, option.ConvertAddOnContentForCard.CardSize, option.ConvertAddOnContentForCard.CardClockRate, option.ConvertAddOnContentForCard.CardLaunchFlags, option.Config);
                }
            }
            finally
            {
                foreach (var stream in streams)
                {
                    if (stream != null)
                    {
                        ((IDisposable)stream).Dispose();
                    }
                }
            }
        }

        internal static void CreateMultiProgramApplication(Option option)
        {
            GlobalSettings.IgnoreErrorUnpublishable = option.CreateMultiProgramApplication.IgnoreErrorUnpublishable;

            bool hasUnpublishableError = false;
            var streams = option.CreateMultiProgramApplication.InputNspFiles.Select(x => (Stream)OpenReadOnlyFileStream(x)).ToList();
            try
            {
                using (var outStream = OpenNewFileStream(option.CreateMultiProgramApplication.OutputFile))
                {
                    ContentArchiveLibraryInterface.CreateMultiProgramApplication(outStream, streams, option.CreateMultiProgramApplication.CardSize, option.CreateMultiProgramApplication.CardClockRate, option.CreateMultiProgramApplication.CardLaunchFlags, option.Config);

                    int errorCount = CheckUnpublishableError(outStream, GlobalSettings.ShowUnpublishableErroFormatXml, GlobalSettings.ErrorUnpublishable, GlobalSettings.IgnoreErrorUnpublishable, option.Config);
                    if (errorCount > 0)
                    {
                        // Unpublishableエラーが発生している場合はファイルを削除する
                        // OpenNewFileStreamでファイルを使用しているのでスコープを抜けた後に削除する。
                        hasUnpublishableError = true;
                    }
                }
            }
            finally
            {
                foreach (var stream in streams)
                {
                    if (stream != null)
                    {
                        ((IDisposable)stream).Dispose();
                    }
                }
            }

            if (GlobalSettings.IgnoreErrorUnpublishable == null && hasUnpublishableError)
            {
                File.Delete(option.CreateMultiProgramApplication.OutputFile);
                throw new ArgumentException("Found one or more unpublishable error in nsp. The nsp will be deleted automatically.");
            }
        }

        internal static void CreateMultiApplicationCardIndicator(Option option)
        {
            GlobalSettings.ErrorUnpublishable = option.CreateMultiApplicationCardIndicator.ErrorUnpublishable;
            GlobalSettings.ShowUnpublishableErroFormatXml = option.CreateMultiApplicationCardIndicator.ShowUnpublishableErroFormatXml;
            GlobalSettings.IgnoreErrorUnpublishable = option.CreateMultiApplicationCardIndicator.IgnoreErrorUnpublishable;
            bool hasUnpublishableError = false;

            var streams = option.CreateMultiApplicationCardIndicator.InputNspFiles.Select(x => (Stream)OpenReadOnlyFileStream(x)).ToList();
            try
            {
                using (var outStream = OpenNewFileStream(option.CreateMultiApplicationCardIndicator.OutputFile))
                {
                    ContentArchiveLibraryInterface.CreateMultiApplicationCardIndicator(outStream, streams, option.CreateMultiApplicationCardIndicator.Id, option.CreateMultiApplicationCardIndicator.CardSize, option.CreateMultiApplicationCardIndicator.CardClockRate, option.CreateMultiApplicationCardIndicator.CardLaunchFlags, option.Config);

                    int errorCount = CheckUnpublishableError(outStream, GlobalSettings.ShowUnpublishableErroFormatXml, GlobalSettings.ErrorUnpublishable, GlobalSettings.IgnoreErrorUnpublishable, option.Config);
                    if (errorCount > 0)
                    {
                        // Unpublishableエラーが発生している場合はファイルを削除する
                        // OpenNewFileStreamでファイルを使用しているのでスコープを抜けた後に削除する。
                        hasUnpublishableError = true;
                    }
                }
            }
            finally
            {
                foreach (var stream in streams)
                {
                    if (stream != null)
                    {
                        ((IDisposable)stream).Dispose();
                    }
                }
            }

            if (GlobalSettings.IgnoreErrorUnpublishable == null && hasUnpublishableError)
            {
                File.Delete(option.CreateMultiApplicationCardIndicator.OutputFile);
                throw new ArgumentException("Found one or more unpublishable error in nsp. The nsp will be deleted automatically.");
            }
        }

        internal static void CopyDirectory(string srcDir, string dstDir)
        {
            if (!Directory.Exists(dstDir))
            {
                Directory.CreateDirectory(dstDir);
            }

            foreach (var fileName in Directory.GetFiles(srcDir))
            {
                var dstPath = Path.Combine(dstDir, Path.GetFileName(fileName));
                File.Copy(fileName, dstPath, true);
                FileInfo fi = new FileInfo(dstPath);
                fi.Attributes = FileAttributes.Normal;
            }

            foreach (var dirName in Directory.GetDirectories(srcDir))
            {
                var dstPath = Path.Combine(dstDir, Path.GetFileName(dirName));
                CopyDirectory(dirName, dstPath);
                DirectoryInfo di = new DirectoryInfo(dstPath);
                di.Attributes = FileAttributes.Directory;
            }
        }

        internal static void MakeJunctionPoint(string path, string junctionSrcPath, bool forceCopy = false)
        {
            if (junctionSrcPath == null || junctionSrcPath == string.Empty)
            {
                return;
            }

            if (Path.GetFullPath(path) == Path.GetFullPath(junctionSrcPath))
            {
                return;
            }

            if (!Directory.Exists(junctionSrcPath))
            {
                throw new DirectoryNotFoundException(string.Format("Directory not found: {0}", junctionSrcPath));
            }

            DeleteDirectoryIfExisted(path);

            if (forceCopy)
            {
                CopyDirectory(junctionSrcPath, path);
            }
            else
            {
                using (Process process = new Process())
                {
                    process.StartInfo.FileName = Environment.GetEnvironmentVariable("ComSpec");
                    process.StartInfo.Arguments = string.Format("/c mklink /j \"{0}\" \"{1}\"", path.Replace('/', '\\'), junctionSrcPath.Replace('/', '\\'));
                    process.StartInfo.CreateNoWindow = true;
                    process.StartInfo.UseShellExecute = false;
                    process.StartInfo.RedirectStandardError = true;
                    process.StartInfo.RedirectStandardOutput = true; // 標準出力を読み捨てる
                    process.Start();

                    string errorMsg = process.StandardError.ReadToEnd();
                    process.WaitForExit();
                    if (process.ExitCode != 0)
                    {
                        Console.Error.WriteLine("Warning: Failed to create junction point: " + errorMsg);
                        Console.Error.WriteLine("Warning: Now copy files from {0} to {1}.", junctionSrcPath, path);

                        CopyDirectory(junctionSrcPath, path);
                    }
                }
            }
        }

        // fileRelPath はリンク元ファイルの srcDir からの相対パス
        internal static void MakeHardLink(string dstDir, string srcDir, string fileRelPath, bool forceCopy = false)
        {
            if (srcDir == null || srcDir == string.Empty)
            {
                return;
            }

            if (Path.GetFullPath(dstDir) == Path.GetFullPath(srcDir))
            {
                return;
            }

            if (!Directory.Exists(srcDir))
            {
                throw new DirectoryNotFoundException(string.Format("Directory not found: {0}", srcDir));
            }

            // リンク元ファイルがディレクトリ下にある場合はそのディレクトリ構成を構築する
            string dirPath = Path.GetDirectoryName(fileRelPath);
            if (dirPath != string.Empty)
            {
                EnsureDirectory(dstDir + "/" + dirPath);
            }

            string srcPath = (srcDir + "/" + fileRelPath).Replace('/', '\\');
            string dstPath = (dstDir + "/" + fileRelPath).Replace('/', '\\');
            if (forceCopy)
            {
                File.Copy(srcPath, dstPath);
            }
            else
            {
                bool isSuccess = FsUtil.CreateHardLink(dstPath, srcPath, IntPtr.Zero);
                if (!isSuccess)
                {
                    int errCode = Marshal.GetLastWin32Error();
                    string errorMsg = FsUtil.GetLastWin32ErrorMessage(errCode);

                    Console.Error.WriteLine("If you want to copy instead of creating hard link, add '--forcibly-copy-with-filtering' option.");
                    Console.Error.WriteLine("LinkPath: {0} -> {1}", srcPath, dstPath);
                    throw new NotSupportedException(string.Format("Failed to create hard link: {0}", errorMsg));
                }
            }
        }

        internal static void DeleteDirectoryIfExisted(string path)
        {
            const int RetryCountMax = 10;
            const int RetryWaitMilliSec = 5000;
            int retryCount = 0;
            while (true)
            {
                try
                {
                    if (Directory.Exists(path))
                    {
                        Directory.Delete(path, true);
                    }
                    break;
                }
                catch
                {
                    Console.Error.WriteLine("Retry #{0} Directory.Delete after {1} sec.", retryCount, RetryWaitMilliSec);
                    System.Threading.Thread.Sleep(RetryWaitMilliSec);
                    retryCount++;
                    if (retryCount == RetryCountMax)
                    {
                        throw;
                    }
                }
            }

            // Directory.Delete は非同期なので、削除されるまで待つ
            const int WaitForDeleteDirectoryTimeOutMilliSec = 1000 * 3;
            const int WaitForDeleteDirectoryWaitUnitMilliSec = 100;
            for (int waitMilliSec = 0; waitMilliSec < WaitForDeleteDirectoryTimeOutMilliSec; waitMilliSec += WaitForDeleteDirectoryWaitUnitMilliSec)
            {
                if (Directory.Exists(path) == false)
                {
                    break;
                }
                System.Threading.Thread.Sleep(WaitForDeleteDirectoryWaitUnitMilliSec);
            }
        }

        internal static void DeleteIncludingJunctionDirectory(string path)
        {
            if (!Directory.Exists(path))
            {
                return;
            }

            foreach (var dirPath in Directory.EnumerateDirectories(path))
            {
                var attributes = File.GetAttributes(dirPath);

                // ジャンクションポイントを含んでいると、Directory.Delete で再帰的に消してくれないので、個別に消す
                if ((attributes & FileAttributes.ReparsePoint) == FileAttributes.ReparsePoint)
                {
                    DeleteDirectoryIfExisted(dirPath);
                }
                else if ((attributes & FileAttributes.Directory) == FileAttributes.Directory)
                {
                    DeleteIncludingJunctionDirectory(dirPath);
                }
            }

            DeleteDirectoryIfExisted(path);
        }

        internal static void EnsureDirectory(string path)
        {
            if (!Directory.Exists(path))
            {
                Directory.CreateDirectory(path);
            }
        }
        private static int CheckUnpublishableError(Stream inStream, bool isShowXml, bool isSpecifiedOption, string ignoreListFilePath, AuthoringConfiguration config)
        {
            // UnpublishableErrorエラーの解析
            UnpublishableErrorModel unpublishableErrorModel;
            ContentArchiveLibraryInterface.GetUnpublishableError(out unpublishableErrorModel, inStream, isSpecifiedOption, ignoreListFilePath, config);

            // UnpublishableErrorエラーの発生数を取得する
            int errorCount = 0;
            int warningCount = 0;
            UnpublishableError.GetUnpublishableErrorCount(out errorCount, out warningCount, unpublishableErrorModel);

            if (isSpecifiedOption == true || errorCount + warningCount > 0)
            {
                // --error-unpublishable が指定されている場合かエラー・警告が発生している場合はUnpublishableErrorエラー情報を表示する
                UnpublishableError.ShowUnpublishableError(unpublishableErrorModel, isShowXml);
            }

            return errorCount;
        }

        private class ConsoleEncodeChange : IDisposable
        {
            private bool disposed;
            private bool isChangeEncoding;
            private Encoding originalEncoding;
            public ConsoleEncodeChange()
            {
                originalEncoding = Console.OutputEncoding;
                isChangeEncoding = false;
            }
            ~ConsoleEncodeChange()
            {
                Dispose(false);
            }
            public void Dispose()
            {
                Dispose(true);
                GC.SuppressFinalize(this);
            }
            protected virtual void Dispose(bool disposing)
            {
                if (this.disposed)
                {
                    return;
                }
                if (isChangeEncoding)
                {
                    Console.OutputEncoding = originalEncoding;
                    isChangeEncoding = false;
                }
                this.disposed = true;
            }
            public void ChangeEncodingToUtf8()
            {
                isChangeEncoding = true;
                originalEncoding = Console.OutputEncoding;
                Console.OutputEncoding = Encoding.UTF8;
            }
        }

        internal static void Main(string[] args)
        {
            Option option = new Option();

            using (var consoleEncodeChange = new ConsoleEncodeChange())
            {
#if !DEBUG
                try
#endif
                {
                    if (option.HasUtf8Option(args))
                    {
                        consoleEncodeChange.ChangeEncodingToUtf8();
                    }

                    option.Parse(args);

                    if (option.IsShowUsage)
                    {
                        option.ShowUsage();
                        Environment.Exit(1);
                    }

                    if (option.IsVerbose)
                    {
                        Log.EnableInfo();
                        Log.EnableProgress();
                    }

                    if (option.CreateFs != null)
                    {
                        CreateFs(option);
                    }
                    else if (option.CreateNca != null)
                    {
                        CreateNca(option);
                    }
                    else if (option.CreateNsp != null)
                    {
                        CreateNsp(option);
                    }
                    else if (option.CreateNspMeta != null)
                    {
                        CreateNspMeta(option);
                    }
                    else if (option.CreateNspd != null)
                    {
                        CreateNspd(option);
                    }
                    else if (option.CreateNacp != null)
                    {
                        CreateNacp(option);
                    }
                    else if (option.MakePatch != null)
                    {
                        MakePatch(option);
                    }
                    else if (option.OptimizePatch != null)
                    {
                        OptimizePatch(option);
                    }
                    else if (option.MakeDelta != null)
                    {
                        MakeDelta(option);
                    }
                    else if (option.MergeDeltaContent != null)
                    {
                        MergeDeltaContent(option);
                    }
                    else if (option.Extract != null)
                    {
                        Extract(option);
                    }
                    else if (option.ExtractNsp != null)
                    {
                        ExtractNsp(option);
                    }
                    else if (option.ExtractNsave != null)
                    {
                        ExtractNsave(option);
                    }
                    else if (option.MergeNsp != null)
                    {
                        MergeNsp(option);
                    }
                    else if (option.Replace != null)
                    {
                        Replace(option);
                    }
                    else if (option.ReplaceNspMeta != null)
                    {
                        ReplaceNspMeta(option);
                    }
                    else if (option.List != null)
                    {
                        List(option);
                    }
                    else if (option.DiffPatch != null)
                    {
                        DiffPatch(option);
                    }
                    else if (option.Verify != null)
                    {
                        Verify(option);
                    }
                    else if (option.ProdEncryption != null)
                    {
                        ProdEncryption(option);
                    }
                    else if (option.ProdEncryptionPatch != null)
                    {
                        ProdEncryptionPatch(option);
                    }
                    else if (option.ValidateMetaFile != null)
                    {
                        ValidateMetaFile(option);
                    }
                    else if (option.ConvertIcon != null)
                    {
                        ConvertIcon(option);
                    }
                    else if (option.ShowVersion != null)
                    {
                        ShowVersion();
                    }
                    else if (option.GetAllXmlModels != null)
                    {
                        GetAllXmlModels(option);
                    }
                    else if (option.GetNspProperty != null)
                    {
                        GetNspProperty(option);
                    }
                    else if (option.GetUnpublishableError != null)
                    {
                        GetUnpublishableError(option);
                    }
                    else if (option.EncryptXci != null)
                    {
                        EncryptXci(option);
                    }
                    else if (option.SplitNsp != null)
                    {
                        SplitNsp(option);
                    }
                    else if (option.ConvertAddOnContentForCard != null)
                    {
                        ConvertAddOnContentForCard(option);
                    }
                    else if (option.CreateMultiProgramApplication != null)
                    {
                        CreateMultiProgramApplication(option);
                    }
                    else if (option.CreateMultiApplicationCardIndicator != null)
                    {
                        CreateMultiApplicationCardIndicator(option);
                    }
                    else if (option.SparsifyNsp != null)
                    {
                        SparsifyNsp(option);
                    }
                }
#if !DEBUG
                catch (InvalidOptionException ex)
                {
                    Console.Error.WriteLine("[Error] {0}", ex.Message);
                    option.ShowUsage();
                    Environment.Exit(1);
                }
                catch (Exception ex)
                {
                    Console.Error.WriteLine("[Error] {0}", ex.Message);
                    var innerException = ex.InnerException;
                    while (innerException != null)
                    {
                        Console.Error.WriteLine("[Error] {0}", innerException.Message);
                        innerException = innerException.InnerException;
                    }
                    Console.Error.WriteLine("\nStackTrace:\n{0}", ex.StackTrace);
                    Environment.Exit(1);
                }
#endif
            }
        }
    }
}
