﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "tool_SystemInitializer.h"
#include "Utility/systemInitializer_Crc32.h"
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_BitUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Result.h>
#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/utilTool/utilTool_ResultHandlingUtility.h>
#include "Utility/systemInitializer_FsBisPartition.h"
#include "Utility/systemInitializer_StorageUtility.h"
#include <algorithm>
#include <Partition/systemInitializer_GptHolder.h>
#include <vector>
#include <nn/crypto/crypto_Sha1Generator.h>
#include <cstdio>
#include <memory>
#include <nn/fs.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/updater/updater.h>
#include <functional>
#include "Utility/systemInitializer_SaveData.h"
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

namespace
{
    std::string ConvertHex(void* buffer, size_t size)
    {
        std::string ret;
        uint8_t *u8buf = reinterpret_cast<uint8_t*>(buffer);
        char buf[3];
        for(size_t i = 0; i < size; i++)
        {
            std::sprintf(buf, "%02x", u8buf[i]);
            ret += buf;
        }

        return ret;
    }

    const size_t StorageWritingBufferSize = 16 * 1024 * 1024;

    NN_ALIGNAS(4096) static uint8_t s_StorageWritingBuffer[StorageWritingBufferSize];

    nn::Result WriteSectorsFromFile(IBlockStorage *pBlockStorage, uint64_t startSectorIndex, uint64_t writeBytes, IFile *pFile, uint64_t startAddress)
    {
        uint64_t restBytes = writeBytes;
        uint64_t currentOffset = startAddress;
        uint64_t currentSectorIndex = startSectorIndex;
        size_t blockSize = static_cast<size_t>(pBlockStorage->GetBlockSize());

        nn::crypto::Sha1Generator sha1generator;

        sha1generator.Initialize();

        while(0 < restBytes)
        {
            uint32_t writeChunkSize = static_cast<size_t>(std::min(restBytes, static_cast<uint64_t>(StorageWritingBufferSize)));

            //
            // http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-14695
            // ここで見つけたバグを回避するためのコード。
            // 理由は分からないが、4192256 バイトを読もうとすると返ってこなくなる。
            // 下層で返ってこなくなるのは tma::file_io::ReadFile。
            // このコードはバグが修正された後に放置しても悪影響は無いはず。
            //
            if(writeChunkSize == 4192256)
            {
                writeChunkSize -= 1024;
            }

            uint32_t writeNumChunks = CalculateNumberOfSectors(writeChunkSize, blockSize);
            size_t readSize;

            NN_UTILTOOL_RESULT_DO(pFile->Read(&readSize, currentOffset, s_StorageWritingBuffer, writeChunkSize));
            sha1generator.Update(s_StorageWritingBuffer, readSize);

            NN_UTILTOOL_RESULT_DO(
                pBlockStorage->Write(
                    currentSectorIndex,
                    writeNumChunks,
                    s_StorageWritingBuffer,
                    nn::util::align_up(writeChunkSize, blockSize)));

            currentSectorIndex += writeNumChunks;
            restBytes -= writeChunkSize;
            currentOffset += writeChunkSize;
        }

        uint8_t buffer[sha1generator.HashSize];
        sha1generator.GetHash(&buffer, sha1generator.HashSize);
        std::string hashValue = ConvertHex(buffer, sha1generator.HashSize);
        NN_UTILTOOL_LOG_DEBUG("Write data sha1hash: %s", hashValue.c_str());

        NN_UTILTOOL_LOG_DEBUG("Write sectors %08llx-%08llx from %08llx-%08llx\n", startSectorIndex, currentSectorIndex, startAddress, currentOffset);

        NN_RESULT_SUCCESS;
    }

    nn::Result WriteSectorsByZero(IBlockStorage *pBlockStorage, uint64_t offsetBytes, uint64_t writeBytes)
    {
        uint64_t restBytes = writeBytes;
        uint64_t currentOffset = offsetBytes;
        uint64_t currentSectorIndex = CalculateNumberOfSectors(offsetBytes, pBlockStorage->GetBlockSize());;
        size_t blockSize = static_cast<size_t>(pBlockStorage->GetBlockSize());

        std::memset(s_StorageWritingBuffer, 0, sizeof(s_StorageWritingBuffer));

        NN_UTILTOOL_LOG_DEBUG("Write zero: offset: %016llx, bytes: %llu, storage size: %llu", offsetBytes, writeBytes, pBlockStorage->GetTotalSize());

        while(0 < restBytes)
        {
            uint32_t writeChunkSize = static_cast<size_t>(std::min(restBytes, static_cast<uint64_t>(StorageWritingBufferSize)));
            uint32_t writeNumChunks = static_cast<uint32_t>(CalculateNumberOfSectors(writeChunkSize, blockSize));

            NN_UTILTOOL_LOG_DEBUG("Write zero: sectorIndex:%lld, numSectors: %lld", currentSectorIndex, writeNumChunks);
            NN_UTILTOOL_RESULT_DO(
                pBlockStorage->Write(
                    currentSectorIndex,
                    writeNumChunks,
                    s_StorageWritingBuffer,
                    writeNumChunks * blockSize));

            currentSectorIndex += writeNumChunks;
            restBytes -= writeChunkSize;
            currentOffset += writeChunkSize;
        }

        NN_RESULT_SUCCESS;
    }

    struct SparseImageHeader
    {
        char        signature[8];         // "SPRSIMG\0"
        int64_t     numChunks;

        static SparseImageHeader MakeInitialHeader()
        {
            SparseImageHeader initialHeader = { "SPRSIMG", 0 };
            return initialHeader;
        }
    };

    struct SparseImageChunk
    {
        int64_t    address;              // 書き込み先の場所（バイト単位）
        int64_t    size;                 // data のサイズのみ表す。構造体のサイズは含めない

        static SparseImageChunk Make(int64_t address, int64_t size)
        {
            SparseImageChunk ret = { address, size };
            return ret;
        }
    };

    nn::Result WriteSparseImageFromFile(IBlockStorage *pBlockStorage, uint64_t startSectorIndex, uint64_t maxWriteSize, IFile *pFile, uint64_t startAddress)
    {
        uint64_t currentSourceAddress = startAddress;
        size_t readSize;
        size_t blockSize = static_cast<size_t>(pBlockStorage->GetBlockSize());
        uint64_t endSectorIndex = startSectorIndex + (maxWriteSize / blockSize);

        // ヘッダを読む
        SparseImageHeader header;
        NN_UTILTOOL_RESULT_DO(pFile->Read(&readSize, currentSourceAddress, &header, sizeof(SparseImageHeader)));
        currentSourceAddress += readSize;

        NN_UTILTOOL_LOG_DEBUG("Num chunks: %lld\n", header.numChunks);

        for (int i = 0; i < header.numChunks; i++)
        {
            SparseImageChunk chunk;
            NN_UTILTOOL_RESULT_DO(pFile->Read(&readSize, currentSourceAddress, &chunk, sizeof(chunk)));
            currentSourceAddress += readSize;

            uint64_t currentSectorIndex = startSectorIndex + (chunk.address / blockSize);

            NN_ABORT_UNLESS(currentSectorIndex + (chunk.size / blockSize) <= endSectorIndex);

            NN_UTILTOOL_LOG_DEBUG("Write chunk: index = %4d, address = 0x%016llx, size = %12lld\n", i, chunk.address, chunk.size);
            NN_UTILTOOL_RESULT_DO(
                WriteSectorsFromFile(pBlockStorage, currentSectorIndex, chunk.size, pFile, currentSourceAddress));
            currentSourceAddress += chunk.size;
        }

        NN_RESULT_SUCCESS;
    }

    bool IsEksEnabled()
    {
        bool isEksEnabled;
        auto size = nn::settings::fwdbg::GetSettingsItemValue(&isEksEnabled, sizeof(isEksEnabled), "systeminitializer", "eks_enabled");
        NN_ABORT_UNLESS_EQUAL(size, sizeof(isEksEnabled));

        return isEksEnabled;
    }

    int64_t GetBctEksOffset()
    {
        int offset;
        auto size = nn::settings::fwdbg::GetSettingsItemValue(&offset, sizeof(offset), "systeminitializer", "bct_eks_offset");
        NN_ABORT_UNLESS(size == sizeof(offset));

        return offset;
    }

    int64_t GetBctVersionOffset()
    {
        int offset;
        auto size = nn::settings::fwdbg::GetSettingsItemValue(&offset, sizeof(offset), "systeminitializer", "bct_version_offset");
        NN_ABORT_UNLESS(size == sizeof(offset));

        return offset;
    }
}

namespace
{
    const size_t BctSize = 16 * 1024;
    const size_t EksSize = 0x200;
    const int64_t BctAddressBase = 0x00000000;
    const int64_t BctAddressOffset = 0x00004000;
    const int64_t EksAddressBase = 0x00180000;
    const int64_t EksAddressOffset = 0x00000200;
}

nn::Result SystemInitializer::Initialize(IBlockStorage *pBlockStorage, IFile *pImageFile, SystemIntializerOption option)
{
    m_pImageFile = pImageFile;
    m_pBlockStorage = pBlockStorage;
    m_Option = option;

    NN_UTILTOOL_RESULT_DO(m_InitialImage.Initialize(m_pImageFile));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::InitializeSystem()
{
    InitialImageCommandPartition commandPartition;

    NN_UTILTOOL_RESULT_DO(m_InitialImage.ReadCommandPartition(&commandPartition));

    NN_UTILTOOL_RESULT_DO(ProcessCommands(commandPartition));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::ProcessCommands(const InitialImageCommandPartition &commandPartition)
{
    NN_UTILTOOL_LOG_VERBOSE("Number of commands: %d", commandPartition.numCommand);

    for(int i = 0; i < commandPartition.numCommand; i++)
    {
        const InitialImageCommand &command = commandPartition.commands[i];
        NN_UTILTOOL_LOG_DEBUG(
            "Command[%d]: id = %d, args=%lld,%lld,%lld,%lld,%lld,%lld,%lld",
            i, command.command,
            command.args[0],command.args[1],command.args[2],command.args[3],command.args[4],command.args[5],command.args[6]);

        NN_UTILTOOL_RESULT_DO(ProcessCommand(commandPartition.commands[i]));
    }

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::ProcessCommand(const InitialImageCommand &command)
{
    switch(command.command)
    {
    case InitialImageCommandType_WritePartitionTable:
        {
            NN_UTILTOOL_RESULT_DO(
                WritePartitionTable(command.args[0], command.args[1], command.args[2], command.args[3]));
        }
        break;
    case InitialImageCommandType_WriteRawImage:
        {
            NN_ABORT("Not implemented: command = %d", command.command);
        }
        break;
    case InitialImageCommandType_UpdatePartitionTable:
        {
            NN_ABORT("Not implemented: command = %d", command.command);
        }
        break;
    case InitialImageCommandType_WritePartitionImage:
        {
            NN_UTILTOOL_RESULT_DO(
                WritePartitionImage(command.args[0], command.args[1], command.args[2]));
        }
        break;
    case InitialImageCommandType_WriteSparsePartitionImage:
        {
            NN_UTILTOOL_RESULT_DO(
                WriteSparsePartitionImage(command.args[0], command.args[1], command.args[2]));
        }
        break;
    case InitialImageCommandType_WriteFsPartitionImage:
        {
            NN_UTILTOOL_RESULT_DO(
                WriteFsPartitionImage(command.stringArgument, command.args[0], command.args[1]));
        }
        break;
    case InitialImageCommandType_WriteFsSparsePartitionImage:
        {
            NN_UTILTOOL_RESULT_DO(
                WriteFsSparsePartitionImage(command.stringArgument, command.args[0], command.args[1]));
        }
        break;
    case InitialImageCommandType_EnsureProtectProductionInfo:
        {
            NN_UTILTOOL_RESULT_DO(
                EnsureProtectProductionInfo(command.args[0], command.args[1], command.args[2], command.args[3]));
        }
        break;
    case InitialImageCommandType_UpdateSecureInfo:
        {
            if (IsEksEnabled())
            {
                NN_UTILTOOL_RESULT_DO(
                    UpdateSecureInfo(command.args[0]));
            }
            else
            {
                NN_ABORT("Not implemented: command = %d", command.command);
            }
        }
        break;
    case InitialImageCommandType_WriteBatchedBootPartitions:
        {
            NN_UTILTOOL_RESULT_DO(
                WriteBatchedBootPartitions(command.stringArgument, command.args[0], command.args[1], command.stringArgument2, command.args[2], command.args[3], command.args[4]));
        }
        break;
    case InitialImageCommandType_EraseEmmcWithEscapingData:
        {
            NN_UTILTOOL_RESULT_DO(
                EraseEmmcWithEscapingData(command.args[0], command.args[1]));
        }
        break;
    case InitialImageCommandType_InitializeZero:
        {
            NN_UTILTOOL_RESULT_DO(
                InitializeZero(command.stringArgument));
        }
        break;
    case InitialImageCommandType_EnableUpdatingFromBootImagePackage:
        {
            NN_UTILTOOL_RESULT_DO(
                EnableUpdatingFromBootImagePackage(command.args[0]));
        }
        break;
    case InitialImageCommandType_WriteDatabase:
        {
            NN_RESULT_DO(
                WriteDatabase(command.stringArgument, command.args[0]));
        }
        break;
    default:
        NN_ABORT("unknown command: %lld", command.command);
    }

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::WritePartitionTable(int64_t mbrIndex, int64_t gptIndex, int64_t gptPartitionIndex, int64_t partitionAddressIndex)
{
    NN_UTILTOOL_LOG_VERBOSE(
        "WritePartitionTable: mbr=%lld(%s), gptHeader=%lld(%s), partitions=%lld(%s), partitionAddresses=%lld(%s)\n",
        mbrIndex, m_InitialImage.GetPartitionInfo(mbrIndex).name,
        gptIndex, m_InitialImage.GetPartitionInfo(gptIndex).name,
        gptPartitionIndex, m_InitialImage.GetPartitionInfo(gptPartitionIndex).name,
        partitionAddressIndex, m_InitialImage.GetPartitionInfo(partitionAddressIndex).name);

    NN_ALIGNAS(4096) static Mbr mbr;
    NN_ALIGNAS(4096) static GptHeader alternateGptHeader;

    GptHolder &newGpt = m_NewGpt;
    GptHolder oldGpt;

    // 書き込み予定のパーティションを読み込む
    size_t readSize;
    NN_UTILTOOL_RESULT_DO(
        m_InitialImage.ReadPartition(&readSize, mbrIndex, 0, &mbr, sizeof(Mbr)));
    NN_UTILTOOL_RESULT_DO(ReadGptFromImage(&newGpt, gptIndex, gptPartitionIndex));

    // 既存のパーティションを読み込む
    NN_UTILTOOL_LOG_DEBUG("Read preavious gpt.");
    NN_UTILTOOL_RESULT_DO(oldGpt.Read(m_pBlockStorage));

    NN_UTILTOOL_LOG_VERBOSE("==============================");
    NN_UTILTOOL_LOG_VERBOSE("Old Partition");
    NN_UTILTOOL_LOG_VERBOSE("==============================");
    PrintGptInfo(*oldGpt.GetHeader(), oldGpt.GetPartitions());

    // パーティションのアドレスを読み込む
    NN_UTILTOOL_LOG_DEBUG("Read partition addresses.");
    std::vector<PartitionAddress> partitionAddresses;
    NN_UTILTOOL_RESULT_DO(
        ReadPartitionAddresses(&partitionAddresses, partitionAddressIndex));

    // パーティションのアドレスを再計算する
    CalculatePartitionAddress(newGpt.GetHeader(), newGpt.GetPartitions(), oldGpt.GetHeader(), oldGpt.GetPartitions(), partitionAddresses.size(), &(partitionAddresses.at(0)), m_pBlockStorage->GetTotalBlocks() * m_pBlockStorage->GetBlockSize(),  m_pBlockStorage->GetBlockSize());
    UpdateCrc32InGptHeader(newGpt.GetHeader(), newGpt.GetPartitions());
    MakeAlternateGpt(&alternateGptHeader, *newGpt.GetHeader(), newGpt.GetPartitions(), m_pBlockStorage->GetBlockSize());

    NN_UTILTOOL_LOG_VERBOSE("==============================");
    NN_UTILTOOL_LOG_VERBOSE("New Partition");
    NN_UTILTOOL_LOG_VERBOSE("==============================");
    PrintGptInfo(*newGpt.GetHeader(), newGpt.GetPartitions());

    NN_ABORT_UNLESS(VerifyGptPartition(*newGpt.GetHeader(), newGpt.GetPartitions()));
    NN_ABORT_UNLESS(VerifyGptPartition(alternateGptHeader, newGpt.GetPartitions()));

    NN_UTILTOOL_RESULT_DO(
        ::WritePartitionTable(m_pBlockStorage, mbr, *newGpt.GetHeader(), alternateGptHeader, newGpt.GetPartitions(), GPT_DEFAULT_PARTITION_ENTRIES));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::WritePartitionImage(int64_t destinationGptIndex, int64_t sourceIndex, int64_t offset)
{
    NN_UTILTOOL_LOG_VERBOSE("WritePartitionImage: gptIndex=%lld from imageIndex=%lld(%s), offset=0x%08llx",
        destinationGptIndex, sourceIndex, m_InitialImage.GetPartitionInfo(sourceIndex).name, offset);

    NN_ABORT_UNLESS_RANGE(destinationGptIndex, 0, m_NewGpt.GetHeader()->numberOfPartitionEntries);

    InitialImageInfo::PartitionInfo sourcePartition = m_InitialImage.GetPartitionInfo(sourceIndex);
    uint64_t startingLba = m_NewGpt.GetPartitions()[destinationGptIndex].startingLba + CalculateNumberOfSectors(offset, m_pBlockStorage->GetBlockSize());

    NN_UTILTOOL_RESULT_DO(
        WriteSectorsFromFile(m_pBlockStorage, startingLba, sourcePartition.size, m_pImageFile, sourcePartition.address));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::WriteSparsePartitionImage(int64_t destinationGptIndex, int64_t sourceIndex, int64_t offset)
{
    NN_UTILTOOL_LOG_VERBOSE("WriteSparsePartitionImage: gptIndex=%lld from imageIndex=%lld(%s), offset=0x%08llx",
        destinationGptIndex, sourceIndex, m_InitialImage.GetPartitionInfo(sourceIndex).name, offset);

    NN_ABORT_UNLESS_RANGE(destinationGptIndex, 0, m_NewGpt.GetHeader()->numberOfPartitionEntries);

    InitialImageInfo::PartitionInfo sourcePartition = m_InitialImage.GetPartitionInfo(sourceIndex);
    uint64_t partitionSize = (m_NewGpt.GetPartitions()[destinationGptIndex].endingLba - m_NewGpt.GetPartitions()[destinationGptIndex].startingLba) * m_pBlockStorage->GetBlockSize();
    uint64_t startingLba = m_NewGpt.GetPartitions()[destinationGptIndex].startingLba + CalculateNumberOfSectors(offset, m_pBlockStorage->GetBlockSize());

    NN_UTILTOOL_RESULT_DO(
        WriteSparseImageFromFile(m_pBlockStorage, startingLba, partitionSize, m_pImageFile, sourcePartition.address));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::WriteFsPartitionImage(const char* destitinationPartitionName, int64_t sourceIndex, int64_t offset)
{
    NN_UTILTOOL_LOG_VERBOSE("WriteFsPartitionImage: destinationName=%s from imageIndex=%lld(%s), offset=0x%08llx",
        destitinationPartitionName, sourceIndex, m_InitialImage.GetPartitionInfo(sourceIndex).name, offset);

    std::unique_ptr<nn::fs::IStorage> pStorage;

    if(!HasBisPartitionName(destitinationPartitionName))
    {
        NN_ABORT("Has no partition: FsPartitionId=%s", destitinationPartitionName);
    }

    NN_UTILTOOL_RESULT_DO(
        nn::fs::OpenBisPartition(&pStorage, ConvertToBisPartitionId(destitinationPartitionName)));

    auto blockStorage = MakeBlockStorageFrom(std::move(pStorage));

    InitialImageInfo::PartitionInfo sourcePartition = m_InitialImage.GetPartitionInfo(sourceIndex);
    uint64_t startingLba = CalculateNumberOfSectors(offset, blockStorage->GetBlockSize());

    NN_UTILTOOL_RESULT_DO(
        WriteSectorsFromFile(blockStorage.get(), startingLba, sourcePartition.size, m_pImageFile, sourcePartition.address));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::WriteFsSparsePartitionImage(const char* destitinationPartitionName, int64_t sourceIndex, int64_t offset)
{
    NN_UTILTOOL_LOG_VERBOSE("WriteFsSparsePartitionImage: destinationName=%s from imageIndex=%lld(%s), offset=0x%08llx",
        destitinationPartitionName, sourceIndex, m_InitialImage.GetPartitionInfo(sourceIndex).name, offset);

    std::unique_ptr<nn::fs::IStorage> pStorage;

    if(!HasBisPartitionName(destitinationPartitionName))
    {
        NN_ABORT("Has no partition: FsPartitionId=%s", destitinationPartitionName);
    }

    NN_UTILTOOL_RESULT_DO(
        nn::fs::OpenBisPartition(&pStorage, ConvertToBisPartitionId(destitinationPartitionName)));

    auto blockStorage = MakeBlockStorageFrom(std::move(pStorage));

    InitialImageInfo::PartitionInfo sourcePartition = m_InitialImage.GetPartitionInfo(sourceIndex);
    uint64_t startingLba = CalculateNumberOfSectors(offset, blockStorage->GetBlockSize());

    NN_UTILTOOL_RESULT_DO(
        WriteSparseImageFromFile(blockStorage.get(), startingLba, blockStorage->GetTotalSize(), m_pImageFile, sourcePartition.address));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::WriteFsPartition(const char* partitionName, int64_t offset, void* buffer, size_t bufferSize)
{
    std::unique_ptr<nn::fs::IStorage> pStorage;

    if(!HasBisPartitionName(partitionName))
    {
        NN_ABORT("Has no partition: FsPartitionId=%s", partitionName);
    }

    NN_UTILTOOL_RESULT_DO(
        nn::fs::OpenBisPartition(&pStorage, ConvertToBisPartitionId(partitionName)));

    auto blockStorage = MakeBlockStorageFrom(std::move(pStorage));

    uint64_t startingLba = CalculateNumberOfSectors(offset, blockStorage->GetBlockSize());

    NN_UTILTOOL_RESULT_DO(
        blockStorage->Write(startingLba, CalculateNumberOfSectors(bufferSize, blockStorage->GetBlockSize()), buffer , bufferSize));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::WriteBatchedBootPartitions(const char* bctPartitionName, int64_t sourceBctIndex, int64_t bctOffset, const char* blPartitionName, int64_t sourceBlIndex, int64_t blOffset, int64_t secureInfoIndex)
{
    NN_UTILTOOL_LOG_VERBOSE("WriteBatchedBootPartitions: bctPartitionName=%s, sourceBctIndex=%lld(%s), bctOffset=%lld, blPartitionName=%s, sourceBlIndex=%lld(%s), blOffset=%lld, secureInfoIndex=%lld",
        bctPartitionName, sourceBctIndex, m_InitialImage.GetPartitionInfo(sourceBctIndex).name, bctOffset,
        blPartitionName, sourceBlIndex, m_InitialImage.GetPartitionInfo(sourceBlIndex).name, blOffset,
        secureInfoIndex);

    InitialImageInfo::PartitionInfo bctPartition = m_InitialImage.GetPartitionInfo(sourceBctIndex);
    InitialImageInfo::PartitionInfo blPartition =  m_InitialImage.GetPartitionInfo(sourceBlIndex);

    NN_ABORT_UNLESS(bctPartition.size <= static_cast<int64_t>(BctSize));

    std::unique_ptr<uint8_t[]> bctBuffer(new uint8_t[BctSize]);
    size_t blBufferSize = nn::util::align_up(blPartition.size, 512);
    std::unique_ptr<uint8_t[]> blBuffer(new uint8_t[blBufferSize]);
    std::memset(bctBuffer.get(), 0, BctSize);
    std::memset(blBuffer.get(), 0, blBufferSize);

    NN_ABORT_UNLESS_NOT_NULL(bctBuffer.get());
    NN_ABORT_UNLESS_NOT_NULL(blBuffer.get());

    NN_UTILTOOL_LOG_DEBUG("bctPartitionSize: %zu", bctPartition.size);
    NN_UTILTOOL_LOG_DEBUG("blPartitionSize: %zu", blPartition.size);

    // read bct from image
    size_t readSize = 0;
    NN_UTILTOOL_RESULT_DO(
        m_pImageFile->Read(&readSize, bctPartition.address, bctBuffer.get(), bctPartition.size));
    NN_ABORT_UNLESS_EQUAL(readSize, bctPartition.size);

    // read bl from image
    NN_UTILTOOL_RESULT_DO(
        m_pImageFile->Read(&readSize, blPartition.address, blBuffer.get(), blPartition.size));
    NN_ABORT_UNLESS_EQUAL(readSize, blPartition.size);

    if (IsEksEnabled())
    {
        // update secureinfo
        UpdateSecureInfo(bctBuffer.get());
    }

    // write bct
    NN_UTILTOOL_RESULT_DO(
        WriteFsPartition(bctPartitionName, bctOffset, bctBuffer.get(), BctSize));

    // write bl
    NN_UTILTOOL_RESULT_DO(
        WriteFsPartition(blPartitionName, blOffset, blBuffer.get(), blBufferSize));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::EraseEmmcWithEscapingData(int64_t targetPartition1, int64_t targetPartition2)
{
    NN_UTILTOOL_LOG_VERBOSE("EraseEmmcWithEscapingData: targetPartition(%lld, %lld)",
        targetPartition1, targetPartition2);

    NN_ABORT_UNLESS(m_Option.enableEraseEmmc, "must not use erase emmc command");

    // MBR を退避する
    NN_ALIGNAS(4096) static Mbr mbr;
    NN_UTILTOOL_RESULT_DO(
        m_pBlockStorage->Read(&mbr, sizeof(mbr), 0, 1));

    // GPT を退避する
    GptHolder gpt;
    NN_ALIGNAS(4096) static GptHeader alternateGptHeader;
    NN_UTILTOOL_RESULT_DO(gpt.Read(m_pBlockStorage));
    gpt.GetHeader()->alternateLba = m_pBlockStorage->GetTotalBlocks() - 1;
    MakeAlternateGpt(&alternateGptHeader, *gpt.GetHeader(), gpt.GetPartitions(), m_pBlockStorage->GetBlockSize());

    // パーティションのサイズを取得
    auto target1 = gpt.GetPartitions()[targetPartition1];
    auto target2 = gpt.GetPartitions()[targetPartition2];
    uint64_t targetPartition1Size = CalculatePartitionSize(target1, this->m_pBlockStorage);
    uint64_t targetPartition2Size = CalculatePartitionSize(target2, this->m_pBlockStorage);
    NN_UTILTOOL_LOG_DEBUG("  - PartitionSize1: %llu", targetPartition1Size);
    NN_UTILTOOL_LOG_DEBUG("  - PartitionSize2: %llu", targetPartition2Size);
    // パーティションのサイズ分メモリを確保
    std::unique_ptr<uint8_t[]> target1Buffer(new uint8_t[targetPartition1Size]);
    std::unique_ptr<uint8_t[]> target2Buffer(new uint8_t[targetPartition2Size]);

    NN_ABORT_UNLESS_NOT_NULL(target1Buffer.get());
    NN_ABORT_UNLESS_NOT_NULL(target2Buffer.get());

    // NANDから読み出す
    NN_UTILTOOL_LOG_DEBUG("Escape partitions");
    NN_UTILTOOL_RESULT_DO(
        this->m_pBlockStorage->Read(target1Buffer.get(), targetPartition1Size, target1.startingLba, target1.GetBlocks()));
    NN_UTILTOOL_RESULT_DO(
        this->m_pBlockStorage->Read(target2Buffer.get(), targetPartition2Size, target2.startingLba, target2.GetBlocks()));

    // eMMC を初期化する
    NN_UTILTOOL_LOG_DEBUG("Erase UserData");
    NN_UTILTOOL_RESULT_DO(
        nn::fs::EraseMmc(nn::fs::MmcPartition::UserData));

    // 退避した奴を書き戻す
    NN_UTILTOOL_LOG_DEBUG("Write escaped partitions");
    NN_UTILTOOL_RESULT_DO(
        this->m_pBlockStorage->Write(target1.startingLba, target1.GetBlocks(), target1Buffer.get(), targetPartition1Size));
    NN_UTILTOOL_RESULT_DO(
        this->m_pBlockStorage->Write(target2.startingLba, target2.GetBlocks(), target2Buffer.get(), targetPartition2Size));

    // MBR, GPT を書き戻す
    NN_UTILTOOL_RESULT_DO(
        ::WritePartitionTable(m_pBlockStorage, mbr, *gpt.GetHeader(), alternateGptHeader, gpt.GetPartitions(), GPT_DEFAULT_PARTITION_ENTRIES));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::InitializeZero(const char* targetPartitionName)
{
    NN_UTILTOOL_LOG_VERBOSE("InitializeZero: targetPartition(%s)",
        targetPartitionName);

    std::unique_ptr<nn::fs::IStorage> pStorage;

    if(!HasBisPartitionName(targetPartitionName))
    {
        NN_ABORT("Has no partition: FsPartitionId=%s", targetPartitionName);
    }

    NN_UTILTOOL_RESULT_DO(
        nn::fs::OpenBisPartition(&pStorage, ConvertToBisPartitionId(targetPartitionName)));

    auto blockStorage = MakeBlockStorageFrom(std::move(pStorage));

    NN_UTILTOOL_LOG_FORCE("storageSize: %llu", blockStorage->GetTotalSize());

    NN_UTILTOOL_RESULT_DO(
        WriteSectorsByZero(blockStorage.get(), 0, blockStorage->GetTotalSize()));

    NN_RESULT_SUCCESS;

}

nn::Result SystemInitializer::EnableUpdatingFromBootImagePackage(int64_t target)
{
    if (target == static_cast<int64_t>(nn::updater::TargetBootMode::Normal))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::updater::MarkVerifyingRequired(nn::updater::TargetBootMode::Normal, s_StorageWritingBuffer, sizeof(s_StorageWritingBuffer)));
    }
    else if (target == static_cast<int64_t>(nn::updater::TargetBootMode::Safe))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::updater::MarkVerifyingRequired(nn::updater::TargetBootMode::Safe, s_StorageWritingBuffer, sizeof(s_StorageWritingBuffer)));
    }
    else
    {
        NN_ABORT("unknown target type: %lld", target);
    }

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::ReadMbrFromStorage(Mbr *pOut)
{
    NN_UTILTOOL_RESULT_DO(m_pBlockStorage->Read(pOut, sizeof(Mbr), 0, 1));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::ReadPrimaryGptFromStorage(GptHeader *pOut)
{
    NN_UTILTOOL_RESULT_DO(m_pBlockStorage->Read(pOut, sizeof(GptHeader), 1, 1));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::ReadAlternateGptFromStorage(GptHeader *pOut)
{
    NN_UTILTOOL_RESULT_DO(m_pBlockStorage->Read(pOut, sizeof(GptHeader), m_pBlockStorage->GetTotalBlocks() - 1, 1));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::ReadGptPartitionsFromStorage(GptPartitionEntry *pOutArray, int numOutArray, const GptHeader &gptHeader)
{
    NN_ABORT_UNLESS(gptHeader.numberOfPartitionEntries <= numOutArray);
    int64_t readSectors = CalculateNumberOfSectors(gptHeader.sizeOfPartitionEntry * gptHeader.numberOfPartitionEntries, m_pBlockStorage->GetBlockSize());
    int64_t startingLba = gptHeader.partitionEntryLba;

    NN_UTILTOOL_RESULT_DO(m_pBlockStorage->Read(pOutArray, sizeof(GptPartitionEntry) * numOutArray, startingLba, readSectors));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::EnsureProtectProductionInfo(bool allowOverwritingProductionInfo, int64_t gptIndex, int64_t gptPartitionIndex, int64_t partitionAddressIndex)
{
    NN_UTILTOOL_LOG_VERBOSE(
        "EnsureProtectProductionInfo: gptHeader=%lld(%s), partitions=%lld(%s), partitionAddresses=%lld(%s)\n",
        gptIndex, m_InitialImage.GetPartitionInfo(gptIndex).name,
        gptPartitionIndex, m_InitialImage.GetPartitionInfo(gptPartitionIndex).name,
        partitionAddressIndex, m_InitialImage.GetPartitionInfo(partitionAddressIndex).name);

    NN_UTILTOOL_LOG_DEBUG(
        "allowOverwritingProductionInfo: %d, enableOverwritingProductionInfo: %d\n",
        allowOverwritingProductionInfo, m_Option.enableOverwritingProductionInfo);

    if(allowOverwritingProductionInfo && m_Option.enableOverwritingProductionInfo)
    {
        NN_RESULT_SUCCESS;
    }

    GptHolder oldGpt;
    GptHolder newGpt;
    std::vector<int> oldProductionInfoPartitionIndexes;
    std::vector<int> newProductionInfoPartitionIndexes;
    InitialImageCommandPartition commandPartition;
    std::vector<PartitionAddress> partitionAddresses;

    NN_UTILTOOL_RESULT_DO(oldGpt.Read(m_pBlockStorage));
    NN_UTILTOOL_RESULT_DO(ReadGptFromImage(&newGpt, gptIndex, gptPartitionIndex));
    GetProductionInfoPartitionIndexes(&oldProductionInfoPartitionIndexes, &oldGpt);
    GetProductionInfoPartitionIndexes(&newProductionInfoPartitionIndexes, &newGpt);
    NN_UTILTOOL_RESULT_DO(m_InitialImage.ReadCommandPartition(&commandPartition));
    NN_UTILTOOL_RESULT_DO(ReadPartitionAddresses(&partitionAddresses, partitionAddressIndex));

    if(HasProductionInfoPartition(*oldGpt.GetHeader(), oldGpt.GetPartitions()))
    {
        NN_UTILTOOL_LOG_DEBUG("Found prodution info partitions");

        bool hasCommandsWriteProductionInfo;
        NN_UTILTOOL_RESULT_DO(
            HasCommandsWriteProductionInfo(&hasCommandsWriteProductionInfo, commandPartition, &oldGpt, &oldProductionInfoPartitionIndexes, &newGpt, &newProductionInfoPartitionIndexes, &partitionAddresses));
        if(hasCommandsWriteProductionInfo)
        {
            NN_UTILTOOL_LOG_ERROR("Don't write the partition of production info. Use a valid initial image.");
            NN_UTILTOOL_RESULT_THROW(nn::fs::ResultTargetLocked());
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::UpdateSecureInfo(int64_t index)
{
    NN_UTILTOOL_LOG_VERBOSE("UpdateSecureInfo: index %lld", index);

    std::unique_ptr<nn::fs::IStorage> pBootPartition1;

    std::unique_ptr<uint8_t[]> bct(new uint8_t[BctSize]);
    std::unique_ptr<uint8_t[]> eks(new uint8_t[EksSize]);

    NN_UTILTOOL_RESULT_DO(
        nn::fs::OpenBisPartition(&pBootPartition1, nn::fs::BisPartitionId::BootPartition1Root));

    NN_UTILTOOL_RESULT_DO(
        ReadBct(bct.get(), BctSize, pBootPartition1.get(), index));

    int32_t bootLoaderVersion;
    NN_UTILTOOL_RESULT_DO(
        ReadBootLoaderVersion(&bootLoaderVersion, bct.get(), BctSize));

    int32_t eksIndex = CalculateEksIndex(bootLoaderVersion);

    NN_UTILTOOL_RESULT_DO(
        ReadEks(eks.get(), EksSize, pBootPartition1.get(), eksIndex));

    NN_UTILTOOL_RESULT_DO(
        UpdateSecureInfo(bct.get(), eks.get()));

    NN_UTILTOOL_LOG_VERBOSE("Parameter: bctIndex=%lld, bootLoaderVersion=%d, eksIndex=%d", index, bootLoaderVersion, eksIndex);

    NN_UTILTOOL_RESULT_DO(
        WriteBct(pBootPartition1.get(), index, bct.get(), BctSize));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::UpdateSecureInfo(void *bct)
{
    std::unique_ptr<nn::fs::IStorage> pBootPartition1;

    NN_UTILTOOL_RESULT_DO(
        nn::fs::OpenBisPartition(&pBootPartition1, nn::fs::BisPartitionId::BootPartition1Root));

    NN_UTILTOOL_RESULT_DO(
        UpdateSecureInfo(bct, pBootPartition1.get()));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::UpdateSecureInfo(void *bct, nn::fs::IStorage *pStorage)
{
    std::unique_ptr<uint8_t[]> eks(new uint8_t[EksSize]);

    int32_t bootLoaderVersion;
    NN_UTILTOOL_RESULT_DO(
        ReadBootLoaderVersion(&bootLoaderVersion, bct, BctSize));

    int32_t eksIndex = CalculateEksIndex(bootLoaderVersion);

    NN_UTILTOOL_RESULT_DO(
        ReadEks(eks.get(), EksSize, pStorage, eksIndex));

    NN_UTILTOOL_RESULT_DO(
        UpdateSecureInfo(bct, eks.get()));

    NN_UTILTOOL_LOG_VERBOSE("Parameter: bootLoaderVersion=%d, eksIndex=%d", bootLoaderVersion, eksIndex);

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::UpdateSecureInfo(void *pOutBctBuffer, void *pSourceEksBuffer)
{
    const size_t CopySize = 0xB0;
    const int64_t SourceEksAddressBegin = 0x0000;
    const int64_t DestinationEksAddressBegin = GetBctEksOffset();

    std::memcpy(reinterpret_cast<uint8_t*>(pOutBctBuffer) + DestinationEksAddressBegin, reinterpret_cast<uint8_t*>(pSourceEksBuffer) + SourceEksAddressBegin, CopySize);

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::ReadBct(void* bct, size_t size, nn::fs::IStorage *pStorage, int64_t index)
{
    NN_ABORT_UNLESS(size == BctSize);

    NN_UTILTOOL_RESULT_DO(
        pStorage->Read(BctAddressBase + BctAddressOffset * index, bct, BctSize));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::ReadEks(void* eks, size_t size, nn::fs::IStorage *pStorage, int64_t index)
{
    NN_ABORT_UNLESS(size == EksSize);

    NN_UTILTOOL_RESULT_DO(
        pStorage->Read(EksAddressBase + EksAddressOffset * index, eks, EksSize));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::WriteBct(nn::fs::IStorage *pStorage, int64_t index, void *buffer, size_t size)
{
    NN_ABORT_UNLESS(size == BctSize);

    NN_UTILTOOL_RESULT_DO(
        pStorage->Write(BctAddressBase + BctAddressOffset * index, buffer, size));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::ReadBootLoaderVersion(int32_t *pOut, void* buffer, size_t size)
{
    NN_ABORT_UNLESS(size == BctSize);
    const int64_t VersionAddress = GetBctVersionOffset();

    *pOut = *reinterpret_cast<int32_t*>(reinterpret_cast<uint8_t*>(buffer) + VersionAddress);

    NN_ABORT_UNLESS(0 <= *pOut && *pOut <= 32);

    NN_RESULT_SUCCESS;
}

int32_t SystemInitializer::CalculateEksIndex(int32_t bootLoaderVersion)
{
    NN_ABORT_UNLESS(0 <= bootLoaderVersion && bootLoaderVersion <= 32);

    if(bootLoaderVersion == 0)
    {
        return 0;
    }
    else
    {
        return bootLoaderVersion - 1;
    }
}

nn::Result SystemInitializer::HasCommandsWriteProductionInfo(bool *pOut, const InitialImageCommandPartition &commandPartition, GptHolder *pOldGpt, std::vector<int> *pOldProductionInfoPartitionIndexes, GptHolder *pNewGpt, std::vector<int> *pNewProductionInfoPartitionIndexes, std::vector<PartitionAddress> *pPartitionAddresses)
{
    for (int i=0; i< commandPartition.numCommand; i++)
    {
        const auto &command = commandPartition.commands[i];

        switch(command.command)
        {
        case InitialImageCommandType_WritePartitionTable:
            {
                bool hasUpdate = true;
                NN_UTILTOOL_RESULT_DO(
                    HasUpdateProductionInfoPartition(&hasUpdate, pOldGpt, pOldProductionInfoPartitionIndexes, pNewGpt, pNewProductionInfoPartitionIndexes, pPartitionAddresses));

                if(hasUpdate)
                {
                    NN_UTILTOOL_LOG_ERROR("Found updates to production info partition.");
                    *pOut = true;
                    NN_RESULT_SUCCESS;
                }
            }
            break;
        case InitialImageCommandType_WritePartitionImage:
        case InitialImageCommandType_WriteSparsePartitionImage:
            {
                auto found = std::find(pNewProductionInfoPartitionIndexes->begin(), pNewProductionInfoPartitionIndexes->end(), command.args[0]);

                if( found == pNewProductionInfoPartitionIndexes->end() )
                {
                    NN_UTILTOOL_LOG_DEBUG("Found a write command to prodution info partition");
                    *pOut = true;
                    NN_RESULT_SUCCESS;
                }
            }
            break;
        case InitialImageCommandType_WriteFsPartitionImage:
        case InitialImageCommandType_WriteFsSparsePartitionImage:
            {
                if(IsProductionInfoFsPartitionId(command.stringArgument))
                {
                    NN_UTILTOOL_LOG_DEBUG("Found a write command to prodution info partition");
                    *pOut = true;
                    NN_RESULT_SUCCESS;
                }
            }
            break;
        case InitialImageCommandType_EnsureProtectProductionInfo:
        case InitialImageCommandType_UpdateSecureInfo:
        case InitialImageCommandType_WriteBatchedBootPartitions:
        case InitialImageCommandType_EraseEmmcWithEscapingData:
        case InitialImageCommandType_InitializeZero:
        case InitialImageCommandType_EnableUpdatingFromBootImagePackage:
        case InitialImageCommandType_WriteDatabase:
            break;

        default:
            NN_ABORT("unknown command: %lld", command.command);
        }
    }

    *pOut = false;
    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::HasUpdateProductionInfoPartition(bool *pOut, GptHolder *pOldGpt, std::vector<int> *pOldProductionInfoPartitionIndexes, GptHolder *pNewGpt, std::vector<int> *pNewProductionInfoPartitionIndexes, std::vector<PartitionAddress> *pPartitionAddresses)
{
    if(pOldProductionInfoPartitionIndexes->size() != pNewProductionInfoPartitionIndexes->size())
    {
        NN_UTILTOOL_LOG_DEBUG("Mismatch number of prodution info partitions. before=%zu, after=%zu", pOldProductionInfoPartitionIndexes->size(), pNewProductionInfoPartitionIndexes->size());
        *pOut = true;
        NN_RESULT_SUCCESS;
    }

    CalculatePartitionAddress(pNewGpt->GetHeader(), pNewGpt->GetPartitions(), pOldGpt->GetHeader(), pOldGpt->GetPartitions(), pPartitionAddresses->size(), &(pPartitionAddresses->at(0)), m_pBlockStorage->GetTotalBlocks() * m_pBlockStorage->GetBlockSize(),  m_pBlockStorage->GetBlockSize());

    for(size_t j = 0; j < pOldProductionInfoPartitionIndexes->size(); j++)
    {
        const auto &oldPartition = pOldGpt->GetPartitions()[pOldProductionInfoPartitionIndexes->at(j)];
        const auto &newPartition = pNewGpt->GetPartitions()[pNewProductionInfoPartitionIndexes->at(j)];
        if( !(oldPartition.startingLba == newPartition.startingLba &&
                oldPartition.endingLba == newPartition.endingLba))
        {
            NN_UTILTOOL_LOG_DEBUG("Mismatch address of prodution info partition");
            *pOut = true;
            NN_RESULT_SUCCESS;
        }
    }

    *pOut = false;

    NN_RESULT_SUCCESS;
}

nn::Result SystemInitializer::ReadGptFromImage(GptHolder *pOut, int64_t gptIndex, int64_t gptPartitionIndex)
{
    size_t readSize;
    NN_UTILTOOL_RESULT_DO(
        m_InitialImage.ReadPartition(&readSize, gptIndex, 0, pOut->GetHeader(), sizeof(GptHeader)));
    NN_UTILTOOL_RESULT_DO(
        m_InitialImage.ReadPartition(&readSize, gptPartitionIndex, 0, pOut->GetPartitions(), sizeof(GptPartitionEntry) * pOut->GetHeader()->numberOfPartitionEntries));

    NN_RESULT_SUCCESS;
}

void SystemInitializer::GetProductionInfoPartitionIndexes(std::vector<int> *pOut, GptHolder *pGpt)
{
    pOut->resize(GPT_DEFAULT_PARTITION_ENTRIES);
    int num;
    ::GetProductionInfoPartitionIndexes(&num, &(pOut->at(0)), GPT_DEFAULT_PARTITION_ENTRIES, *(pGpt->GetHeader()), pGpt->GetPartitions());
    pOut->resize(num);
}

nn::Result SystemInitializer::ReadPartitionAddresses(std::vector<PartitionAddress> *pOut, int64_t partitionAddressIndex)
{
    size_t readSize;
    int numPartitonAddress;

    NN_UTILTOOL_RESULT_DO(
        m_InitialImage.ReadPartition(&readSize, partitionAddressIndex, 0, &numPartitonAddress, sizeof(numPartitonAddress)));

    pOut->resize(numPartitonAddress);

    NN_UTILTOOL_LOG_DEBUG("Num partition addresses = %d.", numPartitonAddress);

    NN_UTILTOOL_RESULT_DO(
        m_InitialImage.ReadPartition(&readSize, partitionAddressIndex, sizeof(numPartitonAddress), &(pOut->at(0)), sizeof(PartitionAddress) * numPartitonAddress));

    NN_RESULT_SUCCESS;
}

bool SystemInitializer::VerifyGptPartition(const GptHeader &gptHeader, const GptPartitionEntry partitions[])
{
    GptHeader header = gptHeader;
    header.headerCrc32 = 0;

    Crc32 crc32;
    uint32_t partitionEntryArrayCrc32 = crc32.Calculate(partitions, sizeof(GptPartitionEntry) * header.numberOfPartitionEntries);
    uint32_t headerCrc32 = crc32.Calculate(&header, header.headerSize);

    return gptHeader.partitionEntryArrayCrc32 == partitionEntryArrayCrc32 && gptHeader.headerCrc32 == headerCrc32;
}


nn::Result SystemInitializer::WriteDatabase(const char* targetPartitionName, int64_t sourceIndex)
{
    // nn::fs::SetLocalSystemAccessLogForDebug(true);

    if (!HasSaveDataSpaceIdName(targetPartitionName))
    {
        NN_ABORT("Has no partition: FsPartitionId=%s", targetPartitionName);
    }

    NN_UTILTOOL_RESULT_DO(
        WriteNcmDatabase(&m_InitialImage, ConvertToSaveDataSpaceId(targetPartitionName), sourceIndex));

    NN_RESULT_SUCCESS;
}

nn::Result WriteInitialImageFromFile(IBlockStorage *pBlockStorage, IFile *pImageFile, SystemIntializerOption option)
{
    SystemInitializer systemInitializer;
    NN_UTILTOOL_RESULT_DO(
        systemInitializer.Initialize(pBlockStorage, pImageFile, option));
    NN_UTILTOOL_RESULT_DO(
        systemInitializer.InitializeSystem());

    NN_RESULT_SUCCESS;
}
