﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "systemInitializer_FsFile.h"
#include <nn/nn_Abort.h>
#include <nn/util/util_BitUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs_Base.h>

#if defined NN_TOOL_USE_CMAC_FILE

#include <nn/manu/manu_Api.h>

nn::Result FsFile::OpenRead(const char *path)
{
    m_Path = path;

    NN_RESULT_SUCCESS;
}

nn::Result FsFile::OpenWrite(const char *path)
{
    m_Path = path;

    NN_RESULT_SUCCESS;
}

void FsFile::Close()
{
    m_Path = nullptr;
}

nn::Result FsFile::Write(int64_t offset, const void* buffer, size_t size, bool flush)
{
    NN_RESULT_DO(
        nn::manu::WriteToHost(buffer, static_cast<size_t>(size), m_Path, offset, size));

    NN_RESULT_SUCCESS;
}

nn::Result FsFile::Read(size_t* pOut, int64_t offset, void* buffer, size_t size)
{
    NN_RESULT_DO(
        nn::manu::ReadFromHost(buffer, size, m_Path, offset, size));
    *pOut = size;

    NN_RESULT_SUCCESS;
}

nn::Result FsFile::GetSize(int64_t* pOut)
{
    size_t outSize;
    NN_RESULT_DO(
        nn::manu::GetFileSize(&outSize, m_Path));

    *pOut = static_cast<int64_t>(outSize);

    NN_RESULT_SUCCESS;
}

nn::Result FsFile::Flush()
{
    NN_RESULT_SUCCESS;
}


bool FsFile::IsValid()
{
    return m_Path != nullptr;
}

nn::Result FsFile::Exists(bool *pOut, const char *name)
{
    size_t outSize;
    nn::Result result = nn::manu::GetFileSize(&outSize, name);
    *pOut = result.IsSuccess();

    NN_RESULT_SUCCESS;
}

nn::Result FsFile::Create(const char *path)
{
    uint8_t buffer[8];
    NN_RESULT_DO(
        nn::manu::WriteToHost(buffer, 0, path, 0, 0));

    NN_RESULT_SUCCESS;
}

#else

nn::Result FsFile::OpenRead(const char *path)
{
    NN_RESULT_DO(
        nn::fs::OpenFile(&m_Handle, path, nn::fs::OpenMode_Read));

    NN_RESULT_SUCCESS;
}

nn::Result FsFile::OpenWrite(const char *path)
{
    NN_RESULT_DO(
        nn::fs::OpenFile(&m_Handle, path, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));

    NN_RESULT_SUCCESS;
}

void FsFile::Close()
{
    nn::fs::CloseFile(m_Handle);
}

nn::Result FsFile::Write(int64_t offset, const void* buffer, size_t size, bool flush)
{
    nn::fs::WriteOption option = {};
    if (flush)
    {
        option = nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush);
    }

    NN_RESULT_DO(
        nn::fs::WriteFile(m_Handle, offset, buffer, size, option));

    NN_RESULT_SUCCESS;
}

nn::Result FsFile::Read(size_t* pOut, int64_t offset, void* buffer, size_t size)
{
    NN_RESULT_DO(
        nn::fs::ReadFile(pOut, m_Handle, offset, buffer, size));

    NN_RESULT_SUCCESS;
}

nn::Result FsFile::GetSize(int64_t* pOut)
{
    NN_RESULT_DO(
        nn::fs::GetFileSize(pOut, m_Handle));

    NN_RESULT_SUCCESS;
}

nn::Result FsFile::Flush()
{
    NN_RESULT_DO(
        nn::fs::FlushFile(m_Handle));
    NN_RESULT_SUCCESS;
}


bool FsFile::IsValid()
{
    return m_Handle.handle != 0;
}

nn::Result FsFile::Exists(bool *pOut, const char *name)
{
    nn::fs::FileHandle handle;

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, name, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            *pOut = false;

            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    *pOut = true;

    NN_RESULT_SUCCESS;
}

nn::Result FsFile::Create(const char *path)
{
    NN_RESULT_DO(
        nn::fs::CreateFile(path, 0));

    NN_RESULT_SUCCESS;
}

#endif
