﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "systemInitializer_BuildInBlockStorage.h"
#include <nn/nn_Abort.h>
#include <nn/util/util_BitUtil.h>

#include <nn/TargetConfigs/build_Os.h>
#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/utilTool/utilTool_ResultHandlingUtility.h>
#include <nn/result/result_HandlingUtility.h>

#ifndef NN_BUILD_CONFIG_OS_WIN32

#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_IStorage.h>

class BuildInStorage : public IBlockStorage
{
public:
    explicit BuildInStorage();
    virtual ~BuildInStorage();
    virtual uint64_t GetTotalSize();
    virtual uint64_t GetBlockSize();
    virtual uint64_t GetTotalBlocks();
    virtual nn::Result Write(uint64_t blockAddress, uint64_t numBlocks, const void* pData, size_t dataSize);
    virtual nn::Result Read(void* pOutBuffer, size_t bufferSize, uint64_t blockAddress, uint64_t numBlocks);
private:
    std::unique_ptr<nn::fs::IStorage> m_pFsStorage;
};

BuildInStorage::BuildInStorage()
{
    NN_UTILTOOL_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::OpenBisPartition(&m_pFsStorage, nn::fs::BisPartitionId::UserDataRoot));
}

BuildInStorage::~BuildInStorage()
{
}

uint64_t BuildInStorage::GetBlockSize()
{
    return 512;
}

uint64_t BuildInStorage::GetTotalBlocks()
{
    int64_t totalSize;
    NN_ABORT_UNLESS(m_pFsStorage->GetSize(&totalSize).IsSuccess());
    return totalSize / GetBlockSize();
}

uint64_t BuildInStorage::GetTotalSize()
{
    int64_t totalSize;
    NN_ABORT_UNLESS(m_pFsStorage->GetSize(&totalSize).IsSuccess());
    return totalSize;
}

nn::Result BuildInStorage::Write(uint64_t blockAddress, uint64_t numBlocks, const void* pData, size_t dataSize)
{
    //NN_ABORT_UNLESS(nn::util::is_aligned(reinterpret_cast<uintptr_t>(pData), 4096U));

    NN_UTILTOOL_RESULT_DO(
        m_pFsStorage->Write(static_cast<uint32_t>(blockAddress) * GetBlockSize(), pData, dataSize));

    NN_RESULT_SUCCESS;
}

nn::Result BuildInStorage::Read(void* pOutBuffer, size_t bufferSize, uint64_t blockAddress, uint64_t numBlocks)
{
    //NN_ABORT_UNLESS(nn::util::is_aligned(reinterpret_cast<uintptr_t>(pOutBuffer), 4096U));

    NN_UTILTOOL_RESULT_DO(
        m_pFsStorage->Read(static_cast<uint32_t>(blockAddress) * GetBlockSize(), pOutBuffer, bufferSize));

    NN_RESULT_SUCCESS;
}

std::shared_ptr<IBlockStorage> GetBuildInBlockStorage()
{
    return std::shared_ptr<IBlockStorage>(new BuildInStorage());
}

#else

std::shared_ptr<IBlockStorage> GetBuildInBlockStorage()
{
    NN_ABORT("not implemented.");
}

#endif

class DryRunBlockStorage : public IBlockStorage
{
public:
    explicit DryRunBlockStorage(std::shared_ptr<IBlockStorage> pStorage) : m_pStorage(pStorage) {}
    virtual ~DryRunBlockStorage() {}
    virtual uint64_t GetTotalSize() { return m_pStorage->GetTotalSize(); }
    virtual uint64_t GetBlockSize() { return m_pStorage->GetBlockSize(); }
    virtual uint64_t GetTotalBlocks() { return m_pStorage->GetTotalBlocks(); }
    virtual nn::Result Write(uint64_t blockAddress, uint64_t numBlocks, const void* pData, size_t dataSize)
    {
        NN_UTILTOOL_LOG_DEBUG("DryRunWrite: address=0x%08llx, numBlock=%lld, dataSize=%d", blockAddress, numBlocks, dataSize);
        return nn::ResultSuccess();
    }
    virtual nn::Result Read(void* pOutBuffer, size_t bufferSize, uint64_t blockAddress, uint64_t numBlocks)
    {
        return m_pStorage->Read(pOutBuffer, bufferSize, blockAddress, numBlocks);
    }
private:
    static const uint64_t MMC_TOTAL_BLOCKS = 0x1d5a000;

    std::shared_ptr<IBlockStorage> m_pStorage;
};


std::shared_ptr<IBlockStorage> GetBuildInBlockStorage(bool isDryRun)
{
    if(isDryRun)
    {
        return std::shared_ptr<IBlockStorage>(new DryRunBlockStorage(GetBuildInBlockStorage()));
    }
    else
    {
        return GetBuildInBlockStorage();
    }
}
