﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "systemInitializer_Allocator.h"

#include <nn/os.h>
#include <nn/mem.h>
#include <nn/init.h>

#if (defined NN_TOOL_REPAIR_INITIALIZER)

namespace
{
    auto s_Allocator = nn::init::GetAllocator();
}

void* Allocate(size_t size)
{
    return s_Allocator->Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    return s_Allocator->Free(p);
}

void* operator new(size_t size)
{
    return Allocate(size);
}

void operator delete(void* pv) NN_NOEXCEPT
{
    Deallocate(pv, 0);
}

void* operator new[](size_t size)
{
    return Allocate(size);
}

void operator delete[](void* pv) NN_NOEXCEPT
{
    Deallocate(pv, 0);
}

#else

namespace{
    const size_t AllocatorBufferSize = 32 * 1024 * 1024;
    uint8_t s_AllocatorBuffer[AllocatorBufferSize];
    nn::mem::StandardAllocator s_Allocator(s_AllocatorBuffer, AllocatorBufferSize);
} // namespace

void* Allocate(size_t size)
{
    return s_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    return s_Allocator.Free(p);
}

#ifndef NN_BUILD_CONFIG_OS_WIN32

void * operator new ( std::size_t size )
{
    return s_Allocator.Allocate(size);
}

void operator delete ( void * ptr ) NN_NOEXCEPT
{
    s_Allocator.Free(ptr);
}

#endif // #ifndef NN_BUILD_CONFIG_OS_WIN32

#endif // #if (defined NN_TOOL_REPAIR_INITIALIZER)
