﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "systemInitializer_InitialImage.h"
#include "../Utility/systemInitializer_IFile.h"
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/utilTool/utilTool_ResultHandlingUtility.h>

nn::Result InitialImage::Initialize(IFile *pImageFile)
{
    NN_ABORT_UNLESS_EQUAL(m_IsInitialized, false);

    m_pImageFile = pImageFile;
    NN_RESULT_DO(ReadInfo(&m_ImageInfo));

    m_IsInitialized = true;

    NN_RESULT_SUCCESS;
}

nn::Result InitialImage::ReadInfo(InitialImageInfo *pOut)
{
    NN_ABORT_UNLESS(m_pImageFile->IsValid());

    size_t readSize;
    NN_RESULT_DO(
        m_pImageFile->Read(&readSize, 0, pOut, sizeof(InitialImageInfo)));
    NN_RESULT_SUCCESS;
}

nn::Result InitialImage::ReadCommandPartition(InitialImageCommandPartition *pOut)
{
    NN_ABORT_UNLESS(m_IsInitialized);

    size_t readSize;

    NN_RESULT_DO(
        ReadPartition(&readSize, 0, 0, &pOut->numCommand, sizeof(int32_t)));

    NN_ABORT_UNLESS(pOut->numCommand <= 64, "Too much initial image commands: %d", pOut->numCommand);

    NN_RESULT_DO(
        ReadPartition(&readSize, 0, 4, pOut->commands, sizeof(InitialImageCommand) * pOut->numCommand));

    NN_RESULT_SUCCESS;
}

nn::Result InitialImage::ReadPartition(size_t* pOut, int64_t partitionIndex, int64_t offset, void* buffer, size_t size)
{
    NN_ABORT_UNLESS(m_IsInitialized);

    return m_pImageFile->Read(pOut, m_ImageInfo.partitions[partitionIndex].address + offset, buffer, size);
}

int InitialImage::GetPartitionCount()
{
    NN_ABORT_UNLESS(m_IsInitialized);

    return m_ImageInfo.numPartitions;
}

InitialImageInfo::PartitionInfo InitialImage::GetPartitionInfo(int64_t index)
{
    NN_ABORT_UNLESS(m_IsInitialized);

    return m_ImageInfo.partitions[index];
}

void PrintInitialImageInfo(const InitialImageInfo &info)
{
    NN_UTILTOOL_LOG_MESSAGE("Signature:%s", info.signature);
    NN_UTILTOOL_LOG_MESSAGE("Version:%d", info.version);
    NN_UTILTOOL_LOG_MESSAGE("NumPartitions:%d", info.numPartitions);
    for(int i = 0; i < info.numPartitions; i++)
    {
        NN_UTILTOOL_LOG_MESSAGE("Partition[%d]: name=%s, address=%08llx, size=%lld",
            i,
            info.partitions[i].name,
            info.partitions[i].address,
            info.partitions[i].size);
    }
}

void PrintCommandPartition(const InitialImageCommandPartition &commandPartition)
{
    NN_UTILTOOL_LOG_MESSAGE("NumCommands: %d", commandPartition.numCommand);

    for(int i = 0; i < commandPartition.numCommand; i++)
    {
        InitialImageCommand command = commandPartition.commands[i];
        NN_UTILTOOL_LOG_MESSAGE("Command[%d]: %s=%lld, args=(%lld, %lld, %lld, %lld, %lld, %lld, %lld, '%s')",
            i,
            GetInitialImageCommandName(static_cast<InitialImageCommandType>(command.command)),
            command.command,
            command.args[0],
            command.args[1],
            command.args[2],
            command.args[3],
            command.args[4],
            command.args[5],
            command.args[6],
            command.stringArgument);
    }
}

const char * GetInitialImageCommandName(InitialImageCommandType commandType)
{
    switch(commandType)
    {
    case InitialImageCommandType_WritePartitionTable:
        return "WritePartitionTable";
    case InitialImageCommandType_WriteRawImage:
        return "WriteRawImage";
    case InitialImageCommandType_UpdatePartitionTable:
        return "UpdatePartitionTable";
    case InitialImageCommandType_WritePartitionImage:
        return "WritePartitionImage";
    case InitialImageCommandType_WriteSparsePartitionImage:
        return "WriteSparsePartitionImage";
    case InitialImageCommandType_WriteFsPartitionImage:
        return "WriteFsPartitionImage";
    case InitialImageCommandType_WriteFsSparsePartitionImage:
        return "WriteFsSparsePartitionImage";
    case InitialImageCommandType_EnsureProtectProductionInfo:
        return "EnsureProtectProductionInfo";
    default:
        NN_ABORT("unknown command: %d", commandType);
    }
}
