﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ExportPartition.h"
#include "ListExportablePartitions.h"
#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/utilTool/utilTool_ResultHandlingUtility.h>
#include "Utility/systemInitializer_FsFile.h"
#include "Utility/systemInitializer_FsBisPartition.h"
#include "Utility/systemInitializer_StorageUtility.h"
#include "Utility/systemInitializer_FileSystemUtility.h"
#include "Partition/systemInitializer_Partition.h"

#include <nn/nn_Abort.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_IStorage.h>

namespace
{
    const int64_t StorageWritingBufferSize = 4 * 1024 * 1024;
    NN_ALIGNAS(4096) static uint8_t s_StorageWritingBuffer[StorageWritingBufferSize];

    nn::Result CopyStorageToFile(IFile *pOutFile, nn::fs::IStorage *pSource, int64_t beginAddress, int64_t size)
    {
        NN_UTILTOOL_LOG_VERBOSE("Begin to export %lld bytes.", size);
        NN_ABORT_UNLESS_ALIGNED(size, 512);

        int64_t storageSize;

        NN_UTILTOOL_RESULT_DO(
            pSource->GetSize(&storageSize));

        NN_ABORT_UNLESS(beginAddress + size <= storageSize);

        int64_t restBytes = size;
        uint64_t currentSourceOffset = beginAddress;
        uint64_t currentOutputOffset = 0;

        while(0 < restBytes)
        {
            size_t writeChunkSize = static_cast<size_t>(std::min(restBytes, static_cast<int64_t>(StorageWritingBufferSize)));

            NN_UTILTOOL_LOG_PROGRESS("Write %zu bytes. Progress: %lld/%lld.", writeChunkSize, currentSourceOffset, size);

            NN_UTILTOOL_RESULT_DO(
                pSource->Read(currentSourceOffset, s_StorageWritingBuffer, writeChunkSize));

            NN_UTILTOOL_RESULT_DO(
                pOutFile->Write(currentOutputOffset, s_StorageWritingBuffer, writeChunkSize, false));

            restBytes -= writeChunkSize;
            currentSourceOffset += writeChunkSize;
            currentOutputOffset += writeChunkSize;
        }

        NN_UTILTOOL_LOG_VERBOSE("Finish to export %lld bytes.", size);

        NN_RESULT_SUCCESS;
    }
}

nn::Result ExportPartition(std::string outputPath, std::string type, std::string name, int64_t size, int64_t beginAddress)
{
    bool hasPartition;
    NN_UTILTOOL_RESULT_DO(
        HasExportablePartition(&hasPartition, type, name));

    if (!hasPartition)
    {
        NN_UTILTOOL_LOG_ERROR("Found no partitions(type='%s', name='%s').", type.c_str(), name.c_str());
        NN_RESULT_THROW(nn::fs::ResultPartitionNotFound());
    }

    bool exist;
    NN_UTILTOOL_RESULT_DO(
        FsFile::Exists(&exist, outputPath.c_str()));

    if(!exist)
    {
        NN_UTILTOOL_RESULT_DO(
            FsFile::Create(outputPath.c_str()));
    }

    FsFile file;
    NN_UTILTOOL_RESULT_DO(
        file.OpenWrite(outputPath.c_str()));
    NN_UTIL_SCOPE_EXIT { file.Flush(); file.Close(); };

    NN_UTILTOOL_LOG_DEBUG("image opened");

    std::unique_ptr<nn::fs::IStorage> blockStorage;
    NN_UTILTOOL_RESULT_DO(
        OpenBlockStorage(&blockStorage, type, name));

    if(size == 0)
    {
        NN_UTILTOOL_RESULT_DO(
            blockStorage->GetSize(&size));
    }

    NN_UTILTOOL_RESULT_DO(
        ExportPartition(&file, blockStorage.get(), size, beginAddress));

    NN_RESULT_SUCCESS;
}

nn::Result ExportPartition(IFile *pOut, nn::fs::IStorage *pSource, int64_t size, int64_t beginAddress)
{
    int64_t storageSize;

    NN_UTILTOOL_RESULT_DO(
        pSource->GetSize(&storageSize));

    NN_ABORT_UNLESS(beginAddress + size <= storageSize);

    NN_UTILTOOL_RESULT_DO(
        CopyStorageToFile(pOut, pSource, beginAddress, size));

    NN_RESULT_SUCCESS;
}
