﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>

#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Base.h>

#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_Thread.h>

#include <nn/ldr/ldr_ProcessManagerApi.h>
#include <nn/spl/spl_Api.h>
#include <nn/pm/pm_Result.h>

#include <nn/sf/sf_HipcServer.h>


#include "pm_Spec.h"

namespace nn { namespace pm {

    namespace
    {
        enum MemoryRegion
        {
            MemoryRegion_Application     = 0,
            MemoryRegion_Applet          = 1,
            MemoryRegion_SecureSystem    = 2,
            MemoryRegion_NonSecureSystem = 3,
        };

        enum ResourceLimitGroup
        {
            ResourceLimitGroup_System,
            ResourceLimitGroup_Application,
            ResourceLimitGroup_Applet,
            ResourceLimitGroup_Count,
        };

        const svc::LimitableResource LimitableResourceNames[] =
        {
            svc::LimitableResource_PhysicalMemoryMax,
            svc::LimitableResource_ThreadCountMax,
            svc::LimitableResource_EventCountMax,
            svc::LimitableResource_TransferMemoryCountMax,
            svc::LimitableResource_SessionCountMax,
        };

        const int LimitableResourceNameCount =
            sizeof(LimitableResourceNames) / sizeof(LimitableResourceNames[0]);

        const int64_t PhysicalMemorySystemReservedSize = 0x00500000;
        int64_t g_PhysicalMemoryLimits[ResourceLimitGroup_Count] = {0};

        int64_t LimitValues[ResourceLimitGroup_Count][LimitableResourceNameCount] =
        {
            // System
            {
                0,   // PhysicalMemoryMax（実行時に設定）
                608, // ThreadCountMax
                600, // EventCountMax
                128, // TransferMemoryCountMax
                794, // SessionCountMax
            },
            // Application
            {
                0,   // PhysicalMemoryMax（実行時に設定）
                96,  // ThreadCountMax
                0,   // EventCountMax
                32,  // TransferMemoryCountMax
                1,   // SessionCountMax
            },
            // Applet
            {
                0,   // PhysicalMemoryMax（実行時に設定）
                96,  // ThreadCountMax
                0,   // EventCountMax
                32,  // TransferMemoryCountMax
                5,   // SessionCountMax
            },
        };


        os::SdkMutex g_LimitValuesMutex;
        int64_t g_CurrentBoostSystemMemorySize = 0;
        svc::Handle                 g_ResourceLimit[ResourceLimitGroup_Count];


        void WaitResourceReleased(const svc::Handle& resourceLimit) NN_NOEXCEPT
        {
            for (auto i = 0; i < LimitableResourceNameCount; ++i)
            {
                while (true)
                {
                    int64_t value;
                    auto result = svc::GetResourceLimitCurrentValue(&value, resourceLimit, LimitableResourceNames[i]);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                    if (value == 0)
                    {
                        break;
                    }
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
                }
            }
        }

        void WaitApplicationHeapReleased() NN_NOEXCEPT
        {
            while (true)
            {
                Bit64 value;

                auto result = svc::GetSystemInfo(&value, svc::SystemInfoType_UsingPhysicalMemorySize, svc::Handle(0), pm::MemoryRegion_Application);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                if (value == 0)
                {
                    break;
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
            }
        }

        void SetResourceLimitLimitValues(svc::Handle resourceLimit, const svc::LimitableResource* limitableResourceNames, const int64_t* limitValues, int count)
        {
            for (auto i = 0; i < count; ++i)
            {
                auto resource = limitableResourceNames[i];
                auto value = limitValues[i];
                Result result = svc::SetResourceLimitLimitValue(resourceLimit, resource, value);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }

        void SetResourceLimitGroupMemoryLimitUnsafe(int group, int64_t limitSize)
        {
            NN_SDK_ASSERT(group >= 0 && group < ResourceLimitGroup_Count);
            NN_SDK_ASSERT(g_LimitValuesMutex.IsLockedByCurrentThread());

            LimitValues[group][svc::LimitableResource_PhysicalMemoryMax] = limitSize;

            SetResourceLimitLimitValues(
                g_ResourceLimit[group],
                LimitableResourceNames,
                LimitValues[group],
                LimitableResourceNameCount);
        }

        Result SetResourceLimitApplicationPhysicalMemoryMaxValues(int64_t limitSize)
        {
            NN_SDK_ASSERT(g_LimitValuesMutex.IsLockedByCurrentThread());

            auto result = svc::SetResourceLimitLimitValue(g_ResourceLimit[ResourceLimitGroup_Application], svc::LimitableResource_PhysicalMemoryMax, limitSize);
            if (result.IsFailure())
            {
                return result;
            }
            LimitValues[ResourceLimitGroup_Application][svc::LimitableResource_PhysicalMemoryMax] = limitSize;

            return ResultSuccess();
        }

        ResourceLimitGroup GetResourceLimitGroup(const nn::ldr::ProgramInfo& pi) NN_NOEXCEPT
        {
            if ((pi.flags & nn::ldr::ProgramInfoFlag_ProgramTypeMask) == nn::ldr::ProgramInfoFlag_Application)
            {
                return ResourceLimitGroup_Application;
            }
            if ((pi.flags & nn::ldr::ProgramInfoFlag_ProgramTypeMask) == nn::ldr::ProgramInfoFlag_Applet)
            {
                return ResourceLimitGroup_Applet;
            }
            return ResourceLimitGroup_System;
        }

        svc::Handle GetResourceLimit(ResourceLimitGroup limitGroup) NN_NOEXCEPT
        {
            return g_ResourceLimit[limitGroup];
        }

    }
    // anonymouse namespace


    Result InitializeSpec() NN_NOEXCEPT
    {
        Result result;

        // ResourceLimit のハンドル生成
        for (auto i = 0; i < ResourceLimitGroup_Count; ++i)
        {
            if (i != ResourceLimitGroup_System)
            {
                result = svc::CreateResourceLimit(&g_ResourceLimit[i]);
                NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);
            }
            else
            {
                Bit64 value;
                result = svc::GetInfo(&value, nn::svc::InfoType_ResourceLimit, nn::svc::Handle(0), 0);
                NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);
                g_ResourceLimit[i] = nn::svc::Handle(value);
            }
        }

        // PhysicalMemoryLimits の取得
        for (auto i = 0; i < ResourceLimitGroup_Count; ++i)
        {
            Bit64 value;
            int64_t limitSize = 0;
            switch (i)
            {
            case ResourceLimitGroup_System:
                {
                    // システムのメモリサイズを取得
                    auto result = svc::GetResourceLimitLimitValue(&limitSize, g_ResourceLimit[ResourceLimitGroup_System], svc::LimitableResource_PhysicalMemoryMax);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    NN_ABORT_UNLESS(limitSize >= 0);

                    // アプリとアプレットのサイズを引く
                    result = svc::GetSystemInfo(&value, nn::svc::SystemInfoType_PhysicalMemorySize, nn::svc::Handle(0), MemoryRegion_Application);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    NN_ABORT_UNLESS(value >= 0);
                    limitSize -= value;
                    NN_ABORT_UNLESS(limitSize >= 0);

                    result = svc::GetSystemInfo(&value, nn::svc::SystemInfoType_PhysicalMemorySize, nn::svc::Handle(0), MemoryRegion_Applet);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    NN_ABORT_UNLESS(value >= 0);
                    limitSize -= value;
                    NN_ABORT_UNLESS(limitSize >= 0);

                    // 埋蔵金のサイズを引く
                    NN_ABORT_UNLESS(limitSize >= PhysicalMemorySystemReservedSize);
                    limitSize -= PhysicalMemorySystemReservedSize;
                }
                break;
            case ResourceLimitGroup_Application:
                {
                    result = svc::GetSystemInfo(&value, nn::svc::SystemInfoType_PhysicalMemorySize, nn::svc::Handle(0), MemoryRegion_Application);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    NN_ABORT_UNLESS(value >= 0);
                    limitSize += value;
                }
                break;
            case ResourceLimitGroup_Applet:
                {
                    result = svc::GetSystemInfo(&value, nn::svc::SystemInfoType_PhysicalMemorySize, nn::svc::Handle(0), MemoryRegion_Applet);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    NN_ABORT_UNLESS(value >= 0);
                    limitSize += value;
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
                break;
            }
            g_PhysicalMemoryLimits[i] = limitSize;
        }

        // ResourceLimit の設定
        {
            std::lock_guard<decltype(g_LimitValuesMutex)> lk(g_LimitValuesMutex);

            for (auto i = 0; i < ResourceLimitGroup_Count; ++i)
            {
                SetResourceLimitGroupMemoryLimitUnsafe(i, g_PhysicalMemoryLimits[i]);
            }
        }

        return result;
    }


    Result BoostSystemMemoryResourceLimit(int64_t boostSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(boostSize >= 0, ResultInvalidSize());
        NN_RESULT_THROW_UNLESS(boostSize % nn::os::MemoryPageSize == 0, ResultInvalidSize());

        auto boostedApplicationMemoryLimit = g_PhysicalMemoryLimits[ResourceLimitGroup_Application] - boostSize;
        NN_RESULT_THROW_UNLESS(boostedApplicationMemoryLimit > 0, ResultInvalidSize());

        // ResourceLimit を変更（必ず 減らす → 増やす の順で行う）
        {
            std::lock_guard<decltype(g_LimitValuesMutex)> lk(g_LimitValuesMutex);
            if (boostSize > g_CurrentBoostSystemMemorySize)
            {
                // アプリのリソースを減らす場合
                NN_RESULT_DO( SetResourceLimitApplicationPhysicalMemoryMaxValues(boostedApplicationMemoryLimit) );
                NN_ABORT_UNLESS_RESULT_SUCCESS( svc::SetUnsafeLimit(static_cast<size_t>(boostSize)) );
            }
            else if (boostSize < g_CurrentBoostSystemMemorySize)
            {
                // アプリのリソースを増やす場合
                NN_RESULT_DO( svc::SetUnsafeLimit(static_cast<size_t>(boostSize)) );
                NN_ABORT_UNLESS_RESULT_SUCCESS( SetResourceLimitApplicationPhysicalMemoryMaxValues(boostedApplicationMemoryLimit) );
            }

            g_CurrentBoostSystemMemorySize = boostSize;
        }

        NN_RESULT_SUCCESS;
    }

    svc::Handle GetResourceLimit(const ldr::ProgramInfo& pi) NN_NOEXCEPT
    {
        return GetResourceLimit(GetResourceLimitGroup(pi));
    }

    void WaitResource(const ldr::ProgramInfo& pi) NN_NOEXCEPT
    {
        const auto limitGroupIndex = GetResourceLimitGroup(pi);

        if (limitGroupIndex == ResourceLimitGroup_Application)
        {
            WaitResourceReleased(GetResourceLimit(limitGroupIndex));
            WaitApplicationHeapReleased();
        }
    }


}}  // namespace nn::pm

