﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/util/util_IntrusiveList.h>
#include <nn/os/os_Mutex.h>
#include <algorithm>
#include "pm_ProcessInfo.h"

namespace nn { namespace pm {


    class ProcessList
        : public nn::util::IntrusiveList<ProcessInfo, ProcessInfo>
    {
    private:
        os::Mutex m_Lock;

    public:
        ProcessList() NN_NOEXCEPT
            : m_Lock(false)
        {
        }

        ProcessInfo*    FindByHandle(svc::Handle handle) NN_NOEXCEPT
        {
            return FindImpl(&ProcessInfo::GetHandle, handle);
        }
        ProcessInfo*    FindById(os::ProcessId processId) NN_NOEXCEPT
        {
            return FindImpl(&ProcessInfo::GetId, processId);
        }
        ProcessInfo*    FindByProgramId(ncm::ProgramId pgid) NN_NOEXCEPT
        {
            return FindImpl(&ProcessInfo::GetProgramId, pgid);
        }

        void Remove(ProcessInfo* p) NN_NOEXCEPT
        {
            erase(GetIterator(p));
        }

        void Lock() NN_NOEXCEPT
        {
            m_Lock.Lock();
        }
        void Unlock() NN_NOEXCEPT
        {
            m_Lock.Unlock();
        }

    private:
        template <typename ValueT>
        ProcessInfo* FindImpl(ValueT (ProcessInfo::*f)() const, ValueT value) NN_NOEXCEPT
        {
            for( iterator i = begin(); i != end(); ++i )
            {
                if( ((*i).*f)() == value )
                {
                    return &*i;
                }
            }

            return NULL;
        }
        iterator GetIterator(const ProcessInfo* p) NN_NOEXCEPT
        {
            for( iterator i = begin(); i != end(); ++i )
            {
                if( &*i == p )
                {
                    return i;
                }
            }

            return end();
        }
    };

    template <typename T>
    class LockedAccessor;

    template <typename T>
    class LockedObject
    {
        friend class LockedAccessor<T>;

    private:
        T   m_Object;
    };

    template <typename T>
    class LockedAccessor
    {
    private:
        typedef LockedObject<T> TargetObjectType;

    private:
        TargetObjectType&   m_LockedObject;

    public:
        explicit LockedAccessor(TargetObjectType* pObj) NN_NOEXCEPT
        : m_LockedObject(*pObj)
        {
            m_LockedObject.m_Object.Lock();
        }
        ~LockedAccessor() NN_NOEXCEPT
        {
            m_LockedObject.m_Object.Unlock();
        }
        NN_IMPLICIT operator T*() NN_NOEXCEPT
        {
            return &m_LockedObject.m_Object;
        }
        T* operator ->() NN_NOEXCEPT
        {
            return &m_LockedObject.m_Object;
        }
        T& operator *() NN_NOEXCEPT
        {
            return  m_LockedObject.m_Object;
        }
        NN_IMPLICIT operator T*() const NN_NOEXCEPT
        {
            return &m_LockedObject.m_Object;
        }
        T* operator ->() const NN_NOEXCEPT
        {
            return &m_LockedObject.m_Object;
        }
        T& operator *()  const NN_NOEXCEPT
        {
            return  m_LockedObject.m_Object;
        }
    };


}}  // namespace nn::pm

