﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <memory>

#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_StdAllocationPolicy.h>
#include <nn/os/os_Thread.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/init/init_Startup.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/ncm/ncm_ServiceName.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentManagerFactory.h>

#include <nn/ncm/ncm_PrivateService.h>

#include <nn/lr/lr_ServiceName.h>
#include <nn/lr/lr_LocationResolverManagerFactory.h>
#include <nn/lr/lr_LocationResolverManagerImpl.h>

#include <nn/spl/spl_Api.h>

using namespace nn;
using namespace nn::os;
using namespace nn::ncm;
using namespace nn::lr;

namespace
{
    typedef sf::ObjectFactory<nn::sf::StdAllocationPolicy<std::allocator>> LocationResolverManagerFactory;

    uint8_t g_ContentManagerThreadStack[16 * 1024] NN_ALIGNAS(StackRegionAlignment);
    uint8_t g_LocationResolverThreadStack[16 * 1024] NN_ALIGNAS(StackRegionAlignment);

    struct ContentManagerServerManagerOption
    {
        static const size_t PointerTransferBufferSize = 1024;
    };

    class ContentManagerServerManager : public sf::HipcSimpleAllInOneServerManager < 32, 3, ContentManagerServerManagerOption >
    {
    public:
        explicit ContentManagerServerManager(sf::SharedPointer<IContentManager> contentManager)
            : m_ContentManager(contentManager)
        {}

        Result Initialize()
        {
            NN_RESULT_DO(RegisterObjectForPort(m_ContentManager, 16, ContentManagerServiceName));

            Start();

            NN_RESULT_SUCCESS;
        }

        Result StartThreads() NN_NOEXCEPT
        {
            NN_RESULT_DO(CreateThread(&m_Thread, ThreadFunction, this, g_ContentManagerThreadStack, sizeof(g_ContentManagerThreadStack), NN_SYSTEM_THREAD_PRIORITY(ncm, ContentManagerServerIpcSession)));
            SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(ncm, ContentManagerServerIpcSession));
            StartThread(&m_Thread);

            NN_RESULT_SUCCESS;
        }

        void Wait() NN_NOEXCEPT
        {
            WaitThread(&m_Thread);
        }

    private:
        static void ThreadFunction(void* p)
        {
            auto thisManager = reinterpret_cast<ContentManagerServerManager*>(p);
            thisManager->LoopAuto();
        }

        ThreadType m_Thread;
        sf::SharedPointer<IContentManager> m_ContentManager;
    };

    struct LocationResolverServerManagerOption
    {
        static const size_t PointerTransferBufferSize = 1024;
    };

    class LocationResolverServerManager : public sf::HipcSimpleAllInOneServerManager < 32, 1, LocationResolverServerManagerOption >
    {
    public:
        LocationResolverServerManager() NN_NOEXCEPT : m_LocationResolverManager(LocationResolverManagerFactory::CreateSharedEmplaced<ILocationResolverManager, LocationResolverManagerImpl>()) {}

        Result Initialize()
        {
            NN_RESULT_DO(RegisterObjectForPort(m_LocationResolverManager, 16, LocationResolverServiceName));

            Start();

            NN_RESULT_SUCCESS;
        }

        Result StartThreads() NN_NOEXCEPT
        {
            NN_RESULT_DO(CreateThread(&m_Thread, ThreadFunction, this, g_LocationResolverThreadStack, sizeof(g_LocationResolverThreadStack), NN_SYSTEM_THREAD_PRIORITY(ncm, LocationResolverServerIpcSession)));
            SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(ncm, LocationResolverServerIpcSession));
            StartThread(&m_Thread);

            NN_RESULT_SUCCESS;
        }

        void Wait() NN_NOEXCEPT
        {
            WaitThread(&m_Thread);
        }

    private:
        static void ThreadFunction(void* p)
        {
            auto thisManager = reinterpret_cast<LocationResolverServerManager*>(p);
            thisManager->LoopAuto();
        }

        ThreadType m_Thread;
        sf::SharedPointer<ILocationResolverManager> m_LocationResolverManager;
    };

    uint8_t g_HeapMemory[3 * 1024 * 1024];
    lmem::HeapHandle g_HeapHandle;

    void* Allocate(size_t size)
    {
        return nn::lmem::AllocateFromExpHeap(g_HeapHandle, size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        nn::lmem::FreeToExpHeap(g_HeapHandle, p);
    }

    void InitializeHeap()
    {
        g_HeapHandle = lmem::CreateExpHeap(g_HeapMemory, sizeof(g_HeapMemory), lmem::CreationOption_NoOption);
    }

    void NcmMain() NN_NOEXCEPT
    {
        NN_SDK_LOG("[NCM] NcmMain.\n");

#if defined NN_DETAIL_NCM_USE_SPL
        spl::Initialize();
#endif

        fs::InitializeWithMultiSessionForSystem();
        fs::SetAllocator(Allocate, Deallocate);
        fs::SetEnabledAutoAbort(false);

        ContentManagerConfig config = {};
#if defined ENABLE_BUILDING_DATABASE
        config.EnableBuildingContentMetaDatabase = true;
#else
        config.EnableBuildingContentMetaDatabase = false;
#endif

#if defined ENABLE_IMPORT_DATABASE_FROM_SDCARD
    #if defined ENABLE_BUILDING_DATABASE
        #error "Do not specify both ENABLE_BUILDING_DATABASE and ENABLE_IMPORT_DATABASE_FROM_SDCARD concurrently."
    #endif
        config.EnableImportContentMetaDatabaseFromSdCard = true;
#else
        config.EnableImportContentMetaDatabaseFromSdCard = false;
#endif

        auto contentManager = CreateDefaultContentManager(config);

        std::unique_ptr<ContentManagerServerManager> contentManagerServerManager(new ContentManagerServerManager(contentManager));
        NN_ABORT_UNLESS_RESULT_SUCCESS(contentManagerServerManager->Initialize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(contentManagerServerManager->StartThreads());

        NN_SDK_LOG("[NCM] ContentManagerServer started.\n");

        ncm::InitializeWithContentManagerObject(contentManager);

        std::unique_ptr<LocationResolverServerManager> locationResolverServerManager(new LocationResolverServerManager());
        NN_ABORT_UNLESS_RESULT_SUCCESS(locationResolverServerManager->Initialize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(locationResolverServerManager->StartThreads());

        NN_SDK_LOG("[NCM] LocationResolverServer started.\n");

        NN_SDK_LOG("[NCM] Waiting request.\n");

        contentManagerServerManager->Wait();
        locationResolverServerManager->Wait();

#if defined NN_DETAIL_NCM_USE_SPL
        spl::Finalize();
#endif
    }
}

void* operator new(size_t size)
{
    return Allocate(size);
}

void operator delete(void* pv) NN_NOEXCEPT
{
    Deallocate(pv, 0);
}

void* operator new[](size_t size)
{
    return Allocate(size);
}

void operator delete[](void* pv) NN_NOEXCEPT
{
    Deallocate(pv, 0);
}

extern "C" void nninitStartup()
{
    InitializeHeap();
}

extern "C" void nninitInitializeSdkModule()
{
}

extern "C" void nninitFinalizeSdkModule()
{
}

extern "C"
void nnMain()
{
    auto threadType = GetCurrentThread();
    ChangeThreadPriority(threadType, NN_SYSTEM_THREAD_PRIORITY(ncm, MainWaitThreads));
    SetThreadNamePointer(threadType, NN_SYSTEM_THREAD_NAME(ncm, MainWaitThreads));
    NcmMain();
}
