﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BitUtil.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_MemoryHeapApi.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/ldr/ldr_Result.h>

#include <nn/fs_Base.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/lr/lr_Result.h>
#include <nn/lr/lr_Service.h>
#include <nn/lr/lr_LocationResolver.h>
#include <algorithm>

#include "ldr_Loader.h"
#include "ldr_CreateProcess.h"
#include "ldr_ArgumentStore.h"

namespace nn { namespace ldr {

    namespace
    {
        ArgumentStore           g_ArgumentStore;
        lmem::HeapHandle        g_HeapHandle;
        Bit8                    g_HeapBuffer[16 * 1024];

        nn::Result GetProgramPath(char* pBuffer, size_t size, const ncm::ProgramLocation& location) NN_NOEXCEPT
        {
            lr::Path path;
            Result result;

            {
                lr::RegisteredLocationResolver resolver;
                result = lr::OpenRegisteredLocationResolver(&resolver);
                if( result.IsFailure() )
                {
                    return result;
                }

                result = resolver.ResolveProgramPath(&path, location.programId);
                if( result.IsSuccess() )
                {
                    NN_ABORT_UNLESS(lr::IsValid(path));
                    std::memset(pBuffer, 0, size);
                    std::memcpy(pBuffer, path.string, std::min(size, sizeof(path)));

                    return ResultSuccess();
                }
                else if ( !lr::ResultProgramNotFound::Includes(result))
                {
                    return result;
                }
            }
            {
                lr::LocationResolver resolver;
                result = lr::OpenLocationResolver(&resolver, static_cast<ncm::StorageId>(location.storageId));
                if( result.IsFailure() )
                {
                    return result;
                }

                result = resolver.ResolveProgramPath(&path, location.programId);
                if( result.IsFailure() )
                {
                    return result;
                }
                NN_ABORT_UNLESS(lr::IsValid(path));
                std::memset(pBuffer, 0, size);
                std::memcpy(pBuffer, path.string, std::min(size, sizeof(path)));
            }

            return ResultSuccess();
        }

        nn::Result RedirectProgramPath(const char* pBuffer, size_t size, const ncm::ProgramLocation& location) NN_NOEXCEPT
        {
            Result result;

            lr::LocationResolver resolver;
            result = lr::OpenLocationResolver(&resolver, static_cast<ncm::StorageId>(location.storageId));
            if( result.IsFailure() )
            {
                return result;
            }

            lr::Path path;
            std::memcpy(path.string, pBuffer, std::min(size, sizeof(path)));

            resolver.RedirectProgramPath(location.programId, path);

            return ResultSuccess();
        }

        void* FsAlloc(size_t size) NN_NOEXCEPT
        {
            return lmem::AllocateFromExpHeap(g_HeapHandle, size);
        }

        void FsDealloc(void* p, size_t size) NN_NOEXCEPT
        {
            NN_UNUSED(size);
            lmem::FreeToExpHeap(g_HeapHandle, p);
        }

        void InitializeFile() NN_NOEXCEPT
        {
            g_HeapHandle = lmem::CreateExpHeap(g_HeapBuffer, sizeof(g_HeapBuffer), 0);
            nn::fs::SetAllocator(FsAlloc, FsDealloc);
        }
    }

    void Loader::Initialize() NN_NOEXCEPT
    {
        InitializeFile();
        nn::lr::Initialize();
    }

    nn::Result Loader::CreateProcess(nn::svc::Handle* pOut, PinId pinId, int flags, nn::svc::Handle resourceLimit) NN_NOEXCEPT
    {
        Result result;

        char path[nn::fs::EntryNameLengthMax];

        ncm::ProgramLocation programLocation;
        result = GetProgramLocationFromPinId(&programLocation, pinId);
        if( result.IsFailure() )
        {
            return result;
        }

        // パスを解決する
        result = GetProgramPath(path, sizeof(path), programLocation);
        if( result.IsFailure() )
        {
            return result;
        }
        // 強制的に終端させる
        path[sizeof(path) - 1] = '\0';

        ncm::ProgramId programId = programLocation.programId;
        return ldr::CreateProcess(pOut, pinId, path, g_ArgumentStore.Get(programId), flags, resourceLimit);
    }

    nn::Result Loader::GetProgramInfo(nn::ldr::ProgramInfo* pOut, const ncm::ProgramLocation& location) NN_NOEXCEPT
    {
        Result result;

        std::memset(pOut, 0, sizeof(nn::ldr::ProgramInfo));

        char path[nn::fs::EntryNameLengthMax];

        // パスを解決する
        result = GetProgramPath(path, sizeof(path), location);
        if( result.IsFailure() )
        {
            return result;
        }
        // 強制的に終端させる
        path[sizeof(path) - 1] = '\0';

        auto programId = location.programId;
        result = ldr::GetProgramInfo(pOut, programId, path);
        if( result.IsFailure() )
        {
            return result;
        }

        if( programId.value != pOut->programId ) // FIXME
        {
            // npdm に記載されている programId でリダイレクト
            ncm::ProgramLocation correctedLocation = location;
            correctedLocation.programId.value = pOut->programId;
            result = RedirectProgramPath(path, sizeof(path), correctedLocation);
            if( result.IsFailure() )
            {
                return result;
            }

            // プログラム引数を再設定
            const ArgumentStore::Entry* entry = g_ArgumentStore.Get(programId);
            if( entry != NULL )
            {
                result = SetProgramArgument(correctedLocation.programId, entry->argument, entry->argumentSize);
                if( result.IsFailure() )
                {
                    return result;
                }
            }
        }

        return ResultSuccess();
    }

    nn::Result Loader::SetProgramArgument(ncm::ProgramId programId, const void* pArgument, size_t size) NN_NOEXCEPT
    {
        return g_ArgumentStore.Set(programId, pArgument, size);
    }

    nn::Result Loader::FlushArguments() NN_NOEXCEPT
    {
        return g_ArgumentStore.Flush();
    }

    nn::Result Loader::PinProgram(PinId* pOutId, const ncm::ProgramLocation& location) NN_NOEXCEPT
    {
        *pOutId = {};
        return ldr::PinProgram(pOutId, location);
    }

    nn::Result Loader::UnpinProgram(PinId id) NN_NOEXCEPT
    {
        return ldr::UnpinProgram(id);
    }

    nn::Result Loader::GetProcessModuleInfo(int* pOutCount, ldr::ModuleInfo* pBuffer, int num, nn::os::ProcessId id) NN_NOEXCEPT
    {
        *pOutCount = 0;
        std::memset(pBuffer, 0, num * sizeof(ldr::ModuleInfo));
        return ldr::GetProcessModuleInfo(pOutCount, pBuffer, num, id);
    }

}}  // nn::ldr
