﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>
#include <random>

#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/fs_Base.h>
#include <nn/fs/fs_Code.h>
#include <nn/ldr/ldr_Types.h>
#include <nn/ldr/ldr_ProcessManagerApi.h>
#include <nn/ldr/ldr_Result.h>
#include <nn/util/util_BitPack.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_TFormatString.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_MemoryMapSelect.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <lz4.h>

#include "ldr_CreateProcess.h"
#include "ldr_DataTypes.h"
#include "ldr_MetaLoader.h"
#include "ldr_CapabilityTest.h"
#include "ldr_AutoClose.h"
#include "ldr_RoManager.h"
#include "ldr_Random.h"

namespace nn { namespace ldr {

    namespace
    {
        const size_t LoadAddressAlign = 0x00200000;
        const size_t MaxSystemResourceSize = 510ul * 1024 * 1024; // align_down(512MB - 128KB, 2MB)

        class MemoryWriter
        {
        public:
            explicit MemoryWriter(Bit8* p) NN_NOEXCEPT
                : m_pPos(p)
            {
            }
            template<typename T>
            void Write(T v) NN_NOEXCEPT
            {
                Write(&v, sizeof(v));
            }
            void Write(const void* p, size_t size) NN_NOEXCEPT
            {
                std::memcpy(m_pPos, p, size);
                m_pPos += size;
            }

        private:
            Bit8*   m_pPos;
        };

        enum
        {
            RtldNso,
            MainNso,
            SubSdkNsoStart,
            SubSdkNsoEnd = SubSdkNsoStart + 9,
            SdkNso,
            NumOfNso,
        };

        const char* GetNsoName(int index) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(0 <= index && index < NumOfNso);
            // "code:/subsdkX" が収まる長さにする
            static char g_Buffer[16];

            switch (index)
            {
            case RtldNso:
                return "code:/rtld";
            case MainNso:
                return "code:/main";
            case SdkNso:
                return "code:/sdk";
            default:
                {
                    nn::util::TSNPrintf(g_Buffer, sizeof(g_Buffer), "code:/subsdk%d", index - SubSdkNsoStart);
                }
                return g_Buffer;
            }
        }

        MetaLoader                  g_MetaLoader;
        NsoHeader                   g_NsoBuffer[NumOfNso];

        struct ProcessCreateInfo
        {
            nnHandle        processHandle;
            uint64_t        argumentAddress;
            size_t          argumentMappingSize;
            uint64_t        nsoLoadAddress[NumOfNso];
            size_t          nsoMemorySize[NumOfNso];
        };

        Result MountCode(const char* pProgramPath, ncm::ProgramId programId) NN_NOEXCEPT
        {
            char ncaPath[nn::fs::EntryNameLengthMax];

            std::strncpy(ncaPath, pProgramPath, sizeof(ncaPath));
            for(size_t i = 0; i < NN_ARRAY_SIZE(ncaPath) && ncaPath[i] != '\0'; i++ )
            {
                if( ncaPath[i] == '\\' )
                {
                    ncaPath[i] = '/';
                }
            }

            Result result = fs::MountCode("code", ncaPath, programId);
            if( result.IsFailure() )
            {
                return result;
            }

            return ResultSuccess();
        }
        void UnmountCode() NN_NOEXCEPT
        {
            fs::Unmount("code");
        }
        class CodeMounter
        {
        public:
            explicit CodeMounter(const char* path, ncm::ProgramId programId) NN_NOEXCEPT
            {
                m_Result = MountCode(path, programId);
            }
            ~CodeMounter() NN_NOEXCEPT
            {
                if( m_Result.IsSuccess() )
                {
                    UnmountCode();
                }
            }
            Result GetResult() const NN_NOEXCEPT
            {
                return m_Result;
            }
            bool IsFailure() const NN_NOEXCEPT
            {
                return m_Result.IsFailure();
            }

        private:
            Result  m_Result;
        };

        Result SearchFreeRegion(uintptr_t* pOut, size_t mappingSize) NN_NOEXCEPT
        {
            uintptr_t heapBegin;
            uintptr_t heapSize;
            uintptr_t rsvBegin;
            uintptr_t rsvSize;
            uintptr_t aslrBegin;
            uintptr_t aslrSize;
            Bit64 value;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_HeapRegionAddress, svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0));
            heapBegin = value;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_HeapRegionSize, svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0));
            heapSize = value;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_ReservedRegionAddress, svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0));
            rsvBegin = value;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_ReservedRegionSize, svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0));
            rsvSize = value;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_AslrRegionAddress, svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0));
            aslrBegin = value;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_AslrRegionSize, svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0));
            aslrSize = value;

            uintptr_t addr = aslrBegin;
            for (;;)
            {
                nn::svc::MemoryInfo  memoryInfo;
                nn::svc::PageInfo   pageInfo;

                if (addr + mappingSize <= addr)
                {
                    return nn::svc::ResultOutOfMemory();
                }

                if (heapSize && !(addr + mappingSize - 1 < heapBegin || heapBegin + heapSize - 1 < addr))
                {
                    if (heapBegin + heapSize <= addr)
                    {
                        return nn::svc::ResultOutOfMemory();
                    }
                    addr = heapBegin + heapSize;
                    continue;
                }

                if (rsvSize && !(addr + mappingSize - 1 < rsvBegin || rsvBegin + rsvSize - 1 < addr))
                {
                    if (rsvBegin + rsvSize <= addr)
                    {
                        return nn::svc::ResultOutOfMemory();
                    }
                    addr = rsvBegin + rsvSize;
                    continue;
                }

                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::QueryMemory(&memoryInfo, &pageInfo, addr));
                if (memoryInfo.state == nn::svc::MemoryState_Free && ((memoryInfo.baseAddress + memoryInfo.size) - addr) >= mappingSize)
                {
                    *pOut = addr;
                    return nn::ResultSuccess();
                }
                if (memoryInfo.baseAddress + memoryInfo.size <= addr)
                {
                    return nn::svc::ResultOutOfMemory();
                }
                if (memoryInfo.baseAddress + memoryInfo.size - 1 >= aslrBegin + aslrSize - 1)
                {
                    return nn::svc::ResultOutOfMemory();
                }
                addr = memoryInfo.baseAddress + memoryInfo.size;
            }
        }

        bool IsApplication(const MetaLoader::Npdm& npdm)
        {
            const auto pKc = reinterpret_cast<const util::BitPack32*>(npdm.pKc);
            const auto kcCount = npdm.pAci->kcSize / sizeof(Bit32);
            const auto programInfoFlag = MakeProgramInfoFlag(pKc, kcCount);
            if ( (programInfoFlag & ProgramInfoFlag_ProgramTypeMask) == ProgramInfoFlag_Application )
            {
                return true;
            }
            return false;
        }

        bool IsApplet(const MetaLoader::Npdm& npdm)
        {
            const auto pKc = reinterpret_cast<const util::BitPack32*>(npdm.pKc);
            const auto kcCount = npdm.pAci->kcSize / sizeof(Bit32);
            const auto programInfoFlag = MakeProgramInfoFlag(pKc, kcCount);
            if ( (programInfoFlag & ProgramInfoFlag_ProgramTypeMask) == ProgramInfoFlag_Applet )
            {
                return true;
            }
            return false;
        }

        nn::Result MakeCreateProcessParameterFlags(
            Bit32*                  pOut,
            const MetaLoader::Npdm& npdm,
            int                     flags) NN_NOEXCEPT
        {
            Bit32 parameterFlags = 0;

            util::BitPack8 metaFlags = { npdm.pMeta->flags };
            if( metaFlags.Get<Meta::Is64BitInstruction>() )
            {
                parameterFlags |= svc::CreateProcessParameterFlag_64Bit;
            }
            auto spaceSize = metaFlags.Get<Meta::AddressSpaceSize>();
            switch( spaceSize )
            {
            case Meta::AddressSpaceSizeType_32Bit:
                parameterFlags |= svc::CreateProcessParameterFlag_AddressSpace32Bit;
                break;

            case Meta::AddressSpaceSizeType_64BitOld:
                parameterFlags |= svc::CreateProcessParameterFlag_AddressSpace64BitOld;
                break;

            case Meta::AddressSpaceSizeType_32BitNoReserved:
                parameterFlags |= svc::CreateProcessParameterFlag_AddressSpace32BitNoReserved;
                break;

            case Meta::AddressSpaceSizeType_64Bit:
                parameterFlags |= svc::CreateProcessParameterFlag_AddressSpace64Bit;
                break;

            default:
                return ResultInvalidMeta();
            }

            if( (flags & CreateProcessFlags_EnableException) != 0 )
            {
                parameterFlags |= nn::svc::CreateProcessParameterFlag_EnableJitDebug;
            }
            if( (flags & CreateProcessFlags_DisableAslr) == 0 )
            {
                parameterFlags |= nn::svc::CreateProcessParameterFlag_EnableAslr;
            }

            if (IsApplication(npdm))
            {
                parameterFlags |= nn::svc::CreateProcessParameterFlag_IsApplication;
            }

            util::BitPack32 acidFlags = { npdm.pAcid->flags };
            switch(acidFlags.Get<AcidHeaderData::MemoryRegion>())
            {
            case MemoryRegion_Application:
                if (IsApplet(npdm))
                {
                    // FIXME: Ocean を新 desc でリビルドするまではこのワークアラウンドが必要
                    parameterFlags |= nn::svc::CreateProcessParameterFlag_UseMemory1;
                }
                else
                {
                    parameterFlags |= nn::svc::CreateProcessParameterFlag_UseMemory0;
                }
                break;

            case MemoryRegion_Applet:
                parameterFlags |= nn::svc::CreateProcessParameterFlag_UseMemory1;
                break;

            case MemoryRegion_SecureSystem:
                parameterFlags |= nn::svc::CreateProcessParameterFlag_UseMemory2;
                break;

            case MemoryRegion_NonSecureSystem:
                parameterFlags |= nn::svc::CreateProcessParameterFlag_UseMemory3;
                break;

            default:
                return ResultInvalidMeta();
            }

            *pOut = parameterFlags;
            return ResultSuccess();
        }

        nn::Result UpdateLoadAddress(
                uint64_t*                       pNsoLoadAddress,
                uint64_t*                       pArgumentAddress,
                svc::CreateProcessParameter*    pParameter,
                const bool*                     pHasNso,
                size_t                          memorySize) NN_NOEXCEPT
        {
            uint64_t memoryBegin;
            size_t leftSpaceSize;
            size_t loadOffset = 0;

            switch (pParameter->flags & svc::CreateProcessParameterFlag_AddressSpaceMask)
            {
            case svc::CreateProcessParameterFlag_AddressSpace64BitOld:
                {
                    memoryBegin = NN_SVC_ADDR_SMALL_MAP36_BEGIN;

                    if (NN_SVC_ADDR_SMALL_MAP36_END < memoryBegin + memorySize)
                    {
                        return nn::svc::ResultOutOfMemory();
                    }
                    leftSpaceSize = NN_SVC_ADDR_SMALL_MAP36_END - (memoryBegin + memorySize);
                }
                break;
            case svc::CreateProcessParameterFlag_AddressSpace64Bit:
                {
                    memoryBegin = NN_SVC_ADDR_MAP39_BEGIN;

                    if (NN_SVC_ADDR_MAP39_END < memoryBegin + memorySize)
                    {
                        return nn::svc::ResultOutOfMemory();
                    }
                    leftSpaceSize = NN_SVC_ADDR_MAP39_END - (memoryBegin + memorySize);
                }
                break;
            case svc::CreateProcessParameterFlag_AddressSpace32Bit:
            case svc::CreateProcessParameterFlag_AddressSpace32BitNoReserved:
                {
                    memoryBegin = NN_SVC_ADDR_SMALL_MAP32_BEGIN;

                    if (NN_SVC_ADDR_SMALL_MAP32_END < memoryBegin + memorySize)
                    {
                        return nn::svc::ResultOutOfMemory();
                    }
                    leftSpaceSize = NN_SVC_ADDR_SMALL_MAP32_END - (memoryBegin + memorySize);
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
                break;
            }

            if (pParameter->flags & nn::svc::CreateProcessParameterFlag_EnableAslr)
            {
                loadOffset += nn::ldr::random::GetValue(leftSpaceSize / LoadAddressAlign) * LoadAddressAlign;
            }

            memoryBegin += loadOffset;
            for (int i = 0; i < NumOfNso; i++)
            {
                if (pHasNso[i])
                {
                    pNsoLoadAddress[i] += memoryBegin;
                }
            }
            if (pArgumentAddress)
            {
                *pArgumentAddress += memoryBegin;
            }

            pParameter->memoryAddress   = memoryBegin;
            pParameter->memoryNumPages  = util::DivideUp(memorySize, os::MemoryPageSize);

            return ResultSuccess();
        }

        nn::Result CreateProcessImpl(
                        ProcessCreateInfo*          pInfo,
                        NsoHeader*                  pNsoHeader,
                        const MetaLoader::Npdm&     npdm,
                        const bool*                 pHasNso,
                        const ArgumentStore::Entry* pArgument,
                        int                         flags,
                        const svc::Handle&          resourceLimit ) NN_NOEXCEPT
        {
            Result          result;
            nn::svc::Handle processHandle;
            size_t          memorySize = 0;
            uint64_t        argumentAddress = 0;
            size_t          argumentMappingSize = 0;
            uint64_t        nsoLoadAddress[NumOfNso] = {};
            size_t          nsoMemorySize[NumOfNso] = {};
            bool            argmentAllocated = false;

            // CreateProcess 用のパラメータを作成
            svc::CreateProcessParameter parameter = {};

            {
                std::memset(parameter.name, 0, sizeof(parameter.name));
                std::memcpy(parameter.name, npdm.pMeta->name, sizeof(parameter.name) - 1);

                parameter.version         = npdm.pMeta->version;
                parameter.programId       = npdm.pAci->programId.value;
                parameter.flags           = 0;
                parameter.resourceLimit   = resourceLimit;
                parameter.extraResourcePageCount = 0;

                result = MakeCreateProcessParameterFlags(&parameter.flags, npdm, flags);
                if (result.IsFailure())
                {
                    return result;
                }

                // メモリ管理用の追加のカーネルリソース用メモリサイズの検証
                if (util::is_aligned(npdm.pMeta->systemResourceSize, nn::os::MemoryBlockUnitSize) == false)
                {
                    NN_SDK_LOG("[ldr]: systemResourceSize %x is not aligned as 0x%zx\n", npdm.pMeta->systemResourceSize, nn::os::MemoryBlockUnitSize);
                    return ResultInvalidSize();
                }
                if (npdm.pMeta->systemResourceSize != 0)
                {
                    // 64bit アプリの場合のみ仮想アドレスメモリの使用を許可（64bitOld はダメ）
                    if ((parameter.flags & svc::CreateProcessParameterFlag_AddressSpace64Bit) == 0)
                    {
                        return ResultInvalidMeta();
                    }

                    // Must be application or applet
                    if (! (IsApplication(npdm) || IsApplet(npdm)))
                    {
                        return ResultInvalidMeta();
                    }

                    // SystemResourceSize の値が最大値を超える場合
                    if (npdm.pMeta->systemResourceSize > MaxSystemResourceSize)
                    {
                        return ResultInvalidMeta();
                    }
                }

                // メモリ管理用の追加のカーネルリソース用メモリのページサイズを設定
                parameter.extraResourcePageCount = npdm.pMeta->systemResourceSize / nn::os::MemoryPageSize;

                // ロード位置の決定
                for (int i = 0; i < NumOfNso; i++)
                {
                    if (pHasNso[i])
                    {
                        nsoLoadAddress[i] = memorySize;
                        uint32_t size = 0;
                        size = std::max(size, pNsoHeader[i].textArrangeOffset + pNsoHeader[i].textSize);
                        size = std::max(size, pNsoHeader[i].rodataArrangeOffset + pNsoHeader[i].rodataSize);
                        size = std::max(size, pNsoHeader[i].dataArrangeOffset + pNsoHeader[i].dataSize + pNsoHeader[i].bssSize);
                        size = nn::util::align_up(size, os::MemoryPageSize);
                        nsoMemorySize[i] = size;
                        memorySize += size;

                        // argment は 最初のDLLでアロケーションされる
                        if (!argmentAllocated && pArgument)
                        {
                            argumentAddress = memorySize;
                            argumentMappingSize = nn::util::align_up(sizeof(int32_t) * 2 + pArgument->argumentSize * 2 + 32 * 1024, os::MemoryPageSize);
                            memorySize += argumentMappingSize;
                            argmentAllocated = true;
                        }
                    }
                }

                result = UpdateLoadAddress(nsoLoadAddress, (argmentAllocated? &argumentAddress: nullptr), &parameter, pHasNso, memorySize);
                if (result.IsFailure())
                {
                    return result;
                }

                for (int i = 0; i < NumOfNso; i++)
                {
                    if (pHasNso[i])
                    {
                        NN_SDK_LOG("[ldr]: %-16.16s %10llx %8llx\n", GetNsoName(i), nsoLoadAddress[i], nsoMemorySize[i]);
                    }
                }
            }


            // CreateProcess
            result = nn::svc::CreateProcess(
                        &processHandle,
                        parameter,
                        reinterpret_cast<const Bit32*>(npdm.pKc),
                        (npdm.pAci->kcSize / sizeof(Bit32)) );

            if (result.IsFailure())
            {
                return result;
            }

            pInfo->processHandle        = processHandle;
            pInfo->argumentAddress      = argumentAddress;
            pInfo->argumentMappingSize  = argumentMappingSize;
            for (int i = 0; i < NumOfNso; i++)
            {
                pInfo->nsoLoadAddress[i] = nsoLoadAddress[i];
                pInfo->nsoMemorySize[i] = nsoMemorySize[i];
            }

            return ResultSuccess();
        }

        Result LoadData(
                    fs::FileHandle nsoFile,
                    size_t fileOffset,
                    size_t fileSize,
                    size_t dataSize,
                    bool isCompressed,
                    bool hasHash,
                    const Bit8 hash[],
                    uintptr_t mapAddress, uintptr_t mapEndAddress)
        {
            Result result;
            if (!isCompressed)
            {
                // 互換性のため
                fileSize = dataSize;
            }
            if (fileSize > dataSize)
            {
                return ResultInvalidNso();
            }
            if (fileSize > 0x7ffffffful || dataSize > 0x7ffffffful)
            {
                return ResultInvalidNso();
            }

            uintptr_t loadAddress = (isCompressed? mapEndAddress - fileSize: mapAddress);

            size_t readSize;
            result = fs::ReadFile(&readSize, nsoFile,
                    fileOffset,
                    reinterpret_cast<void*>(loadAddress),
                    fileSize );
            if (result.IsFailure())
            {
                return result;
            }
            if (readSize != fileSize)
            {
                return ResultInvalidNso();
            }

            if (isCompressed)
            {
                int ret = LZ4_decompress_safe(reinterpret_cast<const char*>(loadAddress), reinterpret_cast<char*>(mapAddress), fileSize, dataSize);
                if (ret != static_cast<int>(dataSize))
                {
                    return ResultInvalidNso();
                }
            }

            if (hasHash)
            {
                Bit8 hashData[NsoHeaderHashSize];
                crypto::GenerateSha256Hash(hashData, sizeof(hashData), reinterpret_cast<void*>(mapAddress), dataSize);
                if (std::memcmp(hashData, hash, NsoHeaderHashSize))
                {
                    NN_SDK_LOG("[ldr]: Hash Verify Failure\n");
                    return ResultInvalidNso();
                }
            }

            return ResultSuccess();
        }

        Result LoadAutoLoadModule(
                    svc::Handle                 processHandle,
                    fs::FileHandle              nsoFile,
                    const NsoHeader&            nsoHeader,
                    uintptr_t                   mapAddress,
                    uint64_t                    loadBaseAddress,
                    size_t                      memorySize) NN_NOEXCEPT
        {
            Result result;

            // 自空間へのメモリマップ
            {
                AutoCloseMap acm(mapAddress, processHandle, loadBaseAddress, memorySize);

                if( acm.IsFailure() )
                {
                    return acm.GetResult();
                }

                // 自動ロードモジュールのロード
                {
                    result = LoadData(
                            nsoFile,
                            nsoHeader.textFileOffset, nsoHeader.textFileSize, nsoHeader.textSize,
                            !!(nsoHeader.flags & NsoFlags_TextCompress),
                            !!(nsoHeader.flags & NsoFlags_TextHash),
                            nsoHeader.textHash, mapAddress + nsoHeader.textArrangeOffset,
                            mapAddress + memorySize);
                    if( result.IsFailure() )
                    {
                        return result;
                    }

                    result = LoadData(
                            nsoFile,
                            nsoHeader.rodataFileOffset, nsoHeader.rodataFileSize, nsoHeader.rodataSize,
                            !!(nsoHeader.flags & NsoFlags_RodataCompress),
                            !!(nsoHeader.flags & NsoFlags_RodataHash),
                            nsoHeader.rodataHash, mapAddress + nsoHeader.rodataArrangeOffset,
                            mapAddress + memorySize);
                    if( result.IsFailure() )
                    {
                        return result;
                    }

                    result = LoadData(
                            nsoFile,
                            nsoHeader.dataFileOffset, nsoHeader.dataFileSize, nsoHeader.dataSize,
                            !!(nsoHeader.flags & NsoFlags_DataCompress),
                            !!(nsoHeader.flags & NsoFlags_DataHash),
                            nsoHeader.dataHash, mapAddress + nsoHeader.dataArrangeOffset,
                            mapAddress + memorySize);
                    if( result.IsFailure() )
                    {
                        return result;
                    }

                    // ゼロクリア
                    std::memset(reinterpret_cast<void*>(mapAddress),                                                        0, nsoHeader.textArrangeOffset);
                    std::memset(reinterpret_cast<void*>(mapAddress + nsoHeader.textArrangeOffset   + nsoHeader.textSize),   0, nsoHeader.rodataArrangeOffset - (nsoHeader.textArrangeOffset + nsoHeader.textSize));
                    std::memset(reinterpret_cast<void*>(mapAddress + nsoHeader.rodataArrangeOffset + nsoHeader.rodataSize), 0, nsoHeader.dataArrangeOffset   - (nsoHeader.rodataArrangeOffset + nsoHeader.rodataSize));
                    std::memset(reinterpret_cast<void*>(mapAddress + nsoHeader.dataArrangeOffset   + nsoHeader.dataSize),   0, nsoHeader.bssSize);
                }

                // アンマップ
            }

            // パーミッション変更
            {
                if (util::align_up(nsoHeader.textSize, os::MemoryPageSize))
                {
                    result = nn::svc::SetProcessMemoryPermission(
                            processHandle,
                            loadBaseAddress + nsoHeader.textArrangeOffset,
                            util::align_up(nsoHeader.textSize, os::MemoryPageSize),
                            nn::svc::MemoryPermission_ReadExecute );
                    if( result.IsFailure() )
                    {
                        return result;
                    }
                }

                if (util::align_up(nsoHeader.rodataSize, os::MemoryPageSize))
                {
                    result = nn::svc::SetProcessMemoryPermission(
                            processHandle,
                            loadBaseAddress + nsoHeader.rodataArrangeOffset,
                            util::align_up(nsoHeader.rodataSize, os::MemoryPageSize),
                            nn::svc::MemoryPermission_Read );
                    if( result.IsFailure() )
                    {
                        return result;
                    }
                }

                size_t rwSize = nsoHeader.dataSize + nsoHeader.bssSize;
                if (rwSize)
                {
                    result = nn::svc::SetProcessMemoryPermission(
                            processHandle,
                            loadBaseAddress + nsoHeader.dataArrangeOffset,
                            util::align_up(rwSize, os::MemoryPageSize),
                            nn::svc::MemoryPermission_ReadWrite );
                    if( result.IsFailure() )
                    {
                        return result;
                    }
                }
            }

            return ResultSuccess();
        }

        Result LoadAutoLoadModules(
                const ProcessCreateInfo&    pci,
                const NsoHeader*            pNsoHeader,
                const bool*                 pHasNso,
                const ArgumentStore::Entry* pArgument) NN_NOEXCEPT
        {
            uintptr_t mapAddress;
            Result result;
            //-------------------------------------------------
            // プロセスのコードをロード
            for (int i = 0; i < NumOfNso; i++)
            {
                if (pHasNso[i])
                {
                    fs::FileHandle nsoFile;
                    result = fs::OpenFile(&nsoFile, GetNsoName(i), fs::OpenMode_Read);
                    if (result.IsFailure())
                    {
                        return result;
                    }

                    {
                        AutoCloseFile acf(nsoFile);

                        result = SearchFreeRegion(&mapAddress, pci.nsoMemorySize[i]);
                        if (result.IsFailure())
                        {
                            return result;
                        }
                        result = LoadAutoLoadModule(pci.processHandle, nsoFile, pNsoHeader[i], mapAddress, pci.nsoLoadAddress[i], pci.nsoMemorySize[i]);
                        if (result.IsFailure())
                        {
                            return result;
                        }
                    }
                }
            }

            if (pArgument)
            {
                {
                    result = SearchFreeRegion(&mapAddress, pci.argumentMappingSize);
                    if (result.IsFailure())
                    {
                        return result;
                    }
                    AutoCloseMap acm(mapAddress, pci.processHandle, pci.argumentAddress, pci.argumentMappingSize);

                    if (acm.IsFailure())
                    {
                        return acm.GetResult();
                    }

                    // 引数データのロード
                    {
                        // 引数領域の構造
                        //  0x00    0x20    ヘッダ
                        //  0x20    size    引数データ

                        ArgumentRegionHeader header;
                        std::memset(&header, 0, sizeof(header));
                        header.regionSize = static_cast<int32_t>(pci.argumentMappingSize);
                        header.argumentSize = static_cast<int32_t>(pArgument->argumentSize);

                        MemoryWriter mw(reinterpret_cast<Bit8*>(mapAddress));
                        mw.Write(header);
                        mw.Write(pArgument->argument, pArgument->argumentSize);
                    }
                }

                {
                    result = nn::svc::SetProcessMemoryPermission(
                            pci.processHandle,
                            pci.argumentAddress,
                            pci.argumentMappingSize,
                            nn::svc::MemoryPermission_ReadWrite);
                    if (result.IsFailure())
                    {
                        return result;
                    }
                }
            }

            return ResultSuccess();
        }
    }

    nn::Result LoadAutoLoadHeaders(NsoHeader* pNsoHeader, bool* pHasNso) NN_NOEXCEPT
    {
        for (int i = 0; i < NumOfNso; i++)
        {
            pHasNso[i] = false;
        }

        for (int i = 0; i < NumOfNso; i++)
        {
            //-------------------------------------------------
            // nso ヘッダをロード
            fs::FileHandle nsoFile;
            Result result = fs::OpenFile(&nsoFile, GetNsoName(i), fs::OpenMode_Read);
            if( result.IsSuccess() )
            {
                AutoCloseFile acf(nsoFile);

                {
                    size_t readSize;
                    result = fs::ReadFile(&readSize, nsoFile, 0, pNsoHeader + i, sizeof(*pNsoHeader));
                    if( result.IsFailure() )
                    {
                        return result;
                    }
                    if( readSize != sizeof(*pNsoHeader) )
                    {
                        return ResultInvalidNso();
                    }
                }
                pHasNso[i] = true;
            }
            else if (SubSdkNsoStart <= i && i <= SubSdkNsoEnd)
            {
                // SubSdkNsoX が見つからなかった場合、次以降を検索しない
                i = SubSdkNsoEnd;
            }
        }

        return ResultSuccess();
    }

    // NSOの構成をチェックする
    nn::Result CheckAutoLoad(const NsoHeader* pNsoHeader, const bool* pHasNso) NN_NOEXCEPT
    {
        if (!pHasNso[MainNso])
        {
            return ResultInvalidNso();
        }

        if (!pHasNso[RtldNso] && !pHasNso[SdkNso])
        {
            // 1つの NSO構成であることのチェック
            for (int i = 0; i < NumOfNso; i++)
            {
                if (i != MainNso && pHasNso[i])
                {
                    return ResultInvalidNso();
                }
            }

            if (pNsoHeader[MainNso].textArrangeOffset != 0)
            {
                return ResultInvalidNso();
            }
        }
        else if (pHasNso[RtldNso])
        {
            for (int i = 0; i < NumOfNso; i++)
            {
                if (pNsoHeader[i].textArrangeOffset != 0)
                {
                    return ResultInvalidNso();
                }
            }
        }
        else
        {
            return ResultInvalidNso();
        }

        return ResultSuccess();
    }


    nn::Result CreateProcess(nn::svc::Handle* pOut, PinId pinId, const char* pProgramPath, const ArgumentStore::Entry* pArgument, int flags, nn::svc::Handle resourceLimit) NN_NOEXCEPT
    {
        Result result;
        ncm::ProgramLocation programLocation;

        // パッチの場合、ここで取得できる location の storageId と、
        // pProgramPath でわたってくるパスが指すストレージが異なる場合がある
        if( ! g_RoManager.GetProgramLocation(&programLocation, pinId) )
        {
            return ResultNotPinned();
        }

        std::memset(g_NsoBuffer, 0, sizeof(g_NsoBuffer));
        NsoHeader* pNsoHeader = g_NsoBuffer;

        {
            CodeMounter mount(pProgramPath, programLocation.programId);

            if( mount.IsFailure() )
            {
                return mount.GetResult();
            }


            //-------------------------------------------------
            // npdm をロード

            MetaLoader::Npdm npdm;
            result = g_MetaLoader.GetCachedOrLoad(&npdm, programLocation.programId);
            if( result.IsFailure() )
            {
                return result;
            }


            //-------------------------------------------------
            // npdm の ACI を npdm の ACID で検査

            {
                const util::BitPack32* pKcd = reinterpret_cast<const util::BitPack32*>(npdm.pKcd);
                const util::BitPack32* pKc  = reinterpret_cast<const util::BitPack32*>(npdm.pKc);
                const int kcdCount = npdm.pAcid->kcdSize / sizeof(Bit32);
                const int kcCount  = npdm.pAci->kcSize / sizeof(Bit32);

                // ACI の ProgramId が ACID で設定される範囲内に収まっているか検証
                if (npdm.pAci->programId.value < npdm.pAcid->programIdMin.value || npdm.pAci->programId.value > npdm.pAcid->programIdMax.value)
                {
                    return ResultInvalidProgramId();
                }

                result = TestCapability(pKcd, kcdCount, pKc, kcCount);
                if( result.IsFailure() )
                {
                    return result;
                }
            }

            bool hasNso[NumOfNso];

            result = LoadAutoLoadHeaders(pNsoHeader, hasNso);
            if (result.IsFailure())
            {
                return result;
            }

            result = CheckAutoLoad(pNsoHeader, hasNso);
            if (result.IsFailure())
            {
                return result;
            }

            //-------------------------------------------------
            // CreateProcess
            // TODO: プロセスごとに指定可能なインターフェース
            ProcessCreateInfo pci = {};
            result = CreateProcessImpl(&pci, pNsoHeader, npdm, hasNso, pArgument, flags, resourceLimit);
            if( result.IsFailure() )
            {
                return result;
            }

            {
                AutoCloseHandle ach(pci.processHandle);

                result = LoadAutoLoadModules(pci, pNsoHeader, hasNso, pArgument);
                if( result.IsFailure() )
                {
                    return result;
                }

                ach.Cancel();
            }

            {
                os::ProcessId processId;
                svc::GetProcessId(&processId.value, pci.processHandle);

                util::BitPack8 metaFlags = { npdm.pMeta->flags };
                auto spaceSize = metaFlags.Get<Meta::AddressSpaceSize>();
                bool is64BitAddressSpace = (spaceSize == Meta::AddressSpaceSizeType_64Bit) || (spaceSize == Meta::AddressSpaceSizeType_64BitOld);
                g_RoManager.RegisterProcess(pinId, processId, npdm.pAci->programId.value, is64BitAddressSpace);

                for (int i = 0; i < NumOfNso; i++)
                {
                    if (hasNso[i])
                    {
                        g_RoManager.AddNso(
                            pinId,
                            pNsoHeader[i].moduleId,
                            pci.nsoLoadAddress[i],
                            pci.nsoMemorySize[i] );
                    }
                }
            }

            *pOut = pci.processHandle;
        }

        return ResultSuccess();
    }

    Result GetProgramInfo(
            ProgramInfo*                    pOut,
            ncm::ProgramId                  programId,
            const char*                     pProgramPath ) NN_NOEXCEPT
    {
        MetaLoader::Npdm npdm;
        Result result;

        {
            CodeMounter mount(pProgramPath, programId);

            if( mount.IsFailure() )
            {
                return mount.GetResult();
            }

            result = g_MetaLoader.Load(&npdm, programId);
        }

        if( result.IsSuccess() )
        {
            pOut->threadPriority    = npdm.pMeta->threadPriority;
            pOut->idealCoreNumber   = npdm.pMeta->idealCoreNumber;
            pOut->stackSize         = npdm.pMeta->stackSize;
            pOut->programId         = npdm.pAci->programId.value;
            int offset = 0;

            if (npdm.pAcid->sacdSize > sizeof(pOut->capabilities) - offset)
            {
                return ResultInternalError();
            }
            std::memcpy(&pOut->capabilities[offset], npdm.pSacd, npdm.pAcid->sacdSize);
            pOut->sacdSize          = npdm.pAcid->sacdSize;
            offset                 += npdm.pAcid->sacdSize;

            if (npdm.pAci->sacSize > sizeof(pOut->capabilities) - offset)
            {
                return ResultInternalError();
            }
            std::memcpy(&pOut->capabilities[offset], npdm.pSac, npdm.pAci->sacSize);
            pOut->sacSize          = npdm.pAci->sacSize;
            offset                += npdm.pAci->sacSize;

            if (npdm.pAcid->facdSize > sizeof(pOut->capabilities) - offset)
            {
                return ResultInternalError();
            }
            std::memcpy(&pOut->capabilities[offset], npdm.pFacd, npdm.pAcid->facdSize);
            pOut->facdSize          = npdm.pAcid->facdSize;
            offset                 += npdm.pAcid->facdSize;

            if (npdm.pAci->facSize > sizeof(pOut->capabilities) - offset)
            {
                return ResultInternalError();
            }
            std::memcpy(&pOut->capabilities[offset], npdm.pFac, npdm.pAci->facSize);
            pOut->facSize          = npdm.pAci->facSize;
            offset                += npdm.pAci->facSize;


            const util::BitPack32* pKc  = reinterpret_cast<const util::BitPack32*>(npdm.pKc);
            const int kcCount  = npdm.pAci->kcSize / sizeof(Bit32);
            pOut->flags = MakeProgramInfoFlag(pKc, kcCount);
        }

        return result;
    }

    Result PinProgram(
            PinId*                          pOutId,
            const ncm::ProgramLocation&     programLocation) NN_NOEXCEPT
    {
        if( g_RoManager.Allocate(pOutId, programLocation) )
        {
            return ResultSuccess();
        }
        else
        {
            return ResultMaxProcess();
        }
    }

    Result UnpinProgram(
            PinId                           id) NN_NOEXCEPT
    {
        if( g_RoManager.Free(id) )
        {
            return ResultSuccess();
        }
        else
        {
            return ResultNotPinned();
        }
    }

    Result GetProcessModuleInfo(
            int*                            pOutCount,
            ldr::ModuleInfo*                pBuffer,
            int                             num,
            nn::os::ProcessId               id ) NN_NOEXCEPT
    {
        if( g_RoManager.GetProcessModuleInfo(pOutCount, pBuffer, num, id) )
        {
            return ResultSuccess();
        }
        else
        {
            return ResultNotPinned();
        }
    }


    Result GetProgramLocationFromPinId(ncm::ProgramLocation* outValue, PinId pinId) NN_NOEXCEPT
    {
        if( g_RoManager.GetProgramLocation(outValue, pinId) )
        {
            return ResultSuccess();
        }
        else
        {
            return ResultNotPinned();
        }
    }


}}  // nn::ldr
