﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_Execution.h>

namespace nn { namespace util {

/* ------------------------------------------------------------
    Cancelable
 */
void Cancelable::Cancel() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    auto before = m_UserCancel.exchange(true);
    if (!before && m_pListener)
    {
        m_pListener->Signal();
    }
}
bool Cancelable::IsCanceled() const NN_NOEXCEPT
{
    return m_UserCancel.load();
}
void Cancelable::AttachListener(Listener* pListener) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_SDK_REQUIRES(pListener);
    NN_SDK_ASSERT(!m_pListener);
    m_pListener = pListener;

    if (m_UserCancel.load())
    {
        m_pListener->Signal();
    }
}
void Cancelable::DetachListener() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_SDK_ASSERT(m_pListener);
    m_pListener = nullptr;
}

/* ------------------------------------------------------------
    CancelInjectable
 */
CancelInjectable::CancelInjectable(const Cancelable* pCancelable) NN_NOEXCEPT
    : m_pCancelable(pCancelable)
{
}
bool CancelInjectable::IsCanceled() const NN_NOEXCEPT
{
    return m_pCancelable && m_pCancelable->IsCanceled();
}

/* ------------------------------------------------------------
    Executable
 */
Executable::~Executable() NN_NOEXCEPT
{
    if (m_Initialized)
    {
        os::DestroySystemEvent(&m_Event);
    }
}
Result Executable::Initialize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_Initialized);
    NN_RESULT_DO(os::CreateSystemEvent(&m_Event, os::EventClearMode_ManualClear, true /* interProcess */));
    m_Initialized = true;
    NN_RESULT_SUCCESS;
}
os::NativeHandle Executable::GetReadableHandle() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    return os::GetReadableHandleOfSystemEvent(&m_Event);
}

void Executable::Execute(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Lock.IsLockedByCurrentThread());
    NN_SDK_REQUIRES(m_Initialized);
    NN_SDK_REQUIRES(!HasDone());

    if (IsCanceled())
    {
        m_pResult = util::nullopt;
    }
    else
    {
        m_pResult.emplace(ExecuteImpl(buffer, bufferSize));
    }

    m_Done.store(true);
    os::SignalSystemEvent(&m_Event);
}

bool Executable::HasDone() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    return m_Done.load();
}
util::optional<Result> Executable::TryGetResult() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    return HasDone() ? m_pResult : util::nullopt;
}
bool Executable::TryLock() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    return m_Lock.TryLock();
}
void Executable::Lock() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    m_Lock.Lock();
}
void Executable:: Unlock() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    m_Lock.Unlock();
}

/* ------------------------------------------------------------
    AbstractExecutor
 */
/*
    AbstractExecutor でのタスク管理の概要:
    - 1. タスクが登録される
    - 2. 登録したタスクが実行される
    - 3. タスクの登録が解除される

    2 の実施は任意で、 2 が実施される前に 3 を実施してよい。
    ただし 2 の実施期間はタスクがロックされるため、 2 が実施中は 3 を実施できない。
    これはタスクの実行中に別スレッドからタスクの内容が破壊されないようにするため。

    1 を実施するのは、基本的に IPC 処理スレッド (dispatcher スレッド)
    2 を実施するのは、基本的に遅延実行スレッド (executor スレッド)
    3 は、両スレッドから実施され得る。
    dispatcher スレッドは N 、 executor スレッドは 1 を想定する。
 */
AbstractExecutor::AbstractExecutor() NN_NOEXCEPT
    : m_Event(os::EventClearMode_ManualClear)
{
}
Result AbstractExecutor::Register(Executable* pTask) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pTask != nullptr);
    NN_RESULT_DO(AddToWaitingList(pTask));
    m_Event.Signal();
    NN_RESULT_SUCCESS;
}
void AbstractExecutor::Unregister(Executable* pTask) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pTask != nullptr);
    DeleteFromWaitingList(pTask);
    // pTask が実行中の場合があるため Lock() できるようになるまで待つ。
    // pTask がキャンセル済みであれば、基本的に即時 Lock() できる。
    pTask->Lock();
    pTask->Unlock();
}
bool AbstractExecutor::TryUnregister(Executable* pTask) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pTask != nullptr);
    DeleteFromWaitingList(pTask);
    if (pTask->TryLock())
    {
        // ロックを取れたら Unregister 成功
        pTask->Unlock();
        return true;
    }
    return false;
}
void AbstractExecutor::Execute(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    Executable* pTask;
    while ((pTask = LockAndGetWaiting()) != nullptr)
    {
        pTask->Execute(buffer, bufferSize);
        // 完了したタスクは unlock 前にリストから削除する。
        // unlock すると pTask が破棄される可能性があり、破棄された pTask の Lock() が別のスレッドから呼ばれる可能性があるため。
        DeleteFromWaitingList(pTask);
        pTask->Unlock();
    }
}
void AbstractExecutor::InitializeMultiWaitHolder(os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    os::InitializeMultiWaitHolder(pMultiWaitHolder, m_Event.GetBase());
}
bool AbstractExecutor::TryWaitSignal() NN_NOEXCEPT
{
    return m_Event.TryWait();
}
void AbstractExecutor::WaitSignal() NN_NOEXCEPT
{
    m_Event.Wait();
}
void AbstractExecutor::ClearSignal() NN_NOEXCEPT
{
    m_Event.Clear();
}

/* ------------------------------------------------------------
    AsyncExecution
 */

AsyncExecution::AsyncExecution(Executable* pExecutable) NN_NOEXCEPT
    : m_pExecutable(pExecutable)
{
    NN_SDK_REQUIRES_NOT_NULL(pExecutable);
}
AsyncExecution::AsyncExecution(AsyncExecution&& rhs) NN_NOEXCEPT
    : m_pExecutable(rhs.m_pExecutable)
    , m_pExecutor(rhs.m_pExecutor)
{
    rhs.m_pExecutable = nullptr;
    rhs.m_pExecutor = nullptr;
}
AsyncExecution::~AsyncExecution() NN_NOEXCEPT
{
    if (IsInitialized())
    {
        Finalize();
    }
}
AsyncExecution& AsyncExecution::operator =(AsyncExecution&& rhs) NN_NOEXCEPT
{
    AsyncExecution tmp(std::move(rhs));
    std::swap(m_pExecutable, tmp.m_pExecutable);
    std::swap(m_pExecutor, tmp.m_pExecutor);
    return *this;
}

bool AsyncExecution::IsInitialized() const NN_NOEXCEPT
{
    return m_pExecutor != nullptr;
}
Result AsyncExecution::Initialize(AbstractExecutor* pExecutor) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pExecutor != nullptr);
    NN_SDK_REQUIRES(!IsInitialized());
    NN_RESULT_DO(m_pExecutable->Initialize());
    NN_RESULT_DO(pExecutor->Register(m_pExecutable));
    m_pExecutor = pExecutor;
    NN_RESULT_SUCCESS;
}
void AsyncExecution::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    Cancel();
    // Lock() を取得できるまで (タスク実行中であれば終了まで) 待つ
    // Cancel() 呼び出し時点で未実行であれば即時取得できる。
    m_pExecutable->Lock();
    NN_SDK_ASSERT(m_pExecutable->HasDone());
    m_pExecutable->Unlock();

    m_pExecutor = nullptr;
}
void AsyncExecution::Cancel() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    m_pExecutable->Cancel();
    if (m_pExecutor->TryUnregister(m_pExecutable))
    {
        // 実行中ではなく (TryUnregister 可)、未実行であれば空実行する
        m_pExecutable->Lock();
        if (!m_pExecutable->HasDone())
        {
            m_pExecutable->Execute(nullptr, 0);
        }
        m_pExecutable->Unlock();
    }
}
}} // ~namespace nn::util
