﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/spl/spl_Api.h>
#include <nn/spl/detail/spl_GeneralInterface.h>
#include <nn/spl/detail/spl_CryptoInterface.h>
#include <nn/spl/detail/spl_FsInterface.h>
#include <nn/spl/detail/spl_DeviceUniqueDataInterface.h>
#include <nn/spl/detail/spl_SslInterface.h>
#include <nn/spl/detail/spl_EsInterface.h>
#include <nn/spl/detail/spl_ManuInterface.h>

namespace nn { namespace spl {

    class InterfaceHolder
    {
    private:
        struct AllocatorTag
        {
        };
        typedef nn::sf::ExpHeapStaticAllocator<1024, AllocatorTag> MyAllocator;

    public:
        InterfaceHolder()
            : m_pGeneralInterface(nullptr)
            , m_pCryptoInterface(nullptr)
            , m_pFsInterface(nullptr)
            , m_pDeviceUniqueDataInterface(nullptr)
            , m_pSslInterface(nullptr)
            , m_pEsInterface(nullptr)
            , m_pManuInterface(nullptr)
        {
        }
        ~InterfaceHolder()
        {
            Release();
        }
        void AcquireGeneralInterface()
        {
            AcquireImpl(&m_pGeneralInterface, spl::detail::PortName);
        }
        void AcquireCryptoInterface()
        {
            AcquireImpl(&m_pCryptoInterface, spl::detail::PortNameCrypto);
            m_pGeneralInterface = m_pCryptoInterface;
        }
        void AcquireFsInterface()
        {
            AcquireImpl(&m_pFsInterface, spl::detail::PortNameFs);
            m_pGeneralInterface = m_pFsInterface;
            m_pCryptoInterface = m_pFsInterface;
        }
        void AcquireSslInterface()
        {
            AcquireImpl(&m_pSslInterface, spl::detail::PortNameSsl);
            m_pGeneralInterface = m_pSslInterface;
            m_pCryptoInterface = m_pSslInterface;
            m_pDeviceUniqueDataInterface = m_pSslInterface;
        }
        void AcquireEsInterface()
        {
            AcquireImpl(&m_pEsInterface, spl::detail::PortNameEs);
            m_pGeneralInterface = m_pEsInterface;
            m_pCryptoInterface = m_pEsInterface;
            m_pDeviceUniqueDataInterface = m_pEsInterface;
        }
        void AcquireManuInterface()
        {
            AcquireImpl(&m_pManuInterface, spl::detail::PortNameManu);
            m_pGeneralInterface = m_pManuInterface;
            m_pCryptoInterface = m_pManuInterface;
            m_pDeviceUniqueDataInterface = m_pManuInterface;
        }
        void Release()
        {
            if (m_pGeneralInterface != nullptr)
            {
                sf::ReleaseSharedObject(m_pGeneralInterface);
                m_pGeneralInterface = nullptr;
                m_pCryptoInterface = nullptr;
                m_pFsInterface = nullptr;
                m_pDeviceUniqueDataInterface = nullptr;
                m_pSslInterface = nullptr;
                m_pEsInterface = nullptr;
                m_pManuInterface = nullptr;
            }
        }
        nn::spl::detail::IGeneralInterface* GetGeneralInterface()
        {
            NN_SDK_ASSERT_NOT_NULL(m_pGeneralInterface);
            return m_pGeneralInterface;
        }
        nn::spl::detail::ICryptoInterface* GetCryptoInterface()
        {
            NN_SDK_ASSERT_NOT_NULL(m_pCryptoInterface);
            return m_pCryptoInterface;
        }
        nn::spl::detail::IFsInterface* GetFsInterface()
        {
            NN_SDK_ASSERT_NOT_NULL(m_pFsInterface);
            return m_pFsInterface;
        }
        nn::spl::detail::ISslInterface* GetSslInterface()
        {
            NN_SDK_ASSERT_NOT_NULL(m_pSslInterface);
            return m_pSslInterface;
        }
        nn::spl::detail::IEsInterface* GetEsInterface()
        {
            NN_SDK_ASSERT_NOT_NULL(m_pEsInterface);
            return m_pEsInterface;
        }
        nn::spl::detail::IManuInterface* GetManuInterface()
        {
            NN_SDK_ASSERT_NOT_NULL(m_pManuInterface);
            return m_pManuInterface;
        }
        bool IsCryptoInterfaceAvailable()
        {
            return m_pCryptoInterface != nullptr;
        }
        bool IsFsInterfaceAvailable()
        {
            return m_pFsInterface != nullptr;
        }
        bool IsSslInterfaceAvailable()
        {
            return m_pSslInterface != nullptr;
        }
        bool IsEsInterfaceAvailable()
        {
            return m_pEsInterface != nullptr;
        }
        bool IsManuInterfaceAvailable()
        {
            return m_pManuInterface != nullptr;
        }
        nn::spl::detail::IDeviceUniqueDataInterface* GetDeviceUniqueDataInterface()
        {
            NN_SDK_ASSERT_NOT_NULL(m_pDeviceUniqueDataInterface);
            return m_pDeviceUniqueDataInterface;
        }
    private:
        template <typename Interface>
        void AcquireImpl(Interface** pOut, const char* portName)
        {
            MyAllocator::Initialize(nn::lmem::CreationOption_NoOption);
            nn::sf::SharedPointer<Interface> p;
            auto result = nn::sf::CreateHipcProxyByName<Interface, MyAllocator::Policy>(&p, portName);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            *pOut = p.Detach();
            NN_ABORT_UNLESS(*pOut != nullptr);
        }
    private:
        nn::spl::detail::IGeneralInterface* m_pGeneralInterface;
        nn::spl::detail::ICryptoInterface* m_pCryptoInterface;
        nn::spl::detail::IFsInterface* m_pFsInterface;
        nn::spl::detail::IDeviceUniqueDataInterface* m_pDeviceUniqueDataInterface;
        nn::spl::detail::ISslInterface* m_pSslInterface;
        nn::spl::detail::IEsInterface* m_pEsInterface;
        nn::spl::detail::IManuInterface* m_pManuInterface;
    };

}}  // namespace nn::spl
