﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "resolver_DnsServerHelper.h"
#include <nn/os/os_Result.public.h>
#include <mutex>

extern "C"
{
#include <siglo/BionicPrivateApi.h>
};

#define LOG_MODULE_NAME "serverHelper" // NOLINT(preprocessor/const)
#include <nn/socket/resolver/private/resolver_DebugLog.h>

namespace nn { namespace socket { namespace resolver {

// resolver DNS server address support

const unsigned int NameserverPort = 53;
nn::os::Mutex g_DnsArrayAccessLock(true);
const unsigned int NameserverArrayMax = 2;
nn::socket::SockAddrIn g_DnsIpServerAddressArray[NameserverArrayMax];
size_t g_DnsIpServerAddressArrayLength = 0;


void ZeroDnsServerSettings()
{
    memset(g_DnsIpServerAddressArray, '\0', sizeof(g_DnsIpServerAddressArray));
    g_DnsIpServerAddressArrayLength = 0;
};

//
// DNS ADDRESS SET
//

void LogDnsAddresses(const char* function)
{
    for (unsigned idx=0; idx<g_DnsIpServerAddressArrayLength; ++idx)
    {
        LogInfo("%s: DNS Server[%u]: %s:%u\n",
                function,
                idx,
                nn::socket::InetNtoa(g_DnsIpServerAddressArray[idx].sin_addr),
                nn::socket::InetNtohs(g_DnsIpServerAddressArray[idx].sin_port));
    };
};

// used by BSD CFG
Result SetInaddrDnsAddresses(const uint32_t* pIpv4AddressArray, size_t length) NN_NOEXCEPT
{
    nn::Result result = ResultInternalError();

    if ( NULL != pIpv4AddressArray )
    {
        nn::socket::SockAddrIn sin[NameserverArrayMax] = { {'\0'}, {'\0'} };
        unsigned min = length < NameserverArrayMax ? length : NameserverArrayMax;

        for (unsigned idx=0; idx< min; ++idx)
        {
            sin[idx].sin_family = nn::socket::Family::Af_Inet;
            sin[idx].sin_port = nn::socket::InetHtons(NameserverPort);
            sin[idx].sin_addr.S_addr = pIpv4AddressArray[idx];
            sin[idx].sin_len = sizeof(nn::socket::SockAddrIn);
        };

        SetSockaddrDnsAddresses(sin, length);
    };

    return result;
};

Result SetSockaddrDnsAddresses(const nn::socket::SockAddrIn* addresses, size_t length)
{
    // Please note: Results are used to indicate a failure of the
    // server, but not typical errors (i.e. host name lookup failed)
    std::lock_guard<nn::os::Mutex> lock(g_DnsArrayAccessLock);
    Result result = ResultSuccess();

    memset(g_DnsIpServerAddressArray, '\0', sizeof(g_DnsIpServerAddressArray));
    g_DnsIpServerAddressArrayLength = 0;

    // error check
    if ( NULL == addresses )
    {
        LogMinor("Attempt to set DNS server address array to NULL\n");
        goto bail;
    }
    else if (0 == length)
    {
        LogMinor("Attempt to set DNS server address array to length = 0\n");
        goto bail;
    }
    else if (length > NameserverArrayMax)
    {
        length = NameserverArrayMax;
    };

    memcpy(g_DnsIpServerAddressArray, addresses, sizeof(g_DnsIpServerAddressArray[0]) * length);
    g_DnsIpServerAddressArrayLength = length;
    LogDnsAddresses(__FUNCTION__);

bail:
    return result;
};

//
// DNS ADDRESS GET
//
Result GetDnsAddress(unsigned nIndex, nn::socket::SockAddrIn* pSinAddr) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_DnsArrayAccessLock);
    Result result = ResultSuccess();
    if (NULL == pSinAddr)
    {
        LogDebug("NULL out parameter.\n");
        goto bail;
    };

    memset(pSinAddr, '\0', sizeof(*pSinAddr));

    if (nIndex >= g_DnsIpServerAddressArrayLength)
    {
        result = ResultInternalError();
        goto bail;
    };

    *pSinAddr = g_DnsIpServerAddressArray[nIndex];

bail:
    return result;
};



/**
 * @brief updates the client TLS with the most recent DNS servers
 */
void UpdateBionicServers()
{
    std::lock_guard<nn::os::Mutex> lock(g_DnsArrayAccessLock);

    if ( 0 == g_DnsIpServerAddressArrayLength  )
    {
        ResolverSetDnsServersIntoTLS(0);
    }
    else
    {
        ResolverSetDnsServersIntoTLS(g_DnsIpServerAddressArrayLength);
    };
};

}}}; // nn::socket::resolver

int ResolverSetDnsAddresses(const nn::socket::SockAddrIn* array, unsigned int n)
{
    int rc = 0;

    if (!nn::socket::resolver::SetSockaddrDnsAddresses(array, n).IsSuccess())
    {
        rc = -1;
    };

    return rc;
};
